/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * GraphDisplay.C: Implements GraphDisplay showing nodes and edges.
 *
 */

#include "GraphDisplay.h"

#include "BulletinBoardWidget.h"
#include "GraphNode.h"
#include "GraphEdge.h"
#include "GraphNodePtrList.h"
#include "XArgs.h"

GraphDisplay::GraphDisplay( WidgetBase *par ) 
	     : WidgetBase( par )
{
	xLoc = 10;
	yLoc = 10;

	Dimension w, h;
	par->getSize( &w, &h );

	XArgs xargs;
	xargs.addArg( XmNwidth, w );
	xargs.addArg( XmNheight, h );

	backdrop = new BulletinBoardWidget( par, xargs, "backdrop" );
	backdrop->manage();
	setWidget( backdrop->getWidget() );

	nodes = new GraphNodePtrList;
}

GraphDisplay::~GraphDisplay()
{
	/*
	 * Delete each graph node instance which also deletes the graph
	 * edges coming into each node.  
	 * Then delete the list of nodes and the backdrop.
	 */

	for ( int i = 0; i < nodes->count(); i++ ) {
	    GraphNode *aNode = nodes->getElement( i );
	    delete aNode;
	}

	delete nodes;
	delete backdrop;
}

void
GraphDisplay::addEdge( int fromNode, int toNode )
{
	/*
	 * Note that while we create our edges here, they are deleted
	 * when a GraphNode is deleted --- any edges into the node are
	 * destroyed when the node is destroyed.
	 */
	GraphNode *inputNode = nodes->getElement( fromNode );
	GraphNode *outputNode = nodes->getElement( toNode );

	if ( inputNode != NULL && outputNode != NULL ) {
	    CString lineName = 
		  inputNode->getNodeName() + "_to_" + outputNode->getNodeName();
	    GraphEdge *edge = new GraphEdge( backdrop, NullArgs, lineName,
					     inputNode, outputNode );

	    inputNode->addOutgoingEdge( edge );
	    outputNode->addIncomingEdge( edge );
	    edge->manage();
	}
}

int 
GraphDisplay::addNode( const CString& name )
{
	XArgs xargs;
	xargs.addArg( XmNx, (XtArgVal) xLoc );
	xargs.addArg( XmNy, (XtArgVal) yLoc );
	GraphNode *node = new GraphNode( backdrop, xargs, name );
	node->manage();

	// Move to new position for next node;  We don"t change X, only Y
	yLoc += 30;

	return ( nodes->addElement( node ) );
}

Boolean_
GraphDisplay::containsNode( const CString& name ) const
{
	for ( int i = 0; i < nodes->count(); i++ ) {
	    if ( name == nodes->getElement(i)->getNodeName() ) {
		return TRUE_;
	    }
	}
	return FALSE_;
}

void
GraphDisplay::deleteNode( int nodeIndex )
{
	/*
	 * To delete a node we first delete the GraphNode instance --
	 * also deletes any incoming edges to the node being deleted.
	 * Then, we remove the node from the list of nodes in the graph,
	 * shifting the remaining nodes to fill in the blank spot.
	 *
	 * If the node we deleted was the last one in the graph, reset
	 * our yLoc for positioning nodes in the display.
	 */
	GraphNode *theNode = nodes->getElement( nodeIndex );	
	delete theNode;
	nodes->deleteElementAt( nodeIndex );

	if ( nodes->count() == 0 ) {
	   yLoc = 10;
	}
}

int
GraphDisplay::getNodeIndex( GraphNode *nodePtr ) const
{
	for ( int i = 0; i < nodes->count();  i++ ) {
	    if ( nodePtr == nodes->getElement( i ) ) {
		return i;
	    }
	}
	return -1;
}

void 
GraphDisplay::moveNode( const CString& name, int x, int y ) 
{
	for ( int i = 0; i < nodes->count(); i++ ) {
	    if ( name == nodes->getElement(i)->getNodeName() ) {
	        nodes->getElement(i)->moveTo( x, y );
		return;
	    }
	}
}

void
GraphDisplay::reset()
{
	/*
	 * To reset our graph...
	 *  - Delete each graph node instance which also deletes the graph
	 *    edges coming into each node.  
	 *  - Delete the list of nodes and construct a new one.
	 *  - Reset our yLoc for positioning node bubbles.
	 */

	for ( int i = 0; i < nodes->count(); i++ ) {
	    GraphNode *aNode = nodes->getElement( i );
	    delete aNode;
	}

	delete nodes;
	nodes = new GraphNodePtrList;
	yLoc = 10;
}

Boolean_ 
GraphDisplay::writeLayoutToFP( FILE *fp ) const
{
	Position x, y;
	GraphNode *node;

	for ( int i = 0; i < nodes->count(); i++ ) {
	    node = nodes->getElement( i );
	    node->getPosition( &x, &y );
	    fprintf( fp, "loc %s %d %d\n", 
			  (const char *)node->getNodeName(), x, y );
	}
	return SUCCESS_;
}
