/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Contributing Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * GraphNode.C: Creates and maintains the nodes of the execution graph.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Interface/RCS/GraphNode.C,v 1.20 1994/02/25 04:33:30 aydt Exp $
 *	
 */
#include <Xm/Label.h>

#include "GraphEdge.h"
#include "GraphNode.h"
#include "GraphEdgePtrList.h"
#include "Pablo.h"
#include "PabloMainInterface.h"
#include "PabloResources.h"
#include "util.h"

GraphNode::GraphNode( WidgetBase *par, const XArgs& args, const CString& name )
	  : ShapedLabel( par ), nodeName( name )
{
	_setType( "GraphNode" );
	XmString xmlabel = XmStringCreateLtoR( (char *)name.getValue(),
					       XmSTRING_DEFAULT_CHARSET );
	
	XArgs myargs( args );
	myargs.addArg( XmNlabelString, (XtArgVal) xmlabel );
	
	widget = XmCreateLabel( par->getWidget(), (char *)name.getValue(),
			        myargs.getArgs(), myargs.getNArgs() );
	XmStringFree( xmlabel );

	Dimension width, height;
	getSize( &width, &height );
	width += 5;
	height += 5;
	setSize( width, height );
	getSize( &width, &height );

	incomingEdges = new GraphEdgePtrList;
	outgoingEdges = new GraphEdgePtrList;

	_setupHandlers();
	setupShape();
}

GraphNode::~GraphNode()
{
	/*
	 * To delete a node we must do several things:
	 * 1) For each incoming edge
	 *	- remove the edge entry from the outgoing edge list of
	 *        the originating GraphNode
	 *	- delete the edge
	 * 2) Delete our lists of Edge pointers.
	 * 3) Remove our event handlers
	 * 4) WidgetBase class destructor destroys the widget
	 */

	for ( int i = 0; i < incomingEdges->count(); i++ ) {
	    GraphEdge *anEdge = incomingEdges->getElement(i);
	    anEdge->getSourceNode()->deleteOutgoingEdge( anEdge );
	    delete anEdge;
	}
	
	delete incomingEdges;
	delete outgoingEdges;

        removeMemberEventHandler( ButtonPressMask, FALSE,
                                  &EventHandler::eventHandler1, this );

        removeMemberEventHandler( ButtonMotionMask, FALSE_,
                                  &EventHandler::eventHandler2, this );

        removeMemberEventHandler( ButtonReleaseMask, FALSE_,
                                  &EventHandler::eventHandler3, this );

}

void 
GraphNode::_beginMotion()
{
	int i;
	
	for ( i = 0; i < incomingEdges->count(); i++ ) {
	    incomingEdges->getElement(i)->unmanage();
	}
	for ( i = 0; i < outgoingEdges->count(); i++ ) {
	    outgoingEdges->getElement(i)->unmanage();
	}
}

void 
GraphNode::_endMotion()
{
	int i;
	Position bX, bY, tX, tY;

	getBottomCenter( &bX, &bY );
	getTopCenter( &tX, &tY );
	
	for ( i = 0; i < incomingEdges->count(); i++ ) {
	    incomingEdges->getElement(i)->setEnd( tX, tY );
	    incomingEdges->getElement(i)->manage();
	}

	for ( i = 0; i < outgoingEdges->count(); i++ ) {
	    outgoingEdges->getElement(i)->setStart( bX, bY );
	    outgoingEdges->getElement(i)->manage();
	}
}

void 
GraphNode::_setupHandlers()
{
	addMemberEventHandler( ButtonPressMask, FALSE,
			       &EventHandler::eventHandler1, this );

	addMemberEventHandler( ButtonMotionMask, FALSE_,
			       &EventHandler::eventHandler2, this );

	addMemberEventHandler( ButtonReleaseMask, FALSE_,
			       &EventHandler::eventHandler3, this );
}


void 
GraphNode::addIncomingEdge( GraphEdge *edge )
{
	Position x, y;
	getTopCenter( &x, &y );
	incomingEdges->addElement( edge );
	edge->setEnd( x, y );
}

void 
GraphNode::addOutgoingEdge( GraphEdge *edge )
{
	Position x, y;
	getBottomCenter( &x, &y );
	outgoingEdges->addElement( edge );
	edge->setStart( x, y );
}

void 
GraphNode::deleteOutgoingEdge( GraphEdge *edge )
{
	for ( int i = 0; i < outgoingEdges->count(); i++ ) {
	    if ( edge == outgoingEdges->getElement( i ) ) {
		outgoingEdges->deleteElementAt( i );
		return;
	    }
	}
}

void 
GraphNode::getBottomCenter( Position *x, Position *y )
{
	Dimension w, h;
	Position pX, pY;

	getSize( &w, &h );
	getPosition( &pX, &pY );
	*x = pX + w / 2;
	*y = pY + h;
}

void 
GraphNode::getTopCenter( Position *x, Position *y )
{
	Dimension w, h;
	Position pX, pY;

	getSize( &w, &h );
	getPosition( &pX, &pY );
	*x = pX + w / 2;
	*y = pY;
}

void 
GraphNode::moveTo( Position x, Position y )
{
	Position bX, bY, tX, tY;
	int i;

	setPosition( x, y );

	getBottomCenter( &bX, &bY );
	getTopCenter( &tX, &tY );
	
	for ( i = 0; i < incomingEdges->count(); i++ ) {
	    incomingEdges->getElement(i)->setEnd( tX, tY );
	    incomingEdges->getElement(i)->manage();
	}

	for ( i = 0; i < outgoingEdges->count(); i++ ) {
	    outgoingEdges->getElement(i)->setStart( bX, bY );
	    outgoingEdges->getElement(i)->manage();
	}
}

void 				
GraphNode::eventHandler1( Widget /* w */, XtPointer /* clientData */, 
			  XEvent *event, Boolean * /* continueToDisptch */ )
{
	/* handles press events */

	XButtonEvent *bevent = (XButtonEvent *) event;
	offsetX = bevent->x;
	offsetY = bevent->y;

	if ( bevent->button == 2 ) {
	    _beginMotion();
	}

	Pablo::MainInterface()->nodeCallback( this, event );
}

void 	
GraphNode::eventHandler2( Widget /* w */, XtPointer /* clientData */, 
			  XEvent *event, Boolean * /* continueToDisptch */ )
{
	/* handles motion events */

	Position dx, dy;
	Position pox, poy;

	XMotionEvent *mevent = (XMotionEvent *) event;
	if ( ! ( mevent->state & Button2Mask ) ) {
	    return;
	}

	getParent()->getOrigin( &pox, &poy );

	dx = ( mevent->x_root - offsetX ) - pox;
	dy = ( mevent->y_root - offsetY ) - poy;

	moveTo( dx, dy );
}

void 		
GraphNode::eventHandler3( Widget /* w */, XtPointer /* clientData */, 
			  XEvent *event, Boolean * /* continueToDisptch */ )
{
	/* handles release events */

	XButtonEvent *bevent = (XButtonEvent *) event;
	Position dx, dy;
	Position pox, poy;

	getParent()->getOrigin( &pox, &poy );

	dx = ( bevent->x_root - offsetX ) - pox;
	dy = ( bevent->y_root - offsetY ) - poy;


	if (bevent->button == 2) {
	    moveTo( dx, dy );
	    _endMotion();
	}
}

void 
GraphNode::setupShape()
{
	/* overloads ShapedLabel method */

	GC gc = getGC();
	Pixmap mask = getMask();
	Display *dpy = getDisplay();
	Window w = getWindow();
	Dimension width, height;

	getSize( &width, &height );

	if ( pabloData.roundNodes ) {
	    XSetForeground( dpy, gc, 0 );
	    XFillRectangle( dpy, mask, gc, 0, 0, width, height );
	    XSetForeground( dpy, gc, 1 );
	    XFillArc( dpy, mask, gc, 0, 0, width, height, 0, 360 * 64 );
	} else {
	    XSetForeground( dpy, gc, 1 );
  	    XFillRectangle( dpy, mask, gc, 0, 0, width, height );
	}
	setShape();
}

