/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *               University of Illinois at Urbana-Champaign
 *               Department of Computer Science
 *               1304 W. Springfield Avenue
 *               Urbana, IL     61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *      All Rights Reserved.
 *
 * Author: Keith A. Shields (shielss@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * OutputSpecificationDialog.cc : This module implements a custom dialog for
 *                              specifying output record fields and binding
 *                              them to input pipes
 * 
 *
 */
#include <stdlib.h>

#ifdef ultrix
extern "C" {
/* for some reason, strdup isn't in <string.h> on ULTRIX V4.2 (Rev. 96) */
char* strdup(const char*);
}
#endif

#include "OutputSpecificationDialog.h"

#include "GeneralDialogBox.h"
#include "PabloHelpSystem.h"

OutputSpecificationDialog::OutputSpecificationDialog(Widget parentApp, 
					 InterfaceClass *pabloInterface,
					 ConfigBoard *statusBoard)
{
  _setClassName(MY_CLASS);

  PabloInterface = pabloInterface;
  StatusBoard = statusBoard;

  ModuleResultSelected = False;
  CurrentPipeIndex = NONE_SELECTED;

  ModuleResultBinding.outputFieldList = 
    (OutputFieldEntry *) malloc(sizeof(OutputFieldEntry));

  DialogShell = XtCreatePopupShell("OutputSpecificationDialog", 
				   topLevelShellWidgetClass, parentApp, NULL, 
				   0);
				   
  Widget panedWindow  = XtVaCreateWidget("dialogPane",
					 xmPanedWindowWidgetClass, 
					 DialogShell, 
					 XmNsashWidth, 1, 
					 XmNsashHeight, 1,
					 NULL );

  _setDialogTitle();
  _buildControlArea(panedWindow);
  _buildActionArea(panedWindow);

  XtManageChild(panedWindow);
}



OutputSpecificationDialog::~OutputSpecificationDialog()
{
  int outFieldIdx;
  int numInPipes = PabloInterface->getNumberInputPipes(); 

  for (int pipeIdx=0; pipeIdx<numInPipes; pipeIdx++) {
    for (outFieldIdx=0; outFieldIdx<Pipes[pipeIdx].numOutputFieldsSatisfied;
	 outFieldIdx++) 
      {
	free(Pipes[pipeIdx].outputFieldList[outFieldIdx].outputFieldName);
      }
    free(Pipes[pipeIdx].outputFieldList);
  }

  for (outFieldIdx=0; 
       outFieldIdx<ModuleResultBinding.numOutputFieldsSatisfied;
       outFieldIdx++) 
    {
      free(ModuleResultBinding.outputFieldList[outFieldIdx].outputFieldName);
    }

  free(ModuleResultBinding.outputFieldList); 

  XtDestroyWidget(DialogShell);
}



void OutputSpecificationDialog::_buildActionArea(Widget panedWindow)
{
  Widget actionArea = XtCreateWidget("actionArea", xmFormWidgetClass, 
				     panedWindow, NULL, 0);

  ContinueButton = XtCreateManagedWidget("continue", xmPushButtonWidgetClass,
					 actionArea, NULL, 0);

  ResetButton = XtCreateManagedWidget("reset", xmPushButtonWidgetClass,
				      actionArea, NULL, 0);

  Widget help = XtCreateManagedWidget("help", xmPushButtonWidgetClass,
				      actionArea, NULL, 0);

  addCallback(ContinueButton, XmNactivateCallback, &Callback::callback1, 
	      this, NULL);
  addCallback(ResetButton, XmNactivateCallback, &Callback::callback2, this, 
	      NULL);
  addCallback(help, XmNactivateCallback, &Callback::helpCallback, this, NULL);

  /* fix height of action area */
  Dimension h; 
  XtVaGetValues(ResetButton, XmNheight, &h, NULL);
  XtVaSetValues(actionArea, 
		XmNpaneMaximum, h, 
		XmNpaneMinimum, h,
		NULL);

  XtManageChild( actionArea );
}



void OutputSpecificationDialog::_buildControlArea(Widget panedWindow)
{
  Widget controlArea = XtCreateWidget("controlArea", xmFormWidgetClass, 
				      panedWindow, NULL, 0 );

  char * moduleName = (char *) PabloInterface->getModuleName();
  XmString moduleNameX = XmStringCreateSimple(moduleName);
  Widget moduleNameLabel = XtVaCreateManagedWidget("moduleNameLabel", 
                                                   xmLabelWidgetClass, 
                                                   controlArea,
                                                   XmNlabelString, moduleNameX,
                                                   NULL);
  XmStringFree(moduleNameX);

  _makeRecordNameEditingFrame(controlArea, moduleNameLabel);
  _makePipeFrame(controlArea);
  _makeOutputFieldFrame(controlArea, moduleNameLabel);

  XtManageChild(controlArea);
}



Boolean OutputSpecificationDialog::_isFieldNameValid(char *fieldName)
{
  Boolean isNameValid = False;

  for (int i=0; i< strlen(fieldName); i++) {
    if (fieldName[i] != ' ') {
      isNameValid = True;
    }
  }

  return(isNameValid);
}



Boolean OutputSpecificationDialog::_isOutputFieldUnique(char *fieldName)
{
  int fieldIdx, numOutFields;
  int numInPipes = PabloInterface->getNumberInputPipes();
  Boolean isFieldUnique = True;

    /* an item may be removed from the list, but the internal data structures
       won't be updated till "done" is selected.  Therefore, we don't want to
       look at the currently selected pipe's field list when checking 
       uniqueness since it may not be up to date.  The list has the ability to
       check for the existence of a list item, so we can use that to 
       ensure that we don't add the same item to the list twice */

  for (int pipeIdx=0; pipeIdx<=numInPipes; pipeIdx++) {
    if (pipeIdx != CurrentPipeIndex) {
      numOutFields = Pipes[pipeIdx].numOutputFieldsSatisfied;
      for (fieldIdx=0; fieldIdx<numOutFields; fieldIdx++) {
	if (strcmp(fieldName, Pipes[pipeIdx].outputFieldList[fieldIdx].
		   outputFieldName) == 0) 
	  { isFieldUnique = False; }
      }
    }
  }

  if (!ModuleResultSelected) {
    numOutFields = ModuleResultBinding.numOutputFieldsSatisfied;
    for (fieldIdx=0; fieldIdx<numOutFields; fieldIdx++) {
      if (strcmp(fieldName, ModuleResultBinding.outputFieldList[fieldIdx].
		 outputFieldName)  == 0) 
	{ isFieldUnique = False; }
    }
  }
  return(isFieldUnique);
}



void OutputSpecificationDialog::_makeFieldEditingFrame(Widget outputForm)
{
  FieldEditingFrame =
    XtVaCreateWidget("fieldEditingFrame", xmFrameWidgetClass, outputForm,
		   XmNtopWidget, OutputList,
		   NULL);

  Widget fieldEditingForm = XtCreateWidget("fieldEditingForm", 
					   xmFormWidgetClass, 
					   FieldEditingFrame, NULL, 0);

  Widget fieldEditingLabel = XtCreateManagedWidget("fieldEditingLabel",
						   xmLabelWidgetClass,
						   fieldEditingForm, NULL, 0);

  Widget fieldNameTextWindow =
    XtVaCreateWidget("fieldNameTextWindow", xmScrolledWindowWidgetClass, 
		     fieldEditingForm, 
		     XmNtopWidget, fieldEditingLabel,
		     NULL);

  FieldNameTextField = XtCreateManagedWidget("fieldNameTextField", 
					     xmTextFieldWidgetClass, 
					     fieldNameTextWindow, NULL, 0);

  Widget addFieldButton = 
    XtVaCreateManagedWidget("addFieldButton", xmPushButtonWidgetClass,
			    fieldEditingForm, 
			    XmNtopWidget, fieldNameTextWindow,
			    NULL);

  Widget deleteFieldButton = 
    XtVaCreateManagedWidget("deleteFieldButton", xmPushButtonWidgetClass,
			    fieldEditingForm,
			    XmNtopWidget, fieldNameTextWindow,
			    NULL);

  addCallback(FieldNameTextField, XmNactivateCallback, &Callback::callback6, 
	      this, NULL);

  addCallback(addFieldButton, XmNactivateCallback, &Callback::callback6, this, 
	      NULL);

  addCallback(deleteFieldButton, XmNactivateCallback, &Callback::callback7, 
	      this, NULL);

  XtManageChild(fieldNameTextWindow);
  XtManageChild(fieldEditingForm);
  XtManageChild(FieldEditingFrame);
}



void OutputSpecificationDialog::_makeOutputFieldFrame(Widget controlArea,
						      Widget nameLabel)
{
  OutputFrame = XtVaCreateWidget("outputFrame", xmFrameWidgetClass,
				 controlArea, 
				 XmNtopWidget, nameLabel, 
				 XmNleftWidget, RecordNameEditingFrame,
				 NULL);

  Widget outputForm = XtCreateWidget("outputForm", xmFormWidgetClass,
				     OutputFrame, NULL, 0);
  
  Widget outputLabel = XtCreateManagedWidget("outputLabel", xmLabelWidgetClass,
					     outputForm, NULL, 0);

  PipeNameLabel = XtVaCreateManagedWidget("pipeNameLabel", xmLabelWidgetClass,
					  outputForm,
					  XmNtopWidget, outputLabel,
					  NULL);

  OutputList = XmCreateScrolledList(outputForm, "outputList", NULL,
				    0);

  _makeFieldEditingFrame(outputForm);

  Widget done = 
    XtVaCreateManagedWidget("done", xmPushButtonWidgetClass, outputForm, 
			    XmNtopWidget, FieldEditingFrame, 
			    NULL);

  Widget reset = 
    XtVaCreateManagedWidget("reset", xmPushButtonWidgetClass, outputForm, 
			    XmNtopWidget, FieldEditingFrame,
			    NULL);

  XtVaSetValues(XtParent(OutputList),
		XmNtopWidget, PipeNameLabel,
		NULL);

  XtSetSensitive(OutputFrame, False);

  addCallback(OutputList, XmNbrowseSelectionCallback, &Callback::callback5, 
	      this, NULL);
  addCallback(done, XmNactivateCallback, &Callback::callback8, this, NULL);
  addCallback(reset, XmNactivateCallback, &Callback::callback9, this, NULL);

  XtManageChild(OutputList);
  XtManageChild(outputForm);
  XtManageChild(OutputFrame);
}



void OutputSpecificationDialog::_makePipeFrame(Widget controlArea)
{
  PipeFrame = XtVaCreateWidget("pipeFrame", xmFrameWidgetClass,
			       controlArea, 
			       XmNtopWidget, RecordNameEditingFrame,
			       XmNrightWidget, RecordNameEditingFrame,
			       NULL);

  Widget pipeForm = XtCreateWidget("pipeForm", xmFormWidgetClass,
                                   PipeFrame, NULL, 0);
  
  Widget pipeLabel = XtCreateManagedWidget("pipeLabel", xmLabelWidgetClass,
                                           pipeForm, NULL, 0);

  Widget moduleResultButton = 
    XtCreateManagedWidget("moduleResultButton", xmPushButtonWidgetClass, 
                          pipeForm, NULL, 0);

  // user input is the n+1st pipe 
  Widget userInputButton = 
    XtVaCreateManagedWidget("userInputButton", xmPushButtonWidgetClass, 
			    pipeForm, 
			    XmNbottomWidget, moduleResultButton,
			    XmNuserData, 
			    PabloInterface->getNumberInputPipes(),
			    NULL);

  Widget pipeWindow = XtVaCreateWidget("pipeWindow", 
                                       xmScrolledWindowWidgetClass, 
                                       pipeForm,
                                       XmNtopWidget, pipeLabel,
                                       XmNbottomWidget, userInputButton,
                                       NULL);

  Widget pipeContainer =  XtCreateWidget("pipeContainer", 
                                         xmRowColumnWidgetClass, pipeWindow, 
                                         NULL, 0);

  addCallback(moduleResultButton, XmNactivateCallback, &Callback::callback4, 
	      this, NULL);

  addCallback(userInputButton, XmNactivateCallback, &Callback::callback3, 
	      this, NULL);

  _setupPipes(pipeContainer);
  _setupModuleResultBindings();

  XtManageChild(pipeWindow);
  XtManageChild(pipeContainer);
  XtManageChild(pipeForm);
  XtManageChild(PipeFrame);
}



void OutputSpecificationDialog::_makeRecordNameEditingFrame(Widget controlArea,
							    Widget nameLabel)
{
  RecordNameEditingFrame = XtVaCreateWidget("recordNameEditingFrame", 
					    xmFrameWidgetClass, 
					    controlArea, 
					    XmNtopWidget, nameLabel,
					    NULL);

  Widget recordNameEditingForm = XtCreateWidget("recordNameEditingForm", 
						xmFormWidgetClass, 
						RecordNameEditingFrame, 
						NULL, 0);
  
  Widget recordNameEditingLabel = 
    XtCreateManagedWidget("recordNameEditingLabel",xmLabelWidgetClass,
			  recordNameEditingForm, NULL, 0);

  Widget recordNameTextWindow = 
    XtVaCreateWidget("recordNameTextWindow", xmScrolledWindowWidgetClass,
		     recordNameEditingForm,
		     XmNtopWidget, recordNameEditingLabel,
		     NULL);


  RecordNameTextField = 
    XtCreateManagedWidget("recordNameTextField", xmTextFieldWidgetClass, 
			  recordNameTextWindow, NULL, 0);

  char* outRecName = (char *) PabloInterface->getOutputRecordName();
  XmTextFieldSetString(RecordNameTextField, outRecName);

  XtManageChild(recordNameTextWindow);
  XtManageChild(recordNameEditingForm);
  XtManageChild(RecordNameEditingFrame);
}



void OutputSpecificationDialog::_setDialogTitle()
{
  char buf[1024];
  
  sprintf(buf, "Specify Output Record Fields");

  XtVaSetValues(DialogShell, 
		XmNtitle, buf, 
		NULL);
}



void OutputSpecificationDialog::_setupModuleResultBindings()
{
  char *fieldName;

  int numOutFieldsBound = 
    PabloInterface->getNumberOutputFieldsBoundToInputPipe(MODULE_RESULT);

  ModuleResultBinding.numOutputFieldsSatisfied = numOutFieldsBound;
  ModuleResultBinding.outputFieldList = 
    (OutputFieldEntry *) realloc(ModuleResultBinding.outputFieldList,
				sizeof(OutputFieldEntry)*numOutFieldsBound);


  for (int fieldIdx=0; fieldIdx<numOutFieldsBound; fieldIdx++) {
    fieldName = (char *)
      PabloInterface->getNameOfOutputFieldBoundToInputPipe(MODULE_RESULT,
							   fieldIdx);
    ModuleResultBinding.outputFieldList[fieldIdx].outputFieldName = 
      strdup(fieldName);    
    StatusBoard->addOutputRecordField(fieldName);
  }
}



void OutputSpecificationDialog::_setupOutputFieldBindings(int pipeIdx)
{
  char *fieldName;

  int numOutFieldsBound = 
    PabloInterface->getNumberOutputFieldsBoundToInputPipe(pipeIdx);

  Pipes[pipeIdx].numOutputFieldsSatisfied = numOutFieldsBound;
  Pipes[pipeIdx].outputFieldList = 
    (OutputFieldEntry *) realloc(Pipes[pipeIdx].outputFieldList,
				 sizeof(OutputFieldEntry)*numOutFieldsBound);

  for (int fieldIdx=0; fieldIdx<numOutFieldsBound; fieldIdx++) {
    fieldName = (char *) PabloInterface->getNameOfOutputFieldBoundToInputPipe
                         (pipeIdx, fieldIdx);
    Pipes[pipeIdx].outputFieldList[fieldIdx].outputFieldName = 
      strdup(fieldName);

    StatusBoard->addOutputRecordField(fieldName);
    StatusBoard->addInputPipeOutputFieldBinding(pipeIdx, fieldName);
  }
}



void OutputSpecificationDialog::_setupPipes(Widget pipeContainer)
{
  Widget pipeButton;
  char *pipeName;
  XmString pipeNameX;
  Boolean isPipeSensitive;

  int numInPipes = PabloInterface->getNumberInputPipes();

  /* make pipe entries, counting user input a pipe */
  Pipes = (PipeBindingInfo *) malloc 
    (sizeof(PipeBindingInfo)*(numInPipes+1));

  for (int pipeIdx=0; pipeIdx<numInPipes; pipeIdx++) {
    pipeName = (char *) PabloInterface->getInputPipeName(pipeIdx);
    pipeNameX = XmStringCreateSimple(pipeName);

    // pipes that do not satisfy input ports cannot satisfy output record fields
    isPipeSensitive = PabloInterface->isPipeBound(pipeIdx);

    pipeButton = XtVaCreateManagedWidget(form("pipe%d",pipeIdx), 
                                         xmPushButtonWidgetClass, 
                                         pipeContainer,
                                         XmNlabelString, pipeNameX,
                                         XmNuserData, (XtPointer) pipeIdx,
					 XmNsensitive, isPipeSensitive,
                                         NULL);

    addCallback(pipeButton, XmNactivateCallback, &Callback::callback3, this, 
		NULL);

    Pipes[pipeIdx].outputFieldList = 
      (OutputFieldEntry *) malloc(sizeof(OutputFieldEntry));

    _setupOutputFieldBindings(pipeIdx);
    XmStringFree(pipeNameX);
  }

  /* initialize bindings for user input (treated as pipe n in 0 reference
     array) */
  Pipes[numInPipes].outputFieldList = 
    (OutputFieldEntry *) malloc(sizeof(OutputFieldEntry));
  _setupOutputFieldBindings(numInPipes);
}


void OutputSpecificationDialog::callback1( Widget /* button */, 
					   XtPointer /* ptr1 */,
					   XtPointer /* ptr2 */)
{
  /* callback called when continue is selected */

  DoneRunning = True;
}



void OutputSpecificationDialog::callback2( Widget /* button */, 
					   XtPointer /* ptr1 */,
					   XtPointer /* ptr2 */)
{
  /* callback called when reset is selected */
  XmString tmpXStr;

  int fieldIdx, numOutFields;
  int numInPipes = PabloInterface->getNumberInputPipes();

  /* since the scoreboard is constantly updated, resetting requires
     deletion of the output record fields in the scoreboard.  Reinitializing
     the internal state will cause the proper fields to be added back to the 
     scoreboard */
  for (int pipeIdx=0; pipeIdx<=numInPipes; pipeIdx++) {
    numOutFields = Pipes[pipeIdx].numOutputFieldsSatisfied;
    for (fieldIdx=0; fieldIdx<numOutFields; fieldIdx++) {
      StatusBoard->deleteOutputRecordField
	(Pipes[pipeIdx].outputFieldList[fieldIdx].outputFieldName);
    }
  }

  numOutFields = ModuleResultBinding.numOutputFieldsSatisfied;
  for (fieldIdx=0; fieldIdx<numOutFields; fieldIdx++) {
    StatusBoard->deleteOutputRecordField
      (ModuleResultBinding.outputFieldList[fieldIdx].outputFieldName);
  }

  // reset record name
  char *outRecName = (char *) PabloInterface->getOutputRecordName();
  XmTextFieldSetString(RecordNameTextField, outRecName);

  // user input is referenced as pipe n in arrays
  for (pipeIdx=0; pipeIdx<=numInPipes; pipeIdx++) {
    _setupOutputFieldBindings(pipeIdx);
  }

  _setupModuleResultBindings();

  // update output list to reflect changes caused by reset
  XmListDeleteAllItems(OutputList);

  if (! ModuleResultSelected) {
    if (CurrentPipeIndex != NONE_SELECTED) {
      int numFields = Pipes[CurrentPipeIndex].numOutputFieldsSatisfied;
      for (int fieldIdx=0; fieldIdx<numFields; fieldIdx++) {
	tmpXStr = XmStringCreateSimple
	  (Pipes[CurrentPipeIndex].outputFieldList[fieldIdx].outputFieldName);
	XmListAddItemUnselected(OutputList, tmpXStr, 0);
	XmStringFree(tmpXStr);
      }
    }
  } else {
      int numFields = ModuleResultBinding.numOutputFieldsSatisfied;
      for (int fieldIdx=0; fieldIdx<numFields; fieldIdx++) {
	tmpXStr = XmStringCreateSimple
	  (ModuleResultBinding.outputFieldList[fieldIdx].outputFieldName);
	XmListAddItemUnselected(OutputList, tmpXStr, 0);
	XmStringFree(tmpXStr);
      }
    }

  XmTextFieldSetString(FieldNameTextField,"\0");
}



void OutputSpecificationDialog::callback3( Widget pipe, 
					   XtPointer /* ptr1 */,
					   XtPointer /* ptr2 */)
{
  /* callback called when a pipe is selected */
  XmString tmpXStr, pipeNameX;

  ModuleResultSelected = False;
  XmListDeleteAllItems(OutputList);

  XtVaGetValues(pipe,
		XmNuserData, &CurrentPipeIndex,
		XmNlabelString, &pipeNameX,
		NULL);

  XtVaSetValues(PipeNameLabel,
		XmNlabelString, pipeNameX,
		NULL);

  int numFields = Pipes[CurrentPipeIndex].numOutputFieldsSatisfied;

  for (int fieldIdx=0; fieldIdx<numFields; fieldIdx++) {
    tmpXStr = XmStringCreateSimple
      (Pipes[CurrentPipeIndex].outputFieldList[fieldIdx].outputFieldName);
    XmListAddItemUnselected(OutputList, tmpXStr, 0);
    XmStringFree(tmpXStr);
  }

  XmTextFieldSetString(FieldNameTextField,"\0");
  XtSetSensitive(OutputFrame, True);
}



void OutputSpecificationDialog::callback4( Widget button, 
					   XtPointer /* ptr1 */,
					   XtPointer /* ptr2 */ )
{
  /* callback called when module result is selected */
  XmString tmpXStr, buttonLabel;

  ModuleResultSelected = True;
  XmListDeleteAllItems(OutputList);

  XtVaGetValues(button,
		XmNlabelString, &buttonLabel,
		NULL);

  XtVaSetValues(PipeNameLabel,
		XmNlabelString, buttonLabel,
		NULL);

  int numFields = ModuleResultBinding.numOutputFieldsSatisfied;

  for (int fieldIdx=0; fieldIdx<numFields; fieldIdx++) {
    tmpXStr = XmStringCreateSimple
      (ModuleResultBinding.outputFieldList[fieldIdx].outputFieldName);
    XmListAddItemUnselected(OutputList, tmpXStr, 0);
    XmStringFree(tmpXStr);
  }

  XmTextFieldSetString(FieldNameTextField,"\0");
  XtSetSensitive(OutputFrame, True);
}



void OutputSpecificationDialog::callback5( Widget /* list */, 
					   XtPointer /* ptr1 */,
					   XtPointer /* ptr2 */)
{
  /* callback called when an output list item is selected */
  int *pos_list, pos_count;
  XmStringTable listItems;
  char *listItem;
  
  if (XmListGetSelectedPos(OutputList, &pos_list, &pos_count)) {
    XtVaGetValues(OutputList, 
		  XmNitems, &listItems,
		  NULL);

    XmStringGetLtoR(listItems[pos_list[0]-1], XmSTRING_DEFAULT_CHARSET,
		       &listItem);

    XmTextFieldSetString(FieldNameTextField, listItem);
    free(pos_list);
    free(listItem);
  }
}



void OutputSpecificationDialog::callback6( Widget /* button */, 
					   XtPointer /* ptr1 */,
					   XtPointer /* ptr2 */)
{
  /* callback called when add item is selected */
  char * textFieldContents = XmTextFieldGetString(FieldNameTextField);
  XmString XtextFieldContents = XmStringCreateSimple(textFieldContents);

  Boolean fieldIsUnique = _isOutputFieldUnique(textFieldContents);
  Boolean fieldNameIsValid = _isFieldNameValid(textFieldContents);

  if ((!fieldIsUnique) || (XmListItemExists(OutputList, XtextFieldContents))) {
    Pablo::GeneralDialog()->
      run( "Field %s already exists in this module's output record.",
	  textFieldContents );
  } else {
    if (fieldNameIsValid) {
	StatusBoard->addOutputRecordField(textFieldContents);

	if (!ModuleResultSelected) {
	  StatusBoard->addInputPipeOutputFieldBinding(CurrentPipeIndex,
						      textFieldContents);
	}

	XmListAddItem(OutputList, XtextFieldContents, 0);
	XmListSelectItem(OutputList, XtextFieldContents, True);
	XtSetSensitive(PipeFrame, False);
	XtSetSensitive(ContinueButton, False);
	XtSetSensitive(ResetButton, False);
      } else {
	Pablo::GeneralDialog()->run( "An invalid field name was entered.");
      }
  }
}



void OutputSpecificationDialog::callback7( Widget /* button */, 
					   XtPointer /* ptr1 */,
					   XtPointer /* ptr2 */ )
{
  /* callback called when delete list item is selected */
  char * textFieldContents = XmTextFieldGetString(FieldNameTextField);
  XmString XtextFieldContents = XmStringCreateSimple(textFieldContents);

  if (XmListItemExists(OutputList, XtextFieldContents)) {
    StatusBoard->deleteOutputRecordField(textFieldContents);
    
    XmListDeleteItem(OutputList, XtextFieldContents);
    XmTextFieldSetString(FieldNameTextField, "\0");
    XmListSelectPos(OutputList, 1, True);
    XtSetSensitive(PipeFrame, False);
    XtSetSensitive(ContinueButton, False);
    XtSetSensitive(ResetButton, False);
  }
}



void OutputSpecificationDialog::callback8( Widget /* button */, 
					   XtPointer /* ptr1 */,
					   XtPointer /* ptr2 */)
{
  /* callback called when done w/ output field binding for a particular 
     pipe is selected */
  int numItems;
  XmStringTable listItems;
  char *listItem;

  XtVaGetValues(OutputList, 
		XmNitems, &listItems, 
		XmNitemCount, &numItems,
		NULL);

  if (! ModuleResultSelected) {
    Pipes[CurrentPipeIndex].numOutputFieldsSatisfied = numItems;
    free(Pipes[CurrentPipeIndex].outputFieldList);
    Pipes[CurrentPipeIndex].outputFieldList = 
	(OutputFieldEntry *) malloc(sizeof(OutputFieldEntry)*numItems);

    for (int fieldIdx=0; fieldIdx<numItems; fieldIdx++) {
      XmStringGetLtoR(listItems[fieldIdx], XmSTRING_DEFAULT_CHARSET, 
		      &listItem);
      Pipes[CurrentPipeIndex].outputFieldList[fieldIdx].outputFieldName =
	strdup(listItem);
      free(listItem);
    } 
  } else {
      ModuleResultBinding.numOutputFieldsSatisfied = numItems;
      free(ModuleResultBinding.outputFieldList);
      ModuleResultBinding.outputFieldList = 
	(OutputFieldEntry *) malloc(sizeof(OutputFieldEntry)*numItems);

      for (int fieldIdx=0; fieldIdx<numItems; fieldIdx++) {
	XmStringGetLtoR(listItems[fieldIdx], XmSTRING_DEFAULT_CHARSET, 
			&listItem);
	ModuleResultBinding.outputFieldList[fieldIdx].outputFieldName =
	  strdup(listItem);
	free(listItem);
      }
    }

  XtSetSensitive(PipeFrame, True);
  XtSetSensitive(ResetButton, True);

  int numInPipes = PabloInterface->getNumberInputPipes();
  int totNumOutFields = 0;

  for (int pipeIdx=0; pipeIdx<=numInPipes; pipeIdx++) {
    totNumOutFields += Pipes[pipeIdx].numOutputFieldsSatisfied;
  }

  totNumOutFields += ModuleResultBinding.numOutputFieldsSatisfied;
  
  if (totNumOutFields > 0) {
    XtSetSensitive(ContinueButton, True);
  } else {
    Pablo::GeneralDialog()->run("This module's output record is empty.");
  }
}



void OutputSpecificationDialog::callback9( Widget /* button */, 
					   XtPointer /* ptr1 */,
					   XtPointer /* ptr2 */)
{
  /*callback called when reset output field binding for a particular 
    pipe is selected */
  XmString tmpXStr;
  XmStringTable listItems;
  int numItems;
  char *tmpStr;

  XtVaGetValues(OutputList,
		XmNitemCount, &numItems,
		XmNitems, &listItems,
		NULL);

  for(int itmIdx=0; itmIdx<numItems; itmIdx++) {
    XmStringGetLtoR(listItems[itmIdx], XmSTRING_DEFAULT_CHARSET, &tmpStr);
    StatusBoard->deleteOutputRecordField(tmpStr);
    free(tmpStr);
  }

  XmListDeleteAllItems(OutputList);

  if (! ModuleResultSelected) {
    int numFields = Pipes[CurrentPipeIndex].numOutputFieldsSatisfied;
    for (int fieldIdx=0; fieldIdx<numFields; fieldIdx++) {
      tmpStr = Pipes[CurrentPipeIndex].outputFieldList[fieldIdx].
	       outputFieldName;
      tmpXStr = XmStringCreateSimple(tmpStr);
      XmListAddItemUnselected(OutputList, tmpXStr, 0);
      StatusBoard->addOutputRecordField(tmpStr);
      StatusBoard->addInputPipeOutputFieldBinding(CurrentPipeIndex, tmpStr);
      XmStringFree(tmpXStr);
    }
  } else {
      int numFields = ModuleResultBinding.numOutputFieldsSatisfied;
      for (int fieldIdx=0; fieldIdx<numFields; fieldIdx++) {
	tmpStr = ModuleResultBinding.outputFieldList[fieldIdx].outputFieldName;
	tmpXStr = XmStringCreateSimple(tmpStr);
	XmListAddItemUnselected(OutputList, tmpXStr, 0);
	StatusBoard->addOutputRecordField(tmpStr);
	XmStringFree(tmpXStr);
      }
    }

  XmTextFieldSetString(FieldNameTextField,"\0");
}



void OutputSpecificationDialog::helpCallback( Widget /* button */, 
					      XtPointer /* ptr1 */,
				      	      XtPointer /* ptr2 */)
{
  /* callback called when help is selected */
  Pablo::HelpSystem()->giveHelpOn( "OutputSpecificationDialog" );
}



void OutputSpecificationDialog::run()
{
  DoneRunning = False;

  XtManageChild(DialogShell);

  while (!DoneRunning) {
    XEvent event;
    XtAppNextEvent(
		   XtWidgetToApplicationContext(DialogShell), &event );
    XtDispatchEvent( &event );
  }
  
  XtUnmanageChild(DialogShell);

  int fieldIdx, numOutFields;

  // set the output record name
  char *tmpStr = XmTextFieldGetString(RecordNameTextField);
  PabloInterface->createOutputRecord(tmpStr);

  /* specify the output record fields - we don't know how the record has 
     changed, so we first clear all output fields then add them one at a time,
     indicating which pipe satisfies the output field - user input is pipe
     n and module result in pipe -1 */

  PabloInterface->removeAllOutputFields();
  int numInPipes = PabloInterface->getNumberInputPipes();

  for (int pipeIdx=0; pipeIdx<numInPipes; pipeIdx++) {
    if (PabloInterface->isPipeBound(pipeIdx)) {
      numOutFields = Pipes[pipeIdx].numOutputFieldsSatisfied;
      for (fieldIdx=0; fieldIdx<numOutFields; fieldIdx++) {
	PabloInterface->addOutputField(Pipes[pipeIdx].outputFieldList[fieldIdx]
				       .outputFieldName, pipeIdx);

	}
    }
  }

  // user input will fail the isPipesBound test, so do separately
  numOutFields = Pipes[numInPipes].numOutputFieldsSatisfied;
  for (fieldIdx=0; fieldIdx<numOutFields; fieldIdx++) {
    PabloInterface->addOutputField(Pipes[numInPipes].outputFieldList[fieldIdx]
				   .outputFieldName, pipeIdx);

  }


  // make bindings between module result and output record fields
    numOutFields = ModuleResultBinding.numOutputFieldsSatisfied;
    for (fieldIdx=0; fieldIdx<numOutFields; fieldIdx++) {
      PabloInterface->
	addOutputField(ModuleResultBinding.outputFieldList[fieldIdx].
		       outputFieldName, MODULE_RESULT);
    }
}





/*
 * Initialize the static data
 */
const char *const OutputSpecificationDialog::MY_CLASS = 
   "OutputSpecificationDialog";
