/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *               University of Illinois at Urbana-Champaign
 *               Department of Computer Science
 *               1304 W. Springfield Avenue
 *               Urbana, IL     61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *      All Rights Reserved.
 *
 * Author: Keith A. Shields (shields@cs.uiuc.edu)
 * 
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * PipePortBindingDialog.cc : This module implements a custom dialog for
 *                             binding input pipes to input ports
 * 
 *
 */

#include <stdlib.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>

#include "PipePortBindingDialog.h"

#include "GeneralDialogBox.h"
#include "PabloHelpSystem.h"

PipePortBindingDialog::PipePortBindingDialog(Widget parentApp, 
					     InterfaceClass *pabloInterface,
					     ConfigBoard *statusBoard)
{
  _setClassName(MY_CLASS);
  PabloInterface = pabloInterface;
  StatusBoard = statusBoard;
  CurrentPipeNum = CurrentPortNum = NONE_SELECTED;

  DialogShell = XtCreatePopupShell("PipePortBindingDialog", 
				   topLevelShellWidgetClass, parentApp, NULL, 
				   0);
				   
  Widget panedWindow  = XtVaCreateWidget("dialogPane", 
					 xmPanedWindowWidgetClass, 
					 DialogShell, 
					 XmNsashWidth, 1, 
					 XmNsashHeight, 1,
					 NULL );

  _setDialogTitle();
  _buildControlArea(panedWindow);
  _buildActionArea(panedWindow);

  _getUnboundPortColors();
  _initializeBindings();

  XtManageChild(panedWindow);
}



PipePortBindingDialog::~PipePortBindingDialog()
{
  free(PipeButtons);
  free(PortButtons);
  free(PipeBoundToPort);
  XtDestroyWidget(DialogShell);
}



void PipePortBindingDialog::_bindPipeToPort(int pipe, int port)
{
  Pixel bg_color, top_shadow, bottom_shadow, fg_ret, select_color;
  Colormap cmap;
  Boolean unboundPortExists = False, allPortsBoundToUserInput = True;

  XtVaGetValues(PipeButtons[pipe],
		XmNbackground, &bg_color, 
		XmNcolormap, &cmap,
		NULL);

  XmGetColors(XtScreen(PipeButtons[pipe]), cmap, bg_color, &fg_ret, 
	      &top_shadow, &bottom_shadow, &select_color);

  XtVaSetValues(PortButtons[port], 
		XmNbackground, bg_color, 
		XmNtopShadowColor, top_shadow,
		XmNbottomShadowColor, bottom_shadow, 
		XmNselectColor, select_color, 
		XmNborderColor, fg_ret,
		NULL);

  PipeBoundToPort[port] = pipe;

  StatusBoard->addInputPipePortBinding(pipe, port);

  int numInPorts = PabloInterface->getNumberInputPorts(); 
  int numInPipes = PabloInterface->getNumberInputPipes();

  for (int portIdx=0; portIdx<numInPorts; portIdx++) {
    if (PipeBoundToPort[portIdx] == NO_PIPE_BOUND) {
      unboundPortExists = True;
    }
  }

  for (portIdx=0; portIdx<numInPorts; portIdx++) {
    if (PipeBoundToPort[portIdx] != numInPipes) {
      allPortsBoundToUserInput = False;
    }
  }

  if (allPortsBoundToUserInput) {
    XtSetSensitive(ContinueButton, False);
    _clearPortBinding(port);
    Pablo::GeneralDialog()->run("User Input cannot satisfy all input ports.");
  }

  if ((! unboundPortExists) && (! allPortsBoundToUserInput)) {
    XtSetSensitive(ContinueButton, True);
  }

  CurrentPipeNum = CurrentPortNum = NONE_SELECTED;
}



void PipePortBindingDialog::_buildActionArea(Widget panedWindow)
{
  Widget actionArea = XtCreateWidget("actionArea", xmFormWidgetClass, 
				     panedWindow, NULL, 0);

  ContinueButton  = XtCreateManagedWidget("continue", 
					  xmPushButtonWidgetClass,
					  actionArea, NULL, 0); 

  Widget reset = XtCreateManagedWidget("reset", 
				       xmPushButtonWidgetClass,
				       actionArea, NULL, 0);

  Widget clear = XtCreateManagedWidget("clear", 
				       xmPushButtonWidgetClass,
				       actionArea, NULL, 0);

  Widget help = XtCreateManagedWidget("help", 
				       xmPushButtonWidgetClass,
				      actionArea, NULL, 0);

  addCallback(reset, XmNactivateCallback, &Callback::callback3, this, NULL);
  addCallback(clear, XmNactivateCallback, &Callback::callback4, this, NULL);
  addCallback(ContinueButton, XmNactivateCallback, &Callback::callback5, 
	      this, NULL);
  addCallback(help, XmNactivateCallback, &Callback::helpCallback, 
	      this, NULL);

  /* fix height of action area */
  Dimension h; 
  XtVaGetValues(reset, XmNheight, &h, NULL);
  XtVaSetValues(actionArea, 
		XmNpaneMaximum, h, 
		XmNpaneMinimum, h,
		NULL);

  XtManageChild( actionArea );
}


void PipePortBindingDialog::_buildControlArea(Widget panedWindow)
{
  Widget controlArea = XtCreateWidget( "controlArea", xmFormWidgetClass, 
				      panedWindow, NULL, 0 );

  const char* moduleName = PabloInterface->getModuleName();
  XmString moduleNameX = XmStringCreateSimple((char *)moduleName);
  Widget moduleNameLabel = XtVaCreateManagedWidget("moduleNameLabel", 
                                                   xmLabelWidgetClass, 
                                                   controlArea,
                                                   XmNlabelString, moduleNameX,
                                                   NULL);
  XmStringFree(moduleNameX);

  Widget pipeFrame = 
    XtVaCreateWidget("pipeFrame", xmFrameWidgetClass, controlArea,
		     XmNtopWidget, moduleNameLabel,
		     NULL);

  Widget portFrame = 
    XtVaCreateWidget("portFrame", xmFrameWidgetClass, controlArea,
		     XmNtopWidget, moduleNameLabel,
		     NULL);

  Widget pipeForm =
    XtCreateWidget("pipeForm", xmFormWidgetClass, pipeFrame, NULL, 0);

  Widget portForm =
    XtCreateWidget("portForm", xmFormWidgetClass, portFrame, NULL, 0);
		   
  Widget pipeLabel =
    XtCreateManagedWidget("pipeLabel", xmLabelWidgetClass, pipeForm, NULL, 0);

  Widget portLabel = 
    XtCreateManagedWidget("portLabel", xmLabelWidgetClass, portForm, NULL, 0);

  _createPipeList(pipeForm, pipeLabel);
  _createPortList(portForm, portLabel);

  XtManageChild(pipeForm);
  XtManageChild(portForm);
  XtManageChild(pipeFrame);
  XtManageChild(portFrame);
  XtManageChild(controlArea);
}



void PipePortBindingDialog::_clearPortBinding(int portNum)
{
  Pixel top_shadow, bottom_shadow, fg_ret, select_color;

  XmGetColors(XtScreen(PortButtons[0]), UnboundColorMap, UnboundBackground,
	      &fg_ret, &top_shadow, &bottom_shadow, &select_color);

  XtVaSetValues(PortButtons[portNum],
		XmNbackground, UnboundBackground,
		XmNtopShadowColor, top_shadow,
		XmNbottomShadowColor, bottom_shadow, 
		XmNselectColor,select_color, 
		XmNborderColor, fg_ret,
		NULL);

  PipeBoundToPort[portNum] = NO_PIPE_BOUND;
}


void PipePortBindingDialog::_createPipeButton(Widget pipeContainer, int pipeNum)
{
  const char *pipeName = PabloInterface->getInputPipeName(pipeNum);
  XmString pipeXName = XmStringCreateSimple((char *)pipeName);

  Widget pipeButton = XtVaCreateManagedWidget(form("pipe%d", pipeNum), 
					      xmPushButtonWidgetClass,
					      pipeContainer,
					      XmNlabelString, pipeXName,
					      XmNuserData, pipeNum, NULL);

  PipeButtons[pipeNum] = pipeButton;

  addCallback(pipeButton, XmNactivateCallback, &Callback::callback1,
	      this, NULL);

  if ( PabloInterface->getNumberRecordsOnInputPipe(pipeNum) == 0 ) {
      XtSetSensitive( pipeButton, False );  // desensitize if no records in pipe
  }

  XmStringFree(pipeXName);
}



void PipePortBindingDialog::_createPipeList(Widget parent, Widget label)
{
  Widget pipeWindow  = XtVaCreateWidget("pipeWindow", 
					xmScrolledWindowWidgetClass, 
					parent,
					XmNtopWidget, label,
					NULL, 0);

  Widget pipeContainer = XtCreateWidget("pipeContainer", 
					xmRowColumnWidgetClass, pipeWindow, 
					NULL, 0);


  const int numInPipes = PabloInterface->getNumberInputPipes();
  // allocate one extra for user input which is treated just like a pipe
  PipeButtons = (Widget *) malloc(sizeof(Widget)*(numInPipes+1));

  for (int i=0; i< numInPipes; i++) {
    _createPipeButton(pipeContainer, i);
  }

  // make the user input button
  Widget userInput = XtVaCreateManagedWidget("userInput", 
					     xmPushButtonWidgetClass, 
					     parent, 
					     XmNuserData, numInPipes,
					     NULL);

  PipeButtons[numInPipes] = userInput;

  addCallback(userInput, XmNactivateCallback, &Callback::callback1,
	      this, NULL);

  XtVaSetValues(pipeWindow, 
		XmNbottomWidget, userInput,
		NULL);

  XtManageChild(pipeContainer);
  XtManageChild(pipeWindow);
}


void PipePortBindingDialog::_createPortButton(Widget portContainer, 
					      int portNum)
{
  const char *portName = PabloInterface->getInputPortName(portNum);
  XmString portXName = XmStringCreateSimple((char *)portName);

  Widget portButton = XtVaCreateManagedWidget(form("port%d", portNum), 
					      xmPushButtonWidgetClass,
					      portContainer,
					      XmNlabelString, portXName,
					      XmNuserData, portNum,
					      NULL);
  PortButtons[portNum] = portButton;
  PipeBoundToPort[portNum] = NO_PIPE_BOUND;

  addCallback(portButton, XmNactivateCallback, &Callback::callback2,
	      this, NULL);

  XmStringFree(portXName);
}



void PipePortBindingDialog::_createPortList(Widget parent, Widget label)
{
  Widget portWindow  = XtVaCreateWidget("portWindow",  
					xmScrolledWindowWidgetClass, 
					parent,
					XmNtopWidget, label,
					NULL, 0);

  Widget portContainer = XtCreateWidget("portContainer", 
					xmRowColumnWidgetClass, portWindow, 
					NULL, 0);

  const int numInPorts = PabloInterface->getNumberInputPorts();
  PortButtons = (Widget *) malloc(sizeof(Widget)*numInPorts);
  PipeBoundToPort = (int *) malloc(sizeof(int)*numInPorts);

  for (int i=0; i< PabloInterface->getNumberInputPorts(); i++) {
    _createPortButton(portContainer, i);
  }

  XtManageChild(portContainer);
  XtManageChild(portWindow);
}



void PipePortBindingDialog::_getUnboundPortColors()
{
  XtVaGetValues(PortButtons[0], 
		XmNbackground, &UnboundBackground,
		XmNcolormap, &UnboundColorMap,
		NULL);
}


void PipePortBindingDialog::_initializeBindings()
{
  int pipe;
  Boolean portBound = False;

  StatusBoard->clearAllInputPipePortBindings();

  const int numInPipes = PabloInterface->getNumberInputPipes();
  const int numInPorts = PabloInterface->getNumberInputPorts();

  for (int portIdx=0; portIdx<numInPorts; portIdx++) {
    pipe = PabloInterface->getPipeBoundToInputPort(portIdx);

    if ((pipe >= 0) && (pipe <= numInPipes)) {
      _bindPipeToPort(pipe, portIdx);
      portBound = True;
    } else {
        _clearPortBinding(portIdx);
      }
  }

  if (! portBound) {
    XtSetSensitive(ContinueButton, False);
  }
}


void PipePortBindingDialog::_setDialogTitle()
{
  char buf[1024];
  
  sprintf(buf, "Bind Input Pipes to Input Ports");

  XtVaSetValues(DialogShell, 
		XmNtitle, buf, 
		NULL);
}



void PipePortBindingDialog::callback1(Widget selectedPipe, XtPointer ,
				      XtPointer )
{
  /* callback called when a pipe is selected */
  XtVaGetValues(selectedPipe, 
		XmNuserData, &CurrentPipeNum, 
		NULL);

  if (CurrentPortNum != NONE_SELECTED) {
    _bindPipeToPort(CurrentPipeNum, CurrentPortNum);
  }
}



void PipePortBindingDialog::callback2(Widget selectedPort, XtPointer ,
				      XtPointer )
{
  /* callback called when a port is selected */
  XtVaGetValues(selectedPort, 
		XmNuserData, &CurrentPortNum,
		NULL);

  if (CurrentPipeNum != NONE_SELECTED) {
    _bindPipeToPort(CurrentPipeNum, CurrentPortNum);
  }
}



void PipePortBindingDialog::callback3(Widget , XtPointer ,
				      XtPointer )
{
  /* callback called when reset is selected */
  /* since the binding information in Pablo isn't updated until the user
     chooses "continue", we can just reinitialize the bindings to give us
     the effect of "reset" */

  _initializeBindings();
  CurrentPipeNum = CurrentPortNum = NONE_SELECTED;
}



void PipePortBindingDialog::callback4(Widget , XtPointer ,
				      XtPointer )
{
  /* callback called when clear is selected */
  for (int port=0; port< PabloInterface->getNumberInputPorts(); port++) {
    _clearPortBinding(port);
  }

  StatusBoard->clearAllInputPipePortBindings();
  XtSetSensitive(ContinueButton, False);
}



void PipePortBindingDialog::callback5(Widget , XtPointer ,
				      XtPointer )
{
  /* callback called when continue is selected */

  DoneRunning = True;
}



void PipePortBindingDialog::helpCallback(Widget , XtPointer ,
				      XtPointer )
{
  /* callback called when help is selected */
  Pablo::HelpSystem()->giveHelpOn( "PipePortBindingDialog" );
}



void PipePortBindingDialog::run()
{
  DoneRunning = False;

  XtManageChild(DialogShell);

  while (!DoneRunning) {
    XEvent event;
    XtAppNextEvent(
		   XtWidgetToApplicationContext(DialogShell), &event );
    XtDispatchEvent( &event );
  }
  
  XtUnmanageChild(DialogShell);

  for (int port=0; port< PabloInterface->getNumberInputPorts(); port++) {
    PabloInterface->bindPipeToInputPort(PipeBoundToPort[port], port);
  }
}



/*
 * Initialize the static data
 */
const char *const PipePortBindingDialog::MY_CLASS = "PipePortBindingDialog";
