/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *               University of Illinois at Urbana-Champaign
 *               Department of Computer Science
 *               1304 W. Springfield Avenue
 *               Urbana, IL     61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *      All Rights Reserved.
 *
 * Author: Keith A. Shields (shields@cs.uiuc.edu)
 * 
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * RecordBindingDialog.cc : This module implements a custom dialog for
 *                          binding input record fields to input ports
 *                          and output record fields
 * 
 *
 */

#include <stdlib.h>

#ifdef ultrix
extern "C" {
/* for some reason, strdup isn't in <string.h> on ULTRIX V4.2 (Rev. 96) */
char* strdup(const char*);
}
#endif

#include "RecordBindingDialog.h"

#include "PabloHelpSystem.h"

RecordBindingDialog::RecordBindingDialog(Widget parentApp, 
					 InterfaceClass *pabloInterface,
					 ConfigBoard *statusBoard)
{
  _setClassName(MY_CLASS);
  PabloInterface = pabloInterface;
  StatusBoard = statusBoard;

  RecordWidgets = (RecordDisplayElements *) malloc(sizeof
						   (RecordDisplayElements));
  FieldButtons = (Widget *) malloc(sizeof(Widget));
  PortButtons = (Widget *) malloc(sizeof(Widget));
  OutputButtons = (Widget *) malloc(sizeof(Widget));

  CurrentPipe = NULL;
  CurrentRecord = NULL;
  ModuleResultSelected = False;

  CurrentFieldNum = CurrentPortNum = CurrentOutputNum =  CurrentRecordIdx =
    CurrentPipeIdx = NONE_SELECTED;

  DialogShell = XtCreatePopupShell("RecordBindingDialog", 
				   topLevelShellWidgetClass, parentApp, NULL, 
				   0);
				   
  Widget panedWindow  = XtVaCreateWidget("dialogPane",
					 xmPanedWindowWidgetClass, 
					 DialogShell, 
					 XmNsashWidth, 1, 
					 XmNsashHeight, 1,
					 NULL );

  _buildControlArea(panedWindow);
  _buildActionArea(panedWindow);

  _checkForContinue();

  if ((PabloInterface->allOutputPortTraitsSet()) &&
      (PabloInterface->getNumberOutputPorts() > 0)) {
	XtSetSensitive(ModuleResultButton, True);
      } else {
	XtSetSensitive(ModuleResultButton, False);
      }

  _setDialogTitle();

  XtManageChild(panedWindow);
}


RecordBindingDialog::~RecordBindingDialog()
{
  int recIdx, fieldIdx, portIdx, outIdx;
  int numInPipes = PabloInterface->getNumberInputPipes();

  for (int pipeIdx=0; pipeIdx<numInPipes; pipeIdx++) {
    for (recIdx=0; recIdx<Pipes[pipeIdx].numRecords; recIdx++) {
      for (fieldIdx=0; fieldIdx<Pipes[pipeIdx].recordList[recIdx].numFields;
	   fieldIdx++) {
	free(Pipes[pipeIdx].recordList[recIdx].fieldList[fieldIdx].fieldName);
	free(Pipes[pipeIdx].recordList[recIdx].fieldList[fieldIdx].
	     isBoundToPort);
	free(Pipes[pipeIdx].recordList[recIdx].fieldList[fieldIdx].
	     isBoundToOutput);
      }
      free(Pipes[pipeIdx].recordList[recIdx].fieldList);
      free(Pipes[pipeIdx].recordList[recIdx].recordName);
    }

    for (portIdx=0; portIdx<Pipes[pipeIdx].numPortsBoundTo; portIdx++) {
      free(Pipes[pipeIdx].portList[portIdx].portName);
    }

    for (outIdx=0; outIdx<Pipes[pipeIdx].numOutputsBoundTo; outIdx++) {
      free(Pipes[pipeIdx].outputList[outIdx].outputName);
    }

    free(Pipes[pipeIdx].pipeName);
    free(Pipes[pipeIdx].recordList);
    free(Pipes[pipeIdx].portList);
    free(Pipes[pipeIdx].outputList);
  }

  for (portIdx=0; portIdx<ModuleResult.numOutputPorts; portIdx++) {
      free(ModuleResult.outputPortList[portIdx].outputPortName);
      free(ModuleResult.outputPortList[portIdx].isBoundToOutput);
    }

  for (outIdx=0; outIdx<ModuleResult.numOutputFieldsBoundTo; outIdx++) {
    free(ModuleResult.outputFieldList[outIdx].outputName);
  }

  free(ModuleResult.outputPortList);
  free(ModuleResult.outputFieldList);

  free(Pipes);
  free(RecordWidgets);
  free(FieldButtons);
  free(PortButtons);
  free(OutputButtons);

  XtDestroyWidget(DialogShell);
}



Boolean RecordBindingDialog::_allOutputsSatisfied(int pipeIdx, int recordIdx)
{
  Boolean allOutputsSatisfied, outputIsSatisfied;
  int fieldIdx, outPortIdx, numFields;
  
  allOutputsSatisfied = True;

  if (pipeIdx == MODULE_RESULT) {
    for (int outIdx=0; outIdx<ModuleResult.numOutputFieldsBoundTo; outIdx++) {
      outputIsSatisfied = False;
      for (outPortIdx = 0; outPortIdx<ModuleResult.numOutputPorts; outPortIdx++) {
	if (ModuleResult.outputPortList[outPortIdx].isBoundToOutput[outIdx]) {
	      outputIsSatisfied = True;
	    }
      }
      if (!outputIsSatisfied) {
	allOutputsSatisfied = False;
      }
    }
  } else {
    for (int outIdx=0; outIdx<Pipes[pipeIdx].numOutputsBoundTo; outIdx++) {
      outputIsSatisfied = False;
      numFields = Pipes[pipeIdx].recordList[recordIdx].numFields;
      for (fieldIdx = 0; fieldIdx<numFields; fieldIdx++) {
	if (Pipes[pipeIdx].recordList[recordIdx].fieldList[fieldIdx].
	    isBoundToOutput[outIdx]) {
	      outputIsSatisfied = True;
	    }
      }
      if (!outputIsSatisfied) {
	allOutputsSatisfied = False;
      }
    }
  }
   
  return(allOutputsSatisfied);
}


Boolean RecordBindingDialog::_allPortsSatisfied(int pipeIdx, int recordIdx)
{
  Boolean allPortsBound, portIsBound;
  int fieldIdx, numFields;

  allPortsBound = True;
  for (int portIdx=0; portIdx<Pipes[pipeIdx].numPortsBoundTo; portIdx++) {
    portIsBound = False;
    numFields = Pipes[pipeIdx].recordList[recordIdx].numFields;
    for (fieldIdx = 0; fieldIdx<numFields; fieldIdx++) {
      if (Pipes[pipeIdx].recordList[recordIdx].fieldList[fieldIdx].
	  isBoundToPort[portIdx]) {
        portIsBound = True;
      }
    }
    if (!portIsBound) {
      allPortsBound = False;
    }
  }
  return(allPortsBound);
}


void RecordBindingDialog::_bindFieldToOutput(int field, int outField)
{
  /* if another field of this record is currently satisfying the output
     field, we must clear that binding */

  if (!ModuleResultSelected) {
    for (int fieldIdx=0; fieldIdx<CurrentRecord->numFields; fieldIdx++) {
      if (CurrentRecord->fieldList[fieldIdx].isBoundToOutput[outField]) {
	CurrentRecord->fieldList[fieldIdx].isBoundToOutput[outField] = False;
	
	PabloInterface->unlinkInputRecordFieldFromOutputField
	    (CurrentPipeIdx, CurrentRecordIdx, fieldIdx, outField);
      }
    }

    // make the binding and show as being bound 
    if (PabloInterface->linkInputRecordFieldToOutputField
	(CurrentPipeIdx, CurrentRecordIdx, field, outField)) 
      {
	CurrentRecord->fieldList[field].isBoundToOutput[outField] = True;
	_showAsBound(FieldButtons[field], OutputButtons[outField]);
      } else {
	CurrentRecord->fieldList[field].isBoundToOutput[outField] = False;
	_showAsUnboundOutput(OutputButtons[outField]);	
      }

    if ((_allPortsSatisfied(CurrentPipeIdx, CurrentRecordIdx)) && 
	(_allOutputsSatisfied(CurrentPipeIdx, CurrentRecordIdx))) {
	  XtSetSensitive(BindingDone, True);
	}
  } else {
    for (int outIdx=0; outIdx<ModuleResult.numOutputPorts; outIdx++) {
      if(ModuleResult.outputPortList[outIdx].isBoundToOutput[outField]) {

	ModuleResult.outputPortList[outIdx].isBoundToOutput[outField] = False;
	PabloInterface->unlinkOutputPortFromOutputField(outIdx, outField);
      }
    }

    if (PabloInterface->linkOutputPortToOutputField(field, outField)) {
      ModuleResult.outputPortList[field].isBoundToOutput[outField] = True;
      _showAsBound(FieldButtons[field], OutputButtons[outField]);

    } else {
      ModuleResult.outputPortList[field].isBoundToOutput[outField] = False;
      _showAsUnboundOutput(OutputButtons[outField]);	
    }

    if (_allOutputsSatisfied(CurrentPipeIdx, CurrentRecordIdx)) {
      XtSetSensitive(BindingDone, True);
    }
  }

  if ((PabloInterface->allOutputPortTraitsSet()) &&
      (PabloInterface->getNumberOutputPorts() > 0)) {
	XtSetSensitive(ModuleResultButton, True);
      } else {
	XtSetSensitive(ModuleResultButton, False);
      }

  CurrentFieldNum = CurrentPortNum = CurrentOutputNum =  NONE_SELECTED;
}



void RecordBindingDialog::_bindFieldToPort(int field, int port)
{
  for (int fieldIdx=0; fieldIdx<CurrentRecord->numFields; fieldIdx++) {
    if (CurrentRecord->fieldList[fieldIdx].isBoundToPort[port]) {

      CurrentRecord->fieldList[fieldIdx].isBoundToPort[port] = False;
      PabloInterface->unlinkInputRecordFieldFromInputPort
	(CurrentPipeIdx, CurrentRecordIdx, fieldIdx, port);
    }
  }

  if (PabloInterface->linkInputRecordFieldToInputPort
      (CurrentPipeIdx, CurrentRecordIdx, field, port)) 
    {
      CurrentRecord->fieldList[field].isBoundToPort[port] = True;
      _showAsBound(FieldButtons[field], PortButtons[port]);
    } else {
      CurrentRecord->fieldList[field].isBoundToPort[port] = False;
      _showAsUnboundPort(PortButtons[port]);
    }

  if ((_allPortsSatisfied(CurrentPipeIdx, CurrentRecordIdx)) && 
      (_allOutputsSatisfied(CurrentPipeIdx, CurrentRecordIdx))) {
	XtSetSensitive(BindingDone, True);
      }

  if ((PabloInterface->allOutputPortTraitsSet()) &&
      (PabloInterface->getNumberOutputPorts() > 0)) {
	XtSetSensitive(ModuleResultButton, True);
      } else {
	XtSetSensitive(ModuleResultButton, False);
      }

  CurrentFieldNum = CurrentPortNum = CurrentOutputNum =  NONE_SELECTED;
}



void RecordBindingDialog::_buildActionArea(Widget panedWindow)
{
  Widget actionArea = XtCreateWidget("actionArea", xmFormWidgetClass, 
				     panedWindow, NULL, 0);


  ContinueButton = 
    XtVaCreateManagedWidget("continue", xmPushButtonWidgetClass, actionArea,
			    XmNsensitive, False,
			    NULL);

  Widget reset = XtCreateManagedWidget("reset", xmPushButtonWidgetClass,
				       actionArea, NULL, 0);

  Widget help = XtCreateManagedWidget("help", xmPushButtonWidgetClass,
				      actionArea, NULL, 0);

  addCallback(ContinueButton, XmNactivateCallback, &Callback::callback8, this,
	      NULL);

  addCallback(reset, XmNactivateCallback, &Callback::callback9, this, NULL);

  addCallback(help, XmNactivateCallback, &Callback::helpCallback, this, NULL);

  /* fix height of action area */
  Dimension h; 
  XtVaGetValues(reset, XmNheight, &h, NULL);
  XtVaSetValues(actionArea, 
		XmNpaneMaximum, h, 
		XmNpaneMinimum, h,
		NULL);

  XtManageChild( actionArea );
}



void RecordBindingDialog::_buildBindingFrame(Widget controlArea, 
					     Widget nameLabel)
{
  Widget bindingFrame = 
    XtVaCreateWidget("bindingFrame", xmFrameWidgetClass, controlArea, 
		     XmNtopWidget, nameLabel,
		     NULL);

  Widget bindingForm = XtCreateWidget("bindingForm", xmFormWidgetClass,
				      bindingFrame, NULL, 0);
  Widget portLabel = XtCreateManagedWidget("portLabel", xmLabelWidgetClass,
					   bindingForm, NULL, 0);
  
  Widget outputLabel = XtCreateManagedWidget("outputLabel", 
					     xmLabelWidgetClass, bindingForm, 
					     NULL, 0);

  Widget portWindow = XtVaCreateManagedWidget("portWindow", 
					       xmScrolledWindowWidgetClass, 
					       bindingForm,
					       XmNtopWidget, portLabel,
					       NULL);

  Widget outputWindow = XtVaCreateManagedWidget("outputWindow", 
						xmScrolledWindowWidgetClass, 
						bindingForm,
						XmNtopWidget, outputLabel,
						NULL);

  PortContainer =  XtCreateWidget("portContainer",  xmRowColumnWidgetClass, 
				  portWindow, NULL, 0);

  OutputContainer =  XtCreateWidget("outputContainer", 
				      xmRowColumnWidgetClass, outputWindow, 
				      NULL, 0);

  BindingDone = XtCreateManagedWidget("done", xmPushButtonWidgetClass, 
				      bindingForm, NULL, 0);

  BindingClear = XtCreateManagedWidget("clear", xmPushButtonWidgetClass, 
				       bindingForm, NULL, 0);

  addCallback(BindingDone, XmNactivateCallback, &Callback::callback6, this, 
	      NULL);
  addCallback(BindingClear, XmNactivateCallback, &Callback::callback7, this, 
	      NULL);

  XtSetSensitive(BindingDone, False);
  XtSetSensitive(BindingClear, False);

  XtManageChild(PortContainer);
  XtManageChild(OutputContainer);
  XtManageChild(portWindow);
  XtManageChild(outputWindow);
  XtManageChild(bindingForm);
  XtManageChild(bindingFrame);
}


void RecordBindingDialog::_buildControlArea(Widget panedWindow)
{
  Widget controlArea = XtCreateWidget("controlArea", xmFormWidgetClass, 
				      panedWindow, NULL, 0 );

  char * moduleName = (char *) PabloInterface->getModuleName();
  XmString moduleNameX = XmStringCreateSimple(moduleName);
  Widget moduleNameLabel = XtVaCreateManagedWidget("moduleNameLabel", 
                                                   xmLabelWidgetClass, 
                                                   controlArea,
                                                   XmNlabelString, moduleNameX,
                                                   NULL);
  XmStringFree(moduleNameX);

  Widget currentPipeLabel =
    XtVaCreateManagedWidget("currentPipeLabel", xmLabelWidgetClass,
			    controlArea, 
			    XmNtopWidget, moduleNameLabel,
			    NULL);

  CurrentPipeNameLabel = 
    XtVaCreateManagedWidget("currentPipeNameLabel", xmLabelWidgetClass,
			    controlArea,
			    XmNtopWidget, moduleNameLabel,
			    XmNleftWidget, currentPipeLabel,
			    NULL);

  Widget currentRecordLabel = 
    XtVaCreateManagedWidget("currentRecordLabel", xmLabelWidgetClass,
			    controlArea,
			    XmNtopWidget,moduleNameLabel,
			    NULL);

  CurrentRecordNameLabel = 
    XtVaCreateManagedWidget("currentRecordNameLabel", xmLabelWidgetClass,
			    controlArea,
			    XmNtopWidget, moduleNameLabel,
			    XmNleftWidget, currentRecordLabel,
			    NULL);

  _buildBindingFrame(controlArea, currentPipeLabel);
  _buildFieldFrame(controlArea, currentPipeLabel);
  _buildCopyDiscardFrame(controlArea);
  _buildRecordFrame(controlArea, currentPipeLabel);
  _buildPipeFrame(controlArea, currentPipeLabel);

  XtManageChild(controlArea);
}



void RecordBindingDialog::_buildCopyDiscardFrame(Widget controlArea)
{  
  CopyDiscardFrame = 
    XtCreateWidget("copyDiscardFrame", xmFrameWidgetClass, controlArea, 
		   NULL, 0);

  Widget copyDiscardForm = XtCreateWidget("copyDiscardForm", xmFormWidgetClass,
					  CopyDiscardFrame, NULL, 0);

  Widget copyThruButton = 
    XtCreateManagedWidget("copyThruButton", xmPushButtonWidgetClass,
			  copyDiscardForm, NULL, 0);

  Widget discardButton = 
    XtVaCreateManagedWidget("discardButton", xmPushButtonWidgetClass,
			    copyDiscardForm, 
			    XmNtopWidget, copyThruButton, 
			    NULL);

  addCallback(copyThruButton, XmNactivateCallback, &Callback::callback11, 
	      this, NULL);

  addCallback(discardButton, XmNactivateCallback, &Callback::callback12, this,
	      NULL);

  XtManageChild(copyDiscardForm);
  XtManageChild(CopyDiscardFrame);
}



void RecordBindingDialog::_buildFieldFrame(Widget controlArea, 
					   Widget nameLabel)
{
  FieldFrame = 
    XtVaCreateWidget("fieldFrame", xmFrameWidgetClass, controlArea, 
		     XmNtopWidget, nameLabel,
		     NULL);

  Widget fieldForm = XtCreateWidget("fieldForm", xmFormWidgetClass,
				    FieldFrame, NULL, 0);

  Widget fieldLabel = XtCreateManagedWidget("fieldLabel", xmLabelWidgetClass,
					     fieldForm, NULL, 0);

  Widget fieldWindow = XtVaCreateManagedWidget("fieldWindow", 
					       xmScrolledWindowWidgetClass, 
					       fieldForm,
					       XmNtopWidget, fieldLabel,
					       NULL);

  FieldContainer =  XtCreateWidget("fieldContainer", 
				   xmRowColumnWidgetClass, fieldWindow, 
				   NULL, 0);
  
  XtManageChild(FieldContainer);
  XtManageChild(fieldForm);
  XtManageChild(FieldFrame);
}



void RecordBindingDialog::_buildPipeFrame(Widget controlArea, Widget nameLabel)
{
  PipeFrame = 
    XtVaCreateWidget("pipeFrame", xmFrameWidgetClass, controlArea, 
		     XmNtopWidget, nameLabel,
		     NULL);

  Widget pipeForm = XtCreateWidget("pipeForm", xmFormWidgetClass,
				   PipeFrame, NULL, 0);
  
  Widget pipeLabel = XtCreateManagedWidget("pipeLabel", xmLabelWidgetClass,
					   pipeForm, NULL, 0);

  ModuleResultButton = 
    XtCreateManagedWidget("moduleResultButton", xmPushButtonWidgetClass, 
			  pipeForm, NULL, 0);

  XtSetSensitive(ModuleResultButton, False);

  Widget pipeWindow = XtVaCreateWidget("pipeWindow", 
				       xmScrolledWindowWidgetClass, 
				       pipeForm,
				       XmNtopWidget, pipeLabel,
				       XmNbottomWidget, ModuleResultButton,
				       NULL);

  Widget pipeContainer =  XtCreateWidget("pipeContainer", 
					 xmRowColumnWidgetClass, pipeWindow, 
					 NULL, 0);

  addCallback(ModuleResultButton, XmNactivateCallback, &Callback::callback10, 
	      this, NULL);

  _setupPipes(pipeContainer);
  _setupModuleResultBindings();

  XtManageChild(pipeLabel);
  XtManageChild(pipeWindow);
  XtManageChild(pipeContainer);
  XtManageChild(pipeForm);
  XtManageChild(PipeFrame);
}



void RecordBindingDialog::_buildRecordFrame(Widget controlArea,
					    Widget nameLabel)
{
  RecordFrame = 
    XtVaCreateWidget("recordFrame", xmFrameWidgetClass, controlArea, 
		     XmNtopWidget, nameLabel,
		     XmNbottomWidget, CopyDiscardFrame,
		     NULL);

  Widget recordForm = XtCreateWidget("recordForm", xmFormWidgetClass,
				      RecordFrame, NULL, 0);

  Widget recordLabel = XtCreateManagedWidget("recordLabel", xmLabelWidgetClass,
					     recordForm, NULL, 0);


  Widget recordWindow = XtVaCreateManagedWidget("recordWindow", 
						xmScrolledWindowWidgetClass, 
						recordForm,
						XmNtopWidget, recordLabel,
						NULL);

  RecordContainer =  
    XtVaCreateWidget("recordContainer", xmFormWidgetClass, recordWindow, 
		     NULL);
		     

  XtManageChild(recordLabel);
  XtManageChild(RecordContainer);
  XtManageChild(recordWindow);
  XtManageChild(recordForm);
  XtManageChild(RecordFrame);
}



void RecordBindingDialog::_checkForContinue()
{
  int recIdx, numInPipes = PabloInterface->getNumberInputPipes();
  Boolean allPipesOK = True; 
  Boolean pipeOK;

  for(int pipeIdx=0; pipeIdx < numInPipes; pipeIdx++) {
    pipeOK = False;
    for (recIdx=0; recIdx<Pipes[pipeIdx].numRecords; recIdx++) {
      if ((_allOutputsSatisfied(pipeIdx, recIdx)) &&
	  (_allPortsSatisfied(pipeIdx, recIdx))) {
	    pipeOK = True;
	  }
    }
    if ( ( recIdx > 0 )  && !( pipeOK ) ) {  // If unsatisfied ports for pipe...
      allPipesOK = False;
    }
  }

  if (PabloInterface->getNumberOutputPorts() > 0) {
    if (!(_allOutputsSatisfied(MODULE_RESULT, 0))) {
      allPipesOK = False;
    }
  }

  if (allPipesOK) {
    XtSetSensitive(ContinueButton, True);
  } else {
    XtSetSensitive(ContinueButton, False);
  }
}



void RecordBindingDialog::_clearAllBindingsForCurrentRecord()
{
  int fieldIdx, portIdx, outputIdx, outPortIdx;

  if (!ModuleResultSelected) {
    if (CurrentRecord != NULL) {
      for (fieldIdx=0; fieldIdx<CurrentRecord->numFields; fieldIdx++) {
	for (portIdx=0; portIdx<CurrentPipe->numPortsBoundTo; portIdx++) {
	  if ( CurrentRecord->fieldList[fieldIdx].isBoundToPort[portIdx] ) {
	    CurrentRecord->fieldList[fieldIdx].isBoundToPort[portIdx] = False;
	    _clearFieldToPortBinding(fieldIdx, portIdx);
	  }
	}

	for (outputIdx=0; outputIdx<CurrentPipe->numOutputsBoundTo; 
	     outputIdx++) {
	       _clearFieldToOutputBinding(fieldIdx, outputIdx);
	     }
      }
    }
  } else {
    for (outPortIdx=0; outPortIdx<ModuleResult.numOutputPorts; outPortIdx++){
      for (outputIdx=0; outputIdx<ModuleResult.numOutputFieldsBoundTo; 
	   outputIdx++) {
	     _clearFieldToOutputBinding(outPortIdx, outputIdx);
      }
    }
  }
}


void RecordBindingDialog::_clearFieldToOutputBinding(int field, int output)
{
  if (!ModuleResultSelected) {
    if ( CurrentRecord->fieldList[field].isBoundToOutput[output] ) {
        PabloInterface->unlinkInputRecordFieldFromOutputField
	  (CurrentPipeIdx, CurrentRecordIdx, field, output);

        CurrentRecord->fieldList[field].isBoundToOutput[output] = False;
        _showAsUnboundOutput(OutputButtons[output]);
    }
  } else {
    if ( ModuleResult.outputPortList[field].isBoundToOutput[output] ) {
        PabloInterface->unlinkOutputPortFromOutputField(field, output);
        ModuleResult.outputPortList[field].isBoundToOutput[output] = False;
        _showAsUnboundOutput(OutputButtons[output]);
    }
  }
}



void RecordBindingDialog::_clearFieldToPortBinding(int fieldNum, int portNum)
{
  PabloInterface->unlinkInputRecordFieldFromInputPort
    (CurrentPipeIdx, CurrentRecordIdx, fieldNum, portNum);

  CurrentRecord->fieldList[fieldNum].isBoundToPort[portNum] = False;
  _showAsUnboundPort(PortButtons[portNum]);
}



void RecordBindingDialog::_destroyChildren(Widget parentWidget)
{
  int numChildren;
  WidgetList children;

  XtVaGetValues(parentWidget, 
		XmNchildren, &children,
		XmNnumChildren, &numChildren,
		NULL);

  for (int i=0; i<numChildren; i++) {
    XtDestroyWidget(children[i]);
  }
}


void RecordBindingDialog::_getUnboundOutputColors()
{
  XtVaGetValues(OutputButtons[0],
		XmNbackground, &UnboundOutputBackground,
		XmNcolormap, &UnboundOutputColorMap,
		NULL);
}


void RecordBindingDialog::_getUnboundPortColors()
{
  XtVaGetValues(PortButtons[0],
		XmNbackground, &UnboundPortBackground,
		XmNcolormap, &UnboundPortColorMap,
		NULL);
}



void RecordBindingDialog::_makeFieldButton(int fieldNum, char *fieldName)
{

  XmString fieldNameX = XmStringCreateSimple(fieldName);

  Widget fieldButton = XtVaCreateManagedWidget(form("field%d", fieldNum), 
						xmPushButtonWidgetClass,
						FieldContainer,
						XmNlabelString, fieldNameX,
						XmNuserData, 
						(XtPointer) fieldNum,
						NULL);

  addCallback(fieldButton, XmNactivateCallback, &Callback::callback3, 
	      this, NULL);
  FieldButtons[fieldNum] = fieldButton;
  XmStringFree(fieldNameX);
}



void RecordBindingDialog::_makeOutputButton(int outputNum, char *outputName)
{
  XmString outputNameX = XmStringCreateSimple(outputName);

  Widget outputButton = XtVaCreateManagedWidget(form("output%d", outputNum), 
						xmPushButtonWidgetClass,
						OutputContainer,
						XmNlabelString, outputNameX,
						XmNuserData, 
						(XtPointer) outputNum,
						NULL);

  addCallback(outputButton, XmNactivateCallback, &Callback::callback5, 
	      this, NULL);
  OutputButtons[outputNum] = outputButton;
  XmStringFree(outputNameX);
}



void RecordBindingDialog::_makePortButton(int portNum, char *portName)
{
  XmString portNameX = XmStringCreateSimple(portName);

  Widget portButton = XtVaCreateManagedWidget(form("port%d", portNum), 
						xmPushButtonWidgetClass,
						PortContainer,
						XmNlabelString, portNameX,
						XmNuserData, 
						(XtPointer) portNum,
						NULL);

  
  addCallback(portButton, XmNactivateCallback, &Callback::callback4, 
	      this, NULL);
  PortButtons[portNum] = portButton;
  XmStringFree(portNameX);
}



void RecordBindingDialog::_makeRecordWidgets(int recordNum, char* recordName,
					     Widget topWidget)
{
  XmString feXstr = XmStringCreateSimple("E");
  XmString ctXstr = XmStringCreateSimple("C");
  XmString recordNameX = XmStringCreateSimple(recordName);

  Widget recordInfoForm;

  if ( topWidget == RecordContainer ) {
    recordInfoForm = XtVaCreateManagedWidget("recordInfoForm", 
		            xmFormWidgetClass, RecordContainer,
		            XmNtopAttachment, XmATTACH_FORM,
		            NULL);
  } else {
    recordInfoForm = XtVaCreateManagedWidget("recordInfoForm", 
		            xmFormWidgetClass, RecordContainer,
		            XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, topWidget,
		            NULL);
  }

  Widget copyThruLabel = 
    XtVaCreateManagedWidget("CTLabel",
			    xmLabelWidgetClass, recordInfoForm,
			    XmNlabelString, ctXstr,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNborderWidth, 0,
			    XmNmarginWidth, 0,
			    NULL);

  Widget fieldExtractLabel = 
    XtVaCreateManagedWidget("FELabel",
			    xmLabelWidgetClass, recordInfoForm,
			    XmNlabelString, feXstr,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_WIDGET,
			    XmNleftWidget, copyThruLabel,
			    XmNborderWidth, 0,
			    XmNmarginWidth, 0,
			    NULL);

  Widget recordButton =
    XtVaCreateManagedWidget("recordButton",
			    xmPushButtonWidgetClass, recordInfoForm,
			    XmNlabelString, recordNameX,
			    XmNuserData, (XtPointer) recordNum,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNleftAttachment, XmATTACH_WIDGET,
			    XmNleftWidget, fieldExtractLabel,
			    NULL);

  addCallback(recordButton, XmNactivateCallback, &Callback::callback2, 
	      this, NULL);

  RecordWidgets[recordNum].widgetContainer = recordInfoForm;
  RecordWidgets[recordNum].copyThruLabel = copyThruLabel;
  RecordWidgets[recordNum].fieldExtractLabel = fieldExtractLabel;
  RecordWidgets[recordNum].recordButton = recordButton;

  _updateRecordStatusLabels(recordNum);

  XmStringFree(feXstr);
  XmStringFree(ctXstr);
  XmStringFree(recordNameX);

}



void RecordBindingDialog::_setCurrentPipeLabel(char *pipeName)
{
  XmString pipeNameX = XmStringCreateSimple(pipeName);
  XtVaSetValues(CurrentPipeNameLabel, 
		XmNlabelString, pipeNameX,
		NULL);
  XmStringFree(pipeNameX);
}



void RecordBindingDialog::_setCurrentRecordLabel(char *recordName)
{
  XmString recordNameX = XmStringCreateSimple(recordName);
  XtVaSetValues(CurrentRecordNameLabel, 
		XmNlabelString, recordNameX,
		NULL);
  XmStringFree(recordNameX);
}



void RecordBindingDialog::_setDialogTitle()
{
  char buf[1024];
  
  sprintf(buf, "Bind Records to Input Ports and Output Record Fields");
  XtVaSetValues(DialogShell, XmNtitle, buf, NULL);
}


void RecordBindingDialog::_setupFieldBooleans(int pipeNum, int recordNum, 
					      int fieldNum)
{
  FieldEntry currentField;

  currentField = Pipes[pipeNum].recordList[recordNum].fieldList[fieldNum];

  for (int port=0; port<Pipes[pipeNum].numPortsBoundTo; port++) {
    currentField.isBoundToPort[port] = 	PabloInterface->
      isInputRecordFieldBoundToInputPort(pipeNum, recordNum, fieldNum, port);
  }

  for (int output=0; output<Pipes[pipeNum].numOutputsBoundTo; output++) {
    currentField.isBoundToOutput[output] =  PabloInterface->
      isInputRecordFieldBoundToOutputField(pipeNum, recordNum, fieldNum, 
					   output);
  }
}



void RecordBindingDialog::_setupFields(int pipeNum, int recordNum)
{
  int numFields = PabloInterface->
    getNumberFieldsInInputPipeRecord(pipeNum, recordNum);

  Pipes[pipeNum].recordList[recordNum].numFields = numFields;
  Pipes[pipeNum].recordList[recordNum].fieldList =
    (FieldEntry *) malloc(sizeof(FieldEntry)*numFields);
    
  for (int fieldIdx=0; fieldIdx<numFields; fieldIdx++) {
    Pipes[pipeNum].recordList[recordNum].fieldList[fieldIdx].fieldName =
      strdup((char *)PabloInterface->getNameOfFieldInInputPipeRecord
	     (pipeNum, recordNum, fieldIdx));
    Pipes[pipeNum].recordList[recordNum].fieldList[fieldIdx].isBoundToPort =
      (Boolean *) malloc(sizeof(Boolean)*Pipes[pipeNum].numPortsBoundTo);
    Pipes[pipeNum].recordList[recordNum].fieldList[fieldIdx].isBoundToOutput =
      (Boolean *) malloc(sizeof(Boolean)*Pipes[pipeNum].numOutputsBoundTo);
    _setupFieldBooleans(pipeNum, recordNum, fieldIdx);
  }
}



void RecordBindingDialog::_setupModuleResultBindings()
{
  char *outPortName, *outFieldName;

  int numOutFieldsBoundTo = 
    PabloInterface->getNumberOutputFieldsBoundToInputPipe(MODULE_RESULT);

  ModuleResult.numOutputFieldsBoundTo = numOutFieldsBoundTo;

  ModuleResult.outputFieldList = 
    (OutputEntry *) malloc (sizeof(OutputEntry)*numOutFieldsBoundTo);

  for (int outFieldIdx=0; outFieldIdx<numOutFieldsBoundTo; outFieldIdx++) {
    outFieldName = (char *) PabloInterface->
      getNameOfOutputFieldBoundToInputPipe(MODULE_RESULT,outFieldIdx);
    ModuleResult.outputFieldList[outFieldIdx].outputName = 
      strdup(outFieldName);
  }

  int numOutPorts = PabloInterface->getNumberOutputPorts();
  ModuleResult.numOutputPorts = numOutPorts;

  ModuleResult.outputPortList = 
    (OutputPortEntry *) malloc (sizeof(OutputPortEntry)*numOutPorts);

  for (int outPortIdx=0; outPortIdx<numOutPorts; outPortIdx++) {
    outPortName = (char *)PabloInterface->getOutputPortName(outPortIdx);
    ModuleResult.outputPortList[outPortIdx].outputPortName = 
      strdup(outPortName);
    
    ModuleResult.outputPortList[outPortIdx].isBoundToOutput = 
      (Boolean *) malloc(sizeof(Boolean)*numOutFieldsBoundTo);

    for (int outFieldIdx=0; outFieldIdx<numOutFieldsBoundTo; outFieldIdx++) {
      ModuleResult.outputPortList[outPortIdx].isBoundToOutput[outFieldIdx] =  
	PabloInterface->isOutputPortBoundToOutputField
	  (outPortIdx, outFieldIdx);
    }
  }
}



void RecordBindingDialog::_setupOutputs(int pipeNum)
{
  int numOutputs = PabloInterface->
    getNumberOutputFieldsBoundToInputPipe(pipeNum);

  Pipes[pipeNum].numOutputsBoundTo = numOutputs;
  Pipes[pipeNum].outputList = 
    (OutputEntry *) malloc(sizeof(OutputEntry)*numOutputs);

  for (int outputIdx=0; outputIdx<numOutputs; outputIdx++) {
    Pipes[pipeNum].outputList[outputIdx].outputName = 
      strdup((char *)PabloInterface->getNameOfOutputFieldBoundToInputPipe
	     (pipeNum, outputIdx));
  }
}



void RecordBindingDialog::_setupPipes(Widget pipeContainer)
{
  Widget pipeButton;
  char *pipeName;
  XmString pipeNameX;

  int numInPipes = PabloInterface->getNumberInputPipes();
  Pipes = (PipeEntry *) malloc(sizeof(PipeEntry)*numInPipes);

  for (int pipeIdx=0; pipeIdx<numInPipes; pipeIdx++) {
    pipeName = (char *) PabloInterface->getInputPipeName(pipeIdx);
    pipeNameX = XmStringCreateSimple(pipeName);
    pipeButton = XtVaCreateManagedWidget(form("pipe%d",pipeIdx), 
					 xmPushButtonWidgetClass, 
					 pipeContainer,
					 XmNlabelString, pipeNameX,
					 XmNuserData, (XtPointer) pipeIdx,
					 NULL);

    addCallback(pipeButton, XmNactivateCallback, &Callback::callback1, 
		this, NULL);

    if ( PabloInterface->getNumberRecordsOnInputPipe(pipeIdx) == 0 ) {
      XtSetSensitive( pipeButton, False );  // desensitize if no records in pipe
    }

    Pipes[pipeIdx].pipeName = strdup(pipeName);
    _setupPorts(pipeIdx);
    _setupOutputs(pipeIdx);
    _setupRecords(pipeIdx);
    XmStringFree(pipeNameX);
  }
}



void RecordBindingDialog::_setupPorts(int pipeNum)
{
  int numPorts = PabloInterface->getNumberInputPortsBoundToInputPipe(pipeNum);

  Pipes[pipeNum].numPortsBoundTo = numPorts;
  Pipes[pipeNum].portList = 
    (PortEntry *) malloc(sizeof(PortEntry)*numPorts);

  for (int portIdx=0; portIdx<numPorts; portIdx++) {
    Pipes[pipeNum].portList[portIdx].portName = 
      strdup((char *)PabloInterface->getNameOfInputPortBoundToInputPipe
	     (pipeNum, portIdx));
  }
}



void RecordBindingDialog::_setupRecords(int pipeNum)
{
  char *recordName;
  int numRecords = PabloInterface->getNumberRecordsOnInputPipe(pipeNum);

  Pipes[pipeNum].numRecords = numRecords;
  Pipes[pipeNum].recordList = 
    (RecordEntry *) malloc(sizeof(RecordEntry)*numRecords);

  for (int recIdx=0; recIdx<numRecords; recIdx++) {

    recordName = (char *) PabloInterface->
      getNameOfRecordOnInputPipe(pipeNum, recIdx);

    Pipes[pipeNum].recordList[recIdx].recordName = strdup(recordName);
    
    Pipes[pipeNum].recordList[recIdx].status = 
      PabloInterface->getStatusOfRecordOnInputPipe(pipeNum, recIdx);

    _setupFields(pipeNum, recIdx);
  }
}



void RecordBindingDialog::_showAsBound(Widget colorSource, Widget colorDest)
{
  Pixel bg_color, top_shadow, bottom_shadow, fg_ret, select_color;
  Colormap cmap;

  XtVaGetValues(colorSource,
                XmNbackground, &bg_color, 
                XmNcolormap, &cmap,
                NULL);

  XmGetColors(XtScreen(colorSource), cmap, bg_color, &fg_ret, 
              &top_shadow, &bottom_shadow, &select_color);

  XtVaSetValues(colorDest,
		XmNbackground, bg_color,
		XmNtopShadowColor, top_shadow,
		XmNbottomShadowColor, bottom_shadow, 
		XmNselectColor,select_color, 
		XmNborderColor, fg_ret,
		NULL);
}


void RecordBindingDialog::_showAsUnboundOutput(Widget colorDest)
{
  Pixel top_shadow, bottom_shadow, fg_ret, select_color;

  // get unbound output colors and give to colorDest
  XmGetColors(XtScreen(OutputButtons[0]), UnboundOutputColorMap, 
	      UnboundOutputBackground, &fg_ret, &top_shadow, &bottom_shadow, 
	      &select_color);
  
  XtVaSetValues(colorDest,
		XmNbackground, UnboundOutputBackground,
		XmNtopShadowColor, top_shadow,
		XmNbottomShadowColor, bottom_shadow, 
		XmNselectColor,select_color, 
		XmNborderColor, fg_ret,
		NULL);
}



void RecordBindingDialog::_showAsUnboundPort(Widget colorDest)
{
  Pixel top_shadow, bottom_shadow, fg_ret, select_color;

  // get unbound port colors

    XmGetColors(XtScreen(PortButtons[0]), UnboundPortColorMap, 
		UnboundPortBackground, &fg_ret, &top_shadow, &bottom_shadow, 
		&select_color);

    XtVaSetValues(colorDest,
		  XmNbackground, UnboundPortBackground,
		  XmNtopShadowColor, top_shadow,
		  XmNbottomShadowColor, bottom_shadow, 
		  XmNselectColor,select_color, 
		  XmNborderColor, fg_ret,
		  NULL);
}



void RecordBindingDialog::_updateRecordStatus(int recIdx)
{
  PabloInterface->
    setStatusOfRecordOnInputPipe(CurrentPipeIdx, recIdx, Pipes[CurrentPipeIdx].
				 recordList[recIdx].status);

  _updateRecordStatusLabels(recIdx);
}



void RecordBindingDialog::_updateRecordStatusLabels(int recIdx)
{
  XmString copyThruLabel, fieldExtractLabel;

  switch (Pipes[CurrentPipeIdx].recordList[recIdx].status) {

  case ActionCopyThruAndExtract:
             copyThruLabel = XmStringCreateSimple("C");
	     fieldExtractLabel = XmStringCreateSimple("E");
	     XtVaSetValues(RecordWidgets[recIdx].copyThruLabel, 
			   XmNlabelString, copyThruLabel, NULL);
	     XtVaSetValues(RecordWidgets[recIdx].fieldExtractLabel, 
			   XmNlabelString, fieldExtractLabel, NULL);
	     break;

  case ActionDiscard:
	     copyThruLabel = XmStringCreateSimple(" ");
	     fieldExtractLabel = XmStringCreateSimple(" ");
	     XtVaSetValues(RecordWidgets[recIdx].copyThruLabel, 
			   XmNlabelString, copyThruLabel, NULL);
	     XtVaSetValues(RecordWidgets[recIdx].fieldExtractLabel, 
			   XmNlabelString, fieldExtractLabel, NULL);
	     break;

  case ActionCopyThru:
	     copyThruLabel = XmStringCreateSimple("C");
	     fieldExtractLabel = XmStringCreateSimple(" ");
	     XtVaSetValues(RecordWidgets[recIdx].copyThruLabel, 
			   XmNlabelString, copyThruLabel, NULL);
	     XtVaSetValues(RecordWidgets[recIdx].fieldExtractLabel, 
			   XmNlabelString, fieldExtractLabel, NULL);
	     break;

  case ActionExtract:
	     copyThruLabel = XmStringCreateSimple(" ");
	     fieldExtractLabel = XmStringCreateSimple("E");
	     XtVaSetValues(RecordWidgets[recIdx].copyThruLabel, 
			   XmNlabelString, copyThruLabel, NULL);
	     XtVaSetValues(RecordWidgets[recIdx].fieldExtractLabel, 
			   XmNlabelString, fieldExtractLabel, NULL);
	     break;
  }
}



void RecordBindingDialog::callback1(Widget pipeButton, XtPointer ,
				    XtPointer )
{
  /* callback called when an input pipe button is pressed */

  int pipeIdx;

  ModuleResultSelected = False;

  XtVaGetValues(pipeButton, XmNuserData, &pipeIdx, NULL);
  CurrentPipe = &Pipes[pipeIdx];
  CurrentPipeIdx = pipeIdx;

  _destroyChildren(RecordContainer);
  _destroyChildren(FieldContainer);
  _destroyChildren(PortContainer);
  _destroyChildren(OutputContainer);

  _setCurrentPipeLabel(CurrentPipe->pipeName);

  int numRecords = CurrentPipe->numRecords;
  RecordWidgets = (RecordDisplayElements *) 
    realloc(RecordWidgets, sizeof(RecordDisplayElements)*numRecords);

  XtUnmanageChild(RecordContainer);

  _makeRecordWidgets(0, CurrentPipe->recordList[0].recordName, RecordContainer);
  for (int recIdx=1; recIdx<numRecords; recIdx++) {
    _makeRecordWidgets(recIdx, CurrentPipe->recordList[recIdx].recordName,
		       RecordWidgets[recIdx-1].widgetContainer);
  }

  XtManageChild(RecordContainer);

  CurrentRecord = NULL;
  CurrentRecordIdx = NONE_SELECTED;
  _setCurrentRecordLabel(" ");

  int numPorts = CurrentPipe->numPortsBoundTo;
  PortButtons = (Widget *) realloc(PortButtons, sizeof(Widget)*numPorts);
  for (int portIdx=0; portIdx<numPorts; portIdx++) {
    _makePortButton(portIdx, CurrentPipe->portList[portIdx].portName);
  }

  int numOutputs = CurrentPipe->numOutputsBoundTo;
  OutputButtons = (Widget *) realloc(OutputButtons, sizeof(Widget)*numOutputs);
  for (int outIdx=0; outIdx<numOutputs; outIdx++) {
    _makeOutputButton(outIdx, CurrentPipe->outputList[outIdx].outputName);
  }

  if (numPorts > 0) {
    _getUnboundPortColors();
  }

  if (numOutputs > 0) {
    _getUnboundOutputColors();
  }

  CurrentFieldNum = CurrentPortNum = CurrentOutputNum =  NONE_SELECTED;
  XtSetSensitive(BindingDone, False);
  XtSetSensitive(BindingClear, False);
}



void RecordBindingDialog::callback2(Widget recordButton, XtPointer ,
				    XtPointer )
{
  /* callback called when a record button is pressed */
  int recordIdx, fieldIdx, portIdx, outputIdx;

  XtVaGetValues(recordButton, XmNuserData, &recordIdx, NULL);
  CurrentRecord = &(CurrentPipe->recordList[recordIdx]);
  CurrentRecordIdx = recordIdx;

  for (portIdx=0; portIdx<CurrentPipe->numPortsBoundTo; portIdx++) {
    _showAsUnboundPort(PortButtons[portIdx]);
  }
  for (outputIdx=0; outputIdx<CurrentPipe->numOutputsBoundTo; outputIdx++) {
    _showAsUnboundOutput(OutputButtons[outputIdx]);
  }

  Widget fieldWindow = XtParent( XtParent(FieldContainer) );
  XtDestroyWidget(FieldContainer);
  FieldContainer =  XtCreateWidget("FieldContainer", 
				   xmRowColumnWidgetClass, fieldWindow, 
				   NULL, 0);
  _setCurrentRecordLabel(CurrentRecord->recordName);

  int numFields = CurrentRecord->numFields;
  FieldButtons = (Widget *) realloc(FieldButtons, sizeof(Widget)*numFields);

  for (fieldIdx=0; fieldIdx<numFields; fieldIdx++) {
    _makeFieldButton(fieldIdx, CurrentRecord->fieldList[fieldIdx].fieldName);

    for (portIdx=0; portIdx<CurrentPipe->numPortsBoundTo; portIdx++) {
      if ((CurrentRecord->fieldList[fieldIdx].isBoundToPort[portIdx])) {
	_showAsBound(FieldButtons[fieldIdx], PortButtons[portIdx]);
      }
    }

    for (outputIdx=0; outputIdx<CurrentPipe->numOutputsBoundTo; outputIdx++) {
      if ((CurrentRecord->fieldList[fieldIdx].isBoundToOutput[outputIdx])) {
	_showAsBound(FieldButtons[fieldIdx], OutputButtons[outputIdx]);
      }
    }
  }

  XtManageChild(FieldContainer);

  CurrentFieldNum = CurrentPortNum = CurrentOutputNum =  NONE_SELECTED;
  XtSetSensitive(BindingDone, False);
  XtSetSensitive(BindingClear, False);
}



void RecordBindingDialog::callback3(Widget fieldButton, XtPointer ,
				    XtPointer )
{
  /* callback called when a input record field button is pressed */
  XtVaGetValues(fieldButton,
		XmNuserData, &CurrentFieldNum,
		NULL);

  if (CurrentPortNum != NONE_SELECTED) {
    XtSetSensitive(PipeFrame, False);
    XtSetSensitive(RecordFrame, False);
    XtSetSensitive(CopyDiscardFrame, False);
    XtSetSensitive(BindingDone, False);
    XtSetSensitive(BindingClear, True);
    XtSetSensitive(ContinueButton, False);	
    _bindFieldToPort(CurrentFieldNum, CurrentPortNum);

    if (CurrentRecord != NULL) {
      if (CurrentRecord->status == ActionDiscard) {
	CurrentRecord->status = ActionExtract;
      } else {
	if (CurrentRecord->status == ActionCopyThru) {
	  CurrentRecord->status = ActionCopyThruAndExtract;
	}
      }
      _updateRecordStatus(CurrentRecordIdx);
    }
  }

  if (CurrentOutputNum != NONE_SELECTED) {
    XtSetSensitive(PipeFrame, False);
    XtSetSensitive(RecordFrame, False);
    XtSetSensitive(CopyDiscardFrame, False);
    XtSetSensitive(BindingDone, False);
    XtSetSensitive(BindingClear, True);
    XtSetSensitive(ContinueButton, False);	
    _bindFieldToOutput(CurrentFieldNum, CurrentOutputNum);

    if (CurrentRecord != NULL) {
      if (CurrentRecord->status == ActionDiscard) {
	CurrentRecord->status = ActionExtract;
      } else {
	if (CurrentRecord->status == ActionCopyThru) {
	  CurrentRecord->status = ActionCopyThruAndExtract;
	}
      }
      _updateRecordStatus(CurrentRecordIdx);
    }
  }
}



void RecordBindingDialog::callback4(Widget portButton, XtPointer ,
				    XtPointer )
{
  /* callback called when a port button is pressed */
  XtVaGetValues(portButton,
		XmNuserData, &CurrentPortNum,
		NULL);

  if (CurrentFieldNum != NONE_SELECTED) {
    XtSetSensitive(PipeFrame, False);
    XtSetSensitive(RecordFrame, False);
    XtSetSensitive(CopyDiscardFrame, False);
    XtSetSensitive(BindingDone, False);
    XtSetSensitive(BindingClear, True);
    XtSetSensitive(ContinueButton, False);	
    _bindFieldToPort(CurrentFieldNum, CurrentPortNum);

    if (CurrentRecord != NULL) {
      if (CurrentRecord->status == ActionDiscard) {
	CurrentRecord->status = ActionExtract;
      } else {
	if (CurrentRecord->status == ActionCopyThru) {
	  CurrentRecord->status = ActionCopyThruAndExtract;
	}
      }
      _updateRecordStatus(CurrentRecordIdx);
    }
  }
}



void RecordBindingDialog::callback5(Widget outFieldButton, XtPointer ,
				    XtPointer )
{
  /* callback called when an output field button is pressed */
  XtVaGetValues(outFieldButton,
		XmNuserData, &CurrentOutputNum,
		NULL);

  if (CurrentFieldNum != NONE_SELECTED) {
    XtSetSensitive(PipeFrame, False);
    XtSetSensitive(RecordFrame, False);
    XtSetSensitive(CopyDiscardFrame, False);
    XtSetSensitive(BindingDone, False);
    XtSetSensitive(BindingClear, True);
    XtSetSensitive(ContinueButton, False);	
    _bindFieldToOutput(CurrentFieldNum, CurrentOutputNum);

    if (CurrentRecord != NULL) {
      if (CurrentRecord->status == ActionDiscard) {
	CurrentRecord->status = ActionExtract;
      } else {
	if (CurrentRecord->status == ActionCopyThru) {
	  CurrentRecord->status = ActionCopyThruAndExtract;
	}
      }
      _updateRecordStatus(CurrentRecordIdx);
    }
  }
}



void RecordBindingDialog::callback6(Widget , XtPointer ,
				    XtPointer )
{
  /* callback called when binding done button is pressed */
  XtSetSensitive(RecordFrame, True);
  XtSetSensitive(PipeFrame, True);
  XtSetSensitive(CopyDiscardFrame, True);

  _checkForContinue();
}



void RecordBindingDialog::callback7(Widget , XtPointer ,
				    XtPointer )
{
 /* callback called when binding clear button is pressed */

  _clearAllBindingsForCurrentRecord();

  if (CurrentRecord != NULL) {
    if (CurrentRecord->status == ActionExtract) {
      CurrentRecord->status = ActionDiscard;
    } else {
      if (CurrentRecord->status == ActionCopyThruAndExtract) {
	CurrentRecord->status = ActionCopyThru;
      }
    }
    _updateRecordStatus(CurrentRecordIdx);
  }

  XtSetSensitive(BindingDone, False);
  XtSetSensitive(BindingClear, False);
  XtSetSensitive(RecordFrame, True);
  XtSetSensitive(PipeFrame, True);
  XtSetSensitive(CopyDiscardFrame, True);


  if ((PabloInterface->allOutputPortTraitsSet()) &&
      (PabloInterface->getNumberOutputPorts() > 0)) {
	XtSetSensitive(ModuleResultButton, True);
      } else {
	XtSetSensitive(ModuleResultButton, False);
      }

  _checkForContinue();
}



void RecordBindingDialog::callback8(Widget , XtPointer ,
				    XtPointer )
{
  /* callback called when continue button is pressed */
  DoneRunning = True;
}



void RecordBindingDialog::callback9(Widget , XtPointer ,
				    XtPointer )
{
  /* callback called when reset is selected */

  PabloInterface->restoreOriginalFieldBindings();

  _destroyChildren(RecordContainer);
  _destroyChildren(FieldContainer);
  _destroyChildren(PortContainer);
  _destroyChildren(OutputContainer);

  CurrentPipe = NULL;
  CurrentRecord = NULL;
  CurrentPipeIdx = NONE_SELECTED;
  CurrentRecordIdx = NONE_SELECTED;

  _setCurrentPipeLabel(" ");
  _setCurrentRecordLabel(" ");

  /* reset internal data structures to what they were when we entered */
  int numInPipes = PabloInterface->getNumberInputPipes(); 
  for (int pipeIdx=0; pipeIdx<numInPipes; pipeIdx++) {
    _setupPorts(pipeIdx);
    _setupOutputs(pipeIdx);
    _setupRecords(pipeIdx);
  }
  _setupModuleResultBindings();

  XtSetSensitive(PipeFrame, True);
  XtSetSensitive(RecordFrame, True);
  XtSetSensitive(CopyDiscardFrame, True);
  XtSetSensitive(BindingDone, False);
  XtSetSensitive(BindingClear, False);

  if ((PabloInterface->allOutputPortTraitsSet()) &&
      (PabloInterface->getNumberOutputPorts() > 0)) {
	XtSetSensitive(ModuleResultButton, True);
      } else {
	XtSetSensitive(ModuleResultButton, False);
      }

  _checkForContinue();
}



void RecordBindingDialog::callback10(Widget , XtPointer ,
				     XtPointer )
{
  /* callback called when module result is selected */
  int outputIdx;

  ModuleResultSelected = True;
  CurrentRecord = NULL;
  CurrentRecordIdx = NONE_SELECTED;
  CurrentPipe = NULL;
  CurrentPipeIdx = NONE_SELECTED;

  _destroyChildren(RecordContainer);
  _destroyChildren(FieldContainer);
  _destroyChildren(PortContainer);
  _destroyChildren(OutputContainer);

  int numOutputs = ModuleResult.numOutputFieldsBoundTo;
  OutputButtons = (Widget *) realloc(OutputButtons, sizeof(Widget)*numOutputs);
  for (int outIdx=0; outIdx<numOutputs; outIdx++) {
    _makeOutputButton(outIdx, ModuleResult.outputFieldList[outIdx].
		      outputName);
  }

  if (numOutputs > 0) {_getUnboundOutputColors();}

  int numOutPorts = ModuleResult.numOutputPorts;
  FieldButtons = (Widget *) realloc(FieldButtons, sizeof(Widget)*numOutPorts);
  for (int outPortIdx=0; outPortIdx<numOutPorts; outPortIdx++) {
    _makeFieldButton(outPortIdx, ModuleResult.outputPortList[outPortIdx].
		     outputPortName);

    for (outputIdx=0; outputIdx<ModuleResult.numOutputFieldsBoundTo; 
	 outputIdx++) {
      if ((ModuleResult.outputPortList[outPortIdx].
	   isBoundToOutput[outputIdx])) {
	_showAsBound(FieldButtons[outPortIdx], OutputButtons[outputIdx]);
      }
    }
  }
}



void RecordBindingDialog::callback11(Widget , XtPointer ,
				     XtPointer )
{
  /* callback called when copy thru is selected */
  if (CurrentRecord != NULL) {
    switch (CurrentRecord->status) {

    case ActionCopyThruAndExtract:  
             CurrentRecord->status = ActionExtract;
	     break;

    case ActionExtract: 
	     CurrentRecord->status = ActionCopyThruAndExtract;
	     break;

    case ActionDiscard:       
	     CurrentRecord->status = ActionCopyThru;
	     break;

    case ActionCopyThru:     
	     CurrentRecord->status = ActionDiscard;
	     break;
     }
    _updateRecordStatus(CurrentRecordIdx);
  }
}



void RecordBindingDialog::callback12(Widget , XtPointer ,
				     XtPointer )
{
  /* callback called when discard is selected */
  if (CurrentRecord != NULL) {
    CurrentRecord->status = ActionDiscard;
    _clearAllBindingsForCurrentRecord();
    _updateRecordStatus(CurrentRecordIdx);
  }

  if ((PabloInterface->allOutputPortTraitsSet()) &&
      (PabloInterface->getNumberOutputPorts() > 0)) {
	XtSetSensitive(ModuleResultButton, True);
      } else {
	XtSetSensitive(ModuleResultButton, False);
      }

  XtSetSensitive(BindingClear, False);
  XtSetSensitive(BindingDone, False);

  _checkForContinue();
}



void RecordBindingDialog::helpCallback(Widget , XtPointer ,
				      XtPointer )
{
  /* callback called when help is selected */
  Pablo::HelpSystem()->giveHelpOn( "RecordBindingDialog" );
}


void RecordBindingDialog::run()
{ 
  DoneRunning = False;

  XtManageChild(DialogShell);

  while (!DoneRunning) {
    XEvent event;
    XtAppNextEvent(
                   XtWidgetToApplicationContext(DialogShell), &event );
    XtDispatchEvent( &event );
  }
  
  XtUnmanageChild(DialogShell);
}



/*
 * Initialize the static data
 */
const char *const RecordBindingDialog::MY_CLASS = "RecordBindingDialog";
