/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Tom Birkett (birkett@cs.uiuc.edu)
 * Contributing Author: Bobby A.A. Nazief (nazief@cs.uiuc.edu)
 *
 * Project Manager and Pincipal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * AsciiPipeReader.C - methods that deal with the Ascii Pipes
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Pipes/RCS/AsciiPipeReader.C,v 1.19 1994/04/09 19:12:53 aydt Exp $
 */

#include <ctype.h>
#include <memory.h>
#include <stdlib.h>
#include <string.h>

#ifdef __PARAGON__
extern "C" {
extern void bcopy( char*, char*, int );
}
#endif

#include "AsciiPipeReader.h"

#include "Array.h"
#include "Attributes.h"
#include "FieldDescriptor.h"
#include "RecordDossier.h"
#include "StreamPipe.h"
#include "StructureDescriptor.h"
#include "SystemErrors.h"
#include "Value.h"

#define ALLOC_SIZE 1024 	/* Size of increments to use when     */
				/* allocating space for packetBuffer  */

/*
 * Constructor and Destructor 
 */

AsciiPipeReader::AsciiPipeReader( StreamPipe *initPipe ) 
		:PipeReader( initPipe )
{
	_setClassName( MY_CLASS );
	currentHeader.type = PIPE_EMPTY;

	if ( initPipe->getInputType() != ASCII_SDDF ) {
	    error( "Can't connect to data stream %s", 
		    initPipe->getObjectName() );
	    pipe = NULL;	
	    bufferSize = 0;
	    packetBuffer = NULL;
	} else {
	    bufferSize = ALLOC_SIZE;

	    packetBuffer = (char *) malloc( bufferSize );
            if ( packetBuffer == NULL ) {
                abort( "Can't alloc space for buffer size %d: %s",
                        bufferSize,  errorString() );
            }

	    shadowBuffer = (char *) malloc( bufferSize );
            if ( shadowBuffer == NULL ) {
                abort( "Can't alloc space for shadow buffer size %d: %s",
                        bufferSize,  errorString() );
            }
	}
}

AsciiPipeReader::~AsciiPipeReader()
{
	if ( packetBuffer != NULL ) { 
	    free( packetBuffer );
	}
	if ( shadowBuffer != NULL ) { 
	    free( shadowBuffer );
	}
}

/*
 * Private method to report unparseable input and abort.
 */
void
AsciiPipeReader::_badInput( const char *message, const char *parseString ) const
{
	char *packetEnd = strstr( parseString, ";;" );
	if ( packetEnd != NULL ) {
	    packetEnd++;	// first ;
	    packetEnd++;	// second ;
	    *packetEnd = '\0';
	}

	abort( "%s\n  While attempting to parse: %s", message, parseString );
}

  
/* 
 * Private routines used to parse the data in the packetBuffer. Parsing always
 * starts from where bufp points.
 */

void
AsciiPipeReader::_getArray( Array *array, MachineDataType type, int dimension )
{
	/* 
	 * Parsing information:
	 *  First expect the Array dimesions values, each enclosed in []'s.
	 *  Then expect the cell values, comma-separated and enclosed in {}'s.
	 *  Upon return, bufp pointing at character following the last '}'.
	 */
	Value 	val;
	int 	*dimValu = new int[dimension];
	int	totalCells = 1;
	int	dnum;

	for ( dnum = 0; dnum < dimension; dnum++ ) {
	    if ( *bufp++ != '[' ) {
		_badInput( "Expected array dimension in []'s in input stream",
			    --bufp );
	    }
	    dimValu[dnum] = atoi( bufp );
	    totalCells *= dimValu[dnum];
	    while( *bufp++ != ']' ){
	    };
	}
	array->setDimSizes( dimValu );

	_skipWhiteSpace();
	if ( *bufp++ != '{' ) {
	    _badInput( "Expected array data delimiter '{' in input stream",
			--bufp );
	} 

	/*
	 * If the input has an array with 0 size, we just ignore any
	 * values that may be a part of the data record.
	 */
	if ( totalCells == 0 ) {
	    // warning( "Zero-sized array in input" );
	} else {
	    if ( type == CHARACTER ) {

	        int maxLength = dimValu[ dimension-1 ];
	        int theLength;   
	        char *string;
	        int cellIdx = 0;
	        int snum;

	        for ( snum = 0; snum < totalCells/maxLength; snum++ ) {
		    _skipWhiteSpace();
		    _getString( &string );
		    theLength = strlen( string );

		    int s;
		    for ( s = 0; s < maxLength; s++ ) {
		        if ( s <= theLength ) {
		    	    val = string[s];
		        } else {
			    val = '\0';
		        }
		        array->setTheCellValue( cellIdx++, val );
		    }
	            if ( *bufp != ',' && !isspace( *bufp ) ) {
		        _badInput( "Expected , or <white-space> in array", 
			            bufp );
	            }
		    bufp++;			// skip the , or space
	        }

	    } else {

	        int cellIdx;
 	        for ( cellIdx = 0; cellIdx < totalCells ; cellIdx++ ) {
		    _skipWhiteSpace();
		    _getValue( val, type );
	            array->setTheCellValue( cellIdx, val );

	            if ( *bufp != ',' && !isspace( *bufp ) ) {
		        _badInput( "Expected , or <white-space> in array", 
			            bufp );
	            }
		    bufp++;			// skip the , or blank
	        }
	    }
	}

	delete[] dimValu;

	while ( *bufp++ != '}' ) {
	};

}

void
AsciiPipeReader::_getAttributes( BaseDescriptor& descriptor )
{
	/*
	 * Parsing rules used:
	 *  Leading white space is skipped.
	 *  Expect two "/" characters - if not then return.
	 *  Expect two strings enclosed in "'s and separated by white space.
	 *  Repeat.
	 *  bufp is left pointing to the first non-white character after
	 *   the (optional) Attributes we parsed.
	 */
	
	char *key;
	char *value;
	do {
	    _skipWhiteSpace();
	    if ( (*bufp != '/') || (*(bufp+1) != '/') ) {
		return;
	    }

	    bufp += 2;				// Pass over the //		
	    _skipWhiteSpace();			
	    _getString( &key );
	    _skipWhiteSpace();
	    _getString( &value );

	    descriptor.insertAttributes( key, value );
	} while ( TRUE_ );

}

int 
AsciiPipeReader::_getCommandTag() 
{
	/*
	 * Parsing rules used:
	 *  First character is expected to be a '%'
	 *  Then a string of digits - the tag.
	 *  Then a ':'.
	 *  bufp is left pointing one character after the ':'.
	 */
	if ( *bufp++ != '%' ) {
	    _badInput( "Control tag has wrong sentinal character", --bufp );
	}

	int tag = atoi( bufp );

	while( *bufp++ != ':' ) { 
	}

	return( tag );
}

int
AsciiPipeReader::_getDataTag() 
{
	/*
	 * Parsing rules used:
	 *  First character is expected to be a '"' and then a name followed
	 *   by another '"'.  
	 *  bufp is left unchanged .
	 */
	if ( *bufp != '"' ) {
	    _badInput( "Data tag has wrong sentinal character", --bufp );
	}

	char *savep = bufp;
	char *theName;

	_getString( &theName );

	CString theTag( nameToTag.fetch( theName ) );
	if ( theTag == CString::NOMATCH ) {
	    abort( "No descriptor found for data with the name %s", theName );
	}

	bufp = savep;		// Reset bufp to original position
	return( atoi( theTag.getValue() ) );
}

int
AsciiPipeReader::_getDescriptorTag()
{
	/*
	 * Parsing rules used:
	 *  First character is expected to be a '#'
	 *  Then a string of digits - the tag.
	 *  Then a ':'.
	 *  bufp is left pointing one character after the ':'.
	 * We also read ahead for the descriptor name so that we can 
	 * insert it and the tag in our nameToTag Attributes instance.
	 * The name is the last string before a '{' character.
	 */
	if ( *bufp++ != '#' ) {
	    _badInput( "Descriptor tag has wrong sentinal character", --bufp );
	}

	int tag = atoi( bufp );

	while( *bufp++ != ':' ) {
	};

	char *savep = bufp;		// remember where we are!
	char *nameStart = NULL;		// first quote around name
	char *nameEnd = NULL;		// second quote around name

	while( *bufp != '{' ) {
	    if ( *bufp == '"' ) {
		nameStart = nameEnd;
		nameEnd = bufp;
	    } 
	    bufp++;
	}

	if ( (nameStart == NULL)  || ( nameEnd == NULL ) ) {
	    abort( "Unable to find name in Descriptor Record with tag %d", 
		    tag );
	}

	nameStart++;
	*nameEnd = '\0';
	nameToTag.insert( nameStart, dec( (long)tag ) );

	*nameEnd = '"';
	bufp = savep;

	return( tag );
}

void
AsciiPipeReader::_getField( FieldDescriptor& fieldDescr )
{
	/* 
	 * Parsing rules used:
	 * First look for attributes.
	 * Next expect:  type name[][];
	 *  with each [] representing a dimension, and no []'s indicating 
	 *  a scalar.
	 * bufp left pointing to character following the ';'.
	 */

	_getAttributes( fieldDescr );

	MachineDataType dataType;
	_getType( dataType );
	fieldDescr.setType( dataType );
	_skipWhiteSpace();

	char *theName;
	_getString( &theName );
	fieldDescr.setName( theName );

	int dimension = 0;
	while( *bufp++ != ';' ) {
	    if ( *bufp == ']') {
		dimension++;
	    }
	}
	fieldDescr.setDimension( dimension );	
}

void
AsciiPipeReader::_getPipeAttributes()
{
	/* 
	 * Parsing rules used: 
	 *  First character is expected to be a '/'
	 *  If we see a '"' expect 2 strings separated by white space and
	 *    delimited with '"' characters.
	 *  If we see another '/' (not enclosed in "'s) that's the 
	 *    end of Attributes.
	 */
	if ( *bufp++ != '/' ) {
	    _badInput( "Pipe Attributes has wrong sentinal character", --bufp );
	}

	/*
	 * First, clear out any old entries.  Save buf pointer so that it
	 * can be restored.
	 */
	char *savep = bufp;
	pipeAttributes.clearEntries();

	char *key;
	char *value;

	while ( *bufp != '/' ) {
	    if ( *bufp == '"' ) {
		_getString( &key );
		_skipWhiteSpace();
		_getString( &value );
		pipeAttributes.insert( key, value );
	    } else {
		bufp++;
	    }
	}
	bufp = savep;
}

void
AsciiPipeReader::_getString( char **string )
{
	/*
	 * Parsing rules used:
	 *   First character is expected to be a '"'.
	 *   '\' is an escape character.
	 *   Another '"' (not preceeded by a '\') marks the end of the string.
	 *   bufp is advanced to the character after the final " before return.
	 */

	if ( *bufp++ != '"' ) {
	    _badInput( "String has wrong sentinal character in input stream",
			--bufp );
	}

	*string = bufp;

	while( *bufp != '"' ) {
	    if ( *bufp == '\\' ) {	// "Move" the string
		/* 
		 * Since this is an overlapping move, we can't use memcpy().
		 * The gnu the Paragon compilers have memmove(), but cfront 
		 * does not. So conditionally use memmove or bcopy.
		 */
#ifdef __GNUG__
	        memmove( (*string)+1, *string, bufp-(*string) );
#else
	        bcopy( *string, (*string)+1, bufp-(*string) );
#endif
	        (*string)++;		// New head of string
	        bufp++;			// Skip the quoted char
	    } 
	    bufp++;
	}

	*bufp = '\0';
	bufp++;				// Advance to char following string
}
			
void
AsciiPipeReader::_getType( MachineDataType& type )
{
	/*
	 * Parsing rules used:
	 *  Expect a string giving the data type.
	 *  bufp is left pointing to whitespace character following type
	 *  string.
	 *  >>> Careful here that you don't add a type with name longer
	 *  >>> than 6 characters in typeNameArray[] as defined in DataTraits.h!
	 */

	char 	tName[7];
	int	i = 0;

	char *savep = bufp;		// in case of error

	while ( (i < 6)  && (! isspace(*bufp)) ) {
	    tName[i++] = *bufp++;
	}
	if ( ! isspace( *bufp) ) {
	    _badInput( "Can't find a valid type name in Input Stream", savep );
	} else {
	    if ( strncmp( tName, typeNameArray[CHARACTER], i ) == 0 ) {
		type = CHARACTER;
	    } else if ( strncmp( tName, typeNameArray[INTEGER], i ) == 0 ) {
		type = INTEGER;
	    } else if ( strncmp( tName, typeNameArray[FLOAT], i ) == 0 ) {
		type = FLOAT;
	    } else if ( strncmp( tName, typeNameArray[DOUBLE], i ) == 0 ) {
		type = DOUBLE;
	    } else {
		_badInput( "Invalid type in input stream", tName );
	    }
	}
}

void
AsciiPipeReader::_getValue( Value& val, MachineDataType type )
{
	/* 
	 * Parsing rules used:
	 *   Look for an item of the specified type on the stream.
	 *   bufp is left pointing one character after the end of the item
	 *    upon return.
	 */

	switch( type ) {

	  case CHARACTER:
		char c;
		c = *bufp++;
		val = c;
		break;

	  case INTEGER:
		int i;
	      	if ( sscanf( bufp, "%d", &i ) != 1 ) {
		    _badInput( "Did not find expected integer in Input Stream",
			 	bufp );
		}
	      	val = i;
		while( isdigit(*bufp) || ( *bufp == '-') ) {
		    bufp++;
		}
	      	break;

	  case FLOAT:
		float f;
	      	if ( sscanf( bufp, "%f", &f ) != 1 ) {
		    _badInput( "Did not find expected float in Input Stream",
				bufp );
		}
	      	val = f;
		while( strchr( "0123456789eE-+.", *bufp ) ){
		    bufp++;
		}
	      	break;

	  case DOUBLE:
		double d;
	      	if ( sscanf( bufp, "%lf", &d ) != 1 ) {
		    _badInput( "Did not find expected double in Input Stream",
				bufp );
		}
	      	val = d;
		while( strchr( "0123456789eE-+.", *bufp ) ){
		    bufp++;
		}
	      	break;

	  case UNDEFINED:
	  default:
		break;
	}
}

Boolean_
AsciiPipeReader::_readPacket()
{
	int  semiColonsFound = 0;
	int  increment = 0;

	bufp = packetBuffer;		// Start a beginning of packetBuffer

	while ( semiColonsFound < 2 ) {
	    bufp += increment;		// Advance in packetBuffer

   	    /*
	     * First, make sure we still have space in packetBuffer to 
	     * read another character. If not, then increase buffer size.
	     */
	    if ( (bufp - packetBuffer) == bufferSize ) {

	    	packetBuffer = (char *) realloc( packetBuffer, 
						 bufferSize+ALLOC_SIZE );
                if ( packetBuffer == NULL ) {
                    abort( "Can't realloc space for buffer size %d: %s",
                            bufferSize+ALLOC_SIZE,  errorString() );
                }

	    	shadowBuffer = (char *) realloc( shadowBuffer, 
						 bufferSize+ALLOC_SIZE );
                if ( shadowBuffer == NULL ) {
                    abort( "Can't realloc space for shadow buffer size %d: %s",
                            bufferSize+ALLOC_SIZE,  errorString() );
                }

		bufp = packetBuffer + bufferSize;	// in case we moved
	    	bufferSize += ALLOC_SIZE;
	    }

	    /* 
	     * Next, read a character from the pipe. If none there, then pipe
	     * is empty before end of packet, so exit with FALSE_.  Don't save
	     * leading spaces in packetBuffer. Keep track of adjacent semicolons
	     * as 2 together signals end of packet.
	     */
	    if ( pipe->get( bufp, 1 ) != 1 ) {
		return( FALSE_ );
	    }

	    if ( ! isspace( *bufp ) ) {		
		increment = 1;
	    }

	    if ( *bufp == ';' ) {
		semiColonsFound++;
	    } else {
		semiColonsFound = 0;
	    }

	}

	/* 
	 * Fill in currentHeader values and update instance variables 
	 * for types as appropriate.
	 */
	currentHeader.length = bufp - packetBuffer;
	bufp = packetBuffer;

	switch( *packetBuffer ) {

	  case '/':
		currentHeader.type = PKT_ATTRIBUTE;
		currentHeader.tag = 0;
		_getPipeAttributes();
		break;

	  case '#':
		currentHeader.type = PKT_DESCRIPTOR;
		currentHeader.tag = _getDescriptorTag();
		break;

	  case '"':
		currentHeader.type = PKT_DATA;
		currentHeader.tag = _getDataTag();
		break;

	  case '%':
		currentHeader.type = PKT_COMMAND;
		currentHeader.tag = _getCommandTag();
		break;

	  default:
		currentHeader.type = PKT_UNKNOWN;
		break;

	}

	/* 
	 * Before returning, make a copy of the packet in a shadow buffer
	 * so that we can restore the "original" in the routines getData()
	 * and getDescriptor().  These methods both call _getString() which
	 * messes with the packetBuffer as it processes it.  If we don't
	 * restore the original before returning from getData() or
	 * getDescriptor(), then any repeat calls to those (which should work)
	 * before the next readPacket, will fail.
	 */
	memcpy( shadowBuffer, packetBuffer, currentHeader.length );
	return( TRUE_ );
}

void
AsciiPipeReader::_skipWhiteSpace()
{
	while( isspace( (int)*bufp ) ){
		bufp++;
	}
}

/*
 * Public methods
 */
	
Boolean_ 		/* virtual */
AsciiPipeReader::getData( RecordDossier& recordDossier)  
{
	if ( currentHeader.type != PKT_DATA ) {
 	    return( FAILURE_);
	}

	char *savep = bufp;

	while ( *bufp++ != '{' ) {
	};		

	Value 	val;
	Array	*ap;
	int	fnum;

	for ( fnum = 0; fnum < recordDossier.entryCount(); fnum++ ) {

	    FieldDescriptor *field = recordDossier.getField( fnum );
	    MachineDataType fieldType = field->getType();
	    int             fieldDim  = field->getDimension();

	    _skipWhiteSpace();

	    if ( fieldDim == 0 ) {	// A Scalar
		_getValue( val, fieldType );
	    } else {			// An Array
		ap = new Array( fieldType, fieldDim );
		_getArray( ap, fieldType, fieldDim );
		val = ap;
		delete ap;
	    }
	    if ( ( *bufp != ',' ) && ( *bufp != '}' )
				  && ( !isspace( *bufp ) ) )  {
		_badInput( "Expected , or } or <white-space> after field value",
			    bufp );
	    }
	    bufp++;			// Skip the , or } or whitespace

	    recordDossier.setValue( fnum, val );

	}

	memcpy( packetBuffer, shadowBuffer, currentHeader.length );
	bufp = savep;
	return( SUCCESS_ );
}

Boolean_ 		/* virtual */
AsciiPipeReader::getDescriptor( StructureDescriptor & structDescr )  
{
	if ( currentHeader.type != PKT_DESCRIPTOR )
		return( FAILURE_ );

	char *savep = bufp;

	_getAttributes( structDescr );

	char *theName;
	_getString( &theName );
	structDescr.setName( theName );

	while( *bufp++ != '{' ) {
	};
	_skipWhiteSpace();

	while( *bufp != '}' ) {
	    FieldDescriptor fieldDescr;
	    _getField( fieldDescr );
	    structDescr.insert( fieldDescr );
	    _skipWhiteSpace();
	}

	memcpy( packetBuffer, shadowBuffer, currentHeader.length );
	bufp = savep;
	return( SUCCESS_ );
}

PacketHeader		/* virtual */
AsciiPipeReader::getPacketHeader( )
{
	/* 
	 * We shouldn't be here if pipe is NULL but we don't explicitly
	 * check it for performance reasons.  User should call
	 * successfulConnection() before calling this.
	 */
	do {
		if ( pipe->isEmpty () ) {
			return( noHeader );
		}
	
		if ( _readPacket() == FALSE_ ) {
			return( noHeader );
		}
		
		switch (currentHeader.type) {

		  case PKT_ATTRIBUTE:
		  case PKT_DESCRIPTOR:
		  case PKT_DATA:
		  case PKT_COMMAND:
			return ( currentHeader );
			//break;

		  case PKT_UNKNOWN:
		  case PIPE_EMPTY:
		  default:
			warning( "found bad packet type = %d in header",
				currentHeader.type );
			break;
		}
	} while ( TRUE_ );
}


/*
 * Initialize the static data structures
 */
const char *const AsciiPipeReader::MY_CLASS = "AsciiPipeReader";
PacketHeader AsciiPipeReader::noHeader = { 0, PIPE_EMPTY, 0 };
