/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Contributing Author: Ruth Aydt (aydt@cs.uiuc.edu)
 * Contributing Author:	Bill Whitehouse (whitehou@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * OutputFileStreamPipe.C:
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Pipes/RCS/OutputFileStreamPipe.C,v 1.17 1994/02/25 04:34:10 aydt Exp $
 */

#ifdef __GNUG__
#include <std.h>
extern "C" {
int ftruncate( int fd, off_t length );
}
#else
#include <fcntl.h> 
#include <sysent.h> 
#include <stdlib.h>
#include <unistd.h>
#endif
#include <sys/stat.h>

#include "OutputFileStreamPipe.h"
#include "SystemErrors.h"

OutputFileStreamPipe::OutputFileStreamPipe( const char *name, 
					    int bytesInWritePool ) 
{
	_setClassName( MY_CLASS );
	_setObjectName( name );

	unsigned short currentUmask = umask( 0 );
	umask( currentUmask );
	int mode = 0666 & ( ~currentUmask );
	fileDescriptor = open( name, O_WRONLY | O_CREAT | O_TRUNC, mode );

	if ( fileDescriptor == -1 ) {
	    error( "Unable to initialize file %s: %s",
	            name, errorString() );
	    theWritePool = NULL;
	} else {
	    writePoolOffset = 0;
	    writePoolSize = bytesInWritePool;
	    theWritePool = new char[ writePoolSize ];
	}
}

OutputFileStreamPipe::~OutputFileStreamPipe()
{
	if ( fileDescriptor != -1 ) {
	    _flushPool();
	    close( fileDescriptor );
	}

	if ( theWritePool != NULL ) {
	     delete[] theWritePool;
	}
}

void
OutputFileStreamPipe::_flushPool( )
{
	if ( writePoolOffset != 0 ) {
	    write( fileDescriptor, theWritePool, writePoolOffset );
	    writePoolOffset = 0;
	}
}

int 		/* virtual */
OutputFileStreamPipe::bytesFree() const
{
	return 10000000;
}

int 		/* virtual */
OutputFileStreamPipe::bytesReady() const
{
	return 0;
}

int 		/* virtual */
OutputFileStreamPipe::get( char *, int )
{
	abort( "Cannot get() from an OutputFileStreamPipe" );
	return 0;		// not reached
}

off_t		/* virtual */
OutputFileStreamPipe::getByteOffset()
{
	_flushPool();
	return( lseek( fileDescriptor, 0, SEEK_CUR ) ); 
}

off_t		/* virtual */
OutputFileStreamPipe::getFileSize()
{
	_flushPool();

	off_t currentOffset = lseek( fileDescriptor, 0, SEEK_CUR ); 
	if ( currentOffset == -1 ) {
	    abort( "lseek to get current offset failed: %s",
						errorString() );
	}

	off_t endOffset = lseek( fileDescriptor, 0, SEEK_END );
	if ( endOffset == -1 ) {
	    abort( "lseek to get end offset failed: %s", errorString() );
	}

	if ( lseek( fileDescriptor, currentOffset, SEEK_SET ) == -1 ) {
	    abort( "lseek to reposition at %d failed: %s", currentOffset, 
						errorString() );
	}

	return endOffset;
}

int 		/* virtual */
OutputFileStreamPipe::getSkippingPeek( char *, int )
{
	abort( "Cannot getSkippingPeek() from an OutputFileStreamPipe" );
	return 0;		// not reached
}

Boolean_	/* virtual */
OutputFileStreamPipe::isEmpty() const
{
	return FALSE_;
}

Boolean_	/* virtual */
OutputFileStreamPipe::isFull() const
{
	return FALSE_;
}

int 		/* virtual */
OutputFileStreamPipe::peek( char *, int )
{
	abort( "Cannot peek() into an OutputFileStreamPipe" );
	return 0;		// not reached
}

int 		/* virtual */
OutputFileStreamPipe::put( const char *data, int length )
{
	if ( fileDescriptor == -1 ) {
	    return 0;
	}

	int bytesWritten = 0;
	int poolBytesFree = writePoolSize - writePoolOffset;

	while ( length > 0 ) {
	    if ( length <= poolBytesFree ) {
	 	// We can write everything remaining to pool 
	        memcpy( &theWritePool[writePoolOffset], &data[bytesWritten], 
	         			                            length );
	 	writePoolOffset += length;
	        bytesWritten += length;
	 	length = 0;
	    } else {
	    	// Bytes left is pool is less than we need to hold everything
	        memcpy( &theWritePool[writePoolOffset], &data[bytesWritten], 
		    				             poolBytesFree );
		bytesWritten += poolBytesFree;
		length -= poolBytesFree;
    
		// Our pool is full, so flush it and update offsets, etc.
	        write( fileDescriptor, theWritePool, writePoolSize ); 
	        writePoolOffset = 0;
		poolBytesFree = writePoolSize;
	    }
	}
	return bytesWritten;
}

void		/* virtual */
OutputFileStreamPipe::setByteOffset( off_t newByteOffset )
{
	/* 
	 * First we flush anything that may be in our buffer.
	 * Then we move to the new offset.
	 */
	
	_flushPool();
	if ( ( lseek( fileDescriptor, newByteOffset, SEEK_SET ) ) == -1 ) {
	    abort( "lseek to offset %d failed: %s", newByteOffset,
						errorString() );
	}
}

void		/* virtual */
OutputFileStreamPipe::truncateAfterByte( off_t byteOffset )
{
	/* 
	 * First we flush anything that may be in our buffer.
	 * Then we move to the new offset.
	 */
	
	_flushPool();
	if ( ( ftruncate( fileDescriptor, byteOffset ) ) == -1 ) {
	    abort( "ftruncate to offset %d failed: %s", byteOffset,
						errorString() );
	}
	if ( ( lseek( fileDescriptor, 0, SEEK_END ) ) == -1 ) {
	    abort( "lseek to end of file after truncate failed: %s",
					        errorString() );
	}
}

void 		/* virtual */
OutputFileStreamPipe::printOn( ostream & os ) const
{
	os << form( "OutputFileStreamPipe %s\n", getObjectName() );
}
/*
 * Initialize the static data structures
 */
const char *const OutputFileStreamPipe::MY_CLASS = "OutputFileStreamPipe";
