/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Brian K. Totty (totty@cs.uiuc.edu)
 * Contributing Author: Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * Array.cc -  methods for the Array class.  Note that many of these methods
 * contain duplicate code... it was added to help performance by reducing the
 * number of methods which must be called to accomplish a task that occurs
 * frequently.  Also note that most of the bounds checking is disabled if
 * ASSERT is not defined.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Primitives/RCS/Array.C,v 1.22 1994/02/25 04:34:26 aydt Exp $
 */

#include <memory.h>
#include <stdlib.h>
#include <stdarg.h>

#include "Array.h"
#include "Assert.h"
#include "CString.h"
#include "DataCharacteristics.h"
#include "GetPut.h"
#include "Value.h"

Array::Array( MachineDataType _type, int _dimension, Boolean_ _useDelta )
{
	Assert( _type != UNDEFINED );
	type = _type;
	dimension = _dimension;

	useDelta = _useDelta;
	deltaCount = 0;
	deltaDataSize = 0;
	deltaIndices = NULL;
	iterateCount = 0;

	cellsInDim = new int[dimension];	
	strides = new int[dimension];
	int i;
	for ( i = 0; i < dimension; i++ ) { 
	    cellsInDim[i] = 0;
	    strides[i] = 0;
	}
	cellCount = 0;				// no data cells yet
	data = NULL;
}

Array::Array( const Array& array )
{
	int	bytesToCopy;

	type = array.type;
	dimension = array.dimension;
	iterateCount = 0;

	useDelta = array.useDelta;

	if ( useDelta == TRUE_ ) {
	    deltaCount = array.deltaCount;
	    deltaDataSize = deltaCount;
	    deltaIndices = new int[deltaCount];
	    bytesToCopy = deltaCount * sizeof( int );
	    memcpy( deltaIndices, array.deltaIndices, bytesToCopy );
	} else {
 	    deltaIndices = NULL;
	    deltaCount = 0;
	    deltaDataSize = 0;
	}

	cellsInDim = new int[dimension];
	strides = new int[dimension];
	bytesToCopy = dimension * sizeof( int );
	memcpy( cellsInDim, array.cellsInDim, bytesToCopy );
	memcpy( strides, array.strides, bytesToCopy );

	cellCount = array.cellCount;
	if ( cellCount > 0 ) {
	    bytesToCopy = cellCount * atomSize( type );
	    data = (DATA_P)malloc( bytesToCopy );
	    Assert( data != NULL );		// Need Exception Handler
	    memcpy( data, array.data, bytesToCopy );
	} else {
	    data = NULL;
	}
}

Array::Array( const Array *arrayP )
{
	int bytesToCopy;

	type = arrayP->type;
	dimension = arrayP->dimension;
	iterateCount = 0;

	useDelta = arrayP->useDelta;

	if ( useDelta == TRUE_ ) {
	    deltaCount = arrayP->deltaCount;
	    deltaDataSize = arrayP->deltaDataSize;
	    deltaIndices = new int[deltaCount];
	    bytesToCopy = deltaCount * sizeof( int );
	    memcpy( deltaIndices, arrayP->deltaIndices, bytesToCopy );
	} else {
 	    deltaIndices = NULL;
	    deltaCount = 0;
	    deltaDataSize = 0;
	}

	cellsInDim = new int[dimension];
	strides = new int [dimension];
	bytesToCopy = dimension * sizeof( int );
	memcpy( cellsInDim, arrayP->cellsInDim, bytesToCopy );
 	memcpy( strides, arrayP->strides, bytesToCopy );

	cellCount = arrayP->cellCount;
	if ( cellCount > 0 ) {
	    bytesToCopy = cellCount * atomSize( type );
	    data = (DATA_P)malloc( bytesToCopy );
	    Assert( data != NULL );		// Need Exception Handler
	    memcpy( data, arrayP->data, bytesToCopy );
	} else {
	    data = NULL;
	}
}

Array::~Array()
{
	delete[] cellsInDim;
	delete[] strides;

	if ( deltaDataSize > 0 ) {
	   delete[] deltaIndices;
	}

	if ( cellCount > 0 ) {
	    free( data );
	}
} 

int
Array::bitsToObj( const char *bufPtr )
{
	int i;

	/*
	 * First, read in the number of cells in each dimension and calculate
	 * the total number of cells in the array.
	 */
	int currentOffset = 0;	
	int dimSize;
	Boolean_ newDimSizes = FALSE_;
	int numberOfCells = 1;

	for ( i = 0; i < dimension; i++ ) {
	    currentOffset += functionGet( bufPtr+currentOffset, &dimSize );
	    if ( dimSize != cellsInDim[i] ) {		// a change!
	        newDimSizes = TRUE_;
	        cellsInDim[i] = dimSize;
	    }
	    numberOfCells *= cellsInDim[i];
	}

	/*
	 * If the data size has changed, free the old space and allocate
	 * space for the data as needed
	 */
	if ( numberOfCells != cellCount ) {
	    if ( cellCount > 0 ) {
		free( data );
	    }
	    cellCount = numberOfCells;
	    if ( cellCount > 0 ) {
	        data = (DATA_P)malloc( cellCount * atomSize( type ) );
	        Assert( data != NULL );	    // Need Exception Handler
	    } else {
	        data = NULL;
	    }
	}
	/*
	 * If our dimSizes have changed, recalculate the strides.
	 */
	if ( newDimSizes ) {
	    if ( cellCount > 0 ) {
	        strides[0] = cellCount / cellsInDim[0];
	        for ( i = 1; i < dimension; i++ ) {
            	    strides[i] = strides[i-1] / cellsInDim[i];
	        }
	    } else {
	        for ( i = 0; i < dimension; i++ ) {
	    	    strides[i] = 0;
	        }
	    }
	}

	/*
	 * Finally, populate the data fields from the binary bitstream
	 */
	if ( cellCount > 0 ) {
	    int bytesToCopy = cellCount * atomSize(type);
	    memcpy( data, bufPtr+currentOffset, bytesToCopy );
	    currentOffset += bytesToCopy;
	}

	/*
	 * Reset the count of delta values for this array
	 */
	resetDelta( 0 );

	return currentOffset;
}

int
Array::bytesNeeded() const
{
	return( ( dimension * sizeof(int) ) + ( cellCount * atomSize(type) ) );
}

int
Array::cbitsToObj( const char *bufPtr, const DataCharacteristics *cnvInfo )
{
	int i;

	/*
	 * First, read in the number of cells in each dimension and calculate
	 * the total number of cells in the array.
	 */
	int currentOffset = 0;	
	int dimSize;
	Boolean_ newDimSizes = FALSE_;
	int numberOfCells = 1;

	for ( i = 0; i < dimension; i++ ) {
	    currentOffset += cnvInfo->functionGet( bufPtr+currentOffset, 
						   &dimSize );
	    if ( dimSize != cellsInDim[i] ) {		// a change!
	        newDimSizes = TRUE_;
	        cellsInDim[i] = dimSize;
	    }
	    numberOfCells *= cellsInDim[i];
	}

	/*
	 * If the data size has changed, free the old space and allocate
	 * space for the data as needed
	 */
	if ( numberOfCells != cellCount ) {
	    if ( cellCount > 0 ) {
		free( data );
	    }
	    cellCount = numberOfCells;
	    if ( cellCount > 0 ) {
	        data = (DATA_P)malloc( cellCount * atomSize( type ) );
	        Assert( data != NULL );	    // Need Exception Handler
	    } else {
	        data = NULL;
	    }
	}

	/*
	 * If our dimSizes have changed, recalculate the strides.
	 */
	if ( newDimSizes ) {
	    if ( cellCount > 0 ) {
	        strides[0] = cellCount / cellsInDim[0];
	        for ( i = 1; i < dimension; i++ ) {
            	    strides[i] = strides[i-1] / cellsInDim[i];
	        }
	    } else {
	        for ( i = 0; i < dimension; i++ ) {
	    	    strides[i] = 0;
	        }
	    }
	}

	/*
	 * Finally, populate the data fields from the non-native binary 
	 * bitstream
	 */
        switch ( type ) {
            case CHARACTER:
	        {
	            char *cp = (char *)data;
		    for ( i = 0; i < cellCount; i++ ) {
		        currentOffset += cnvInfo->functionGet( 
					       bufPtr+currentOffset, &cp[i] );
		    }
	        }
                break;
            case INTEGER:
		{
	            int *ip = (int *)data;
		    for ( i = 0; i < cellCount; i++ ) {
		        currentOffset += cnvInfo->functionGet( 
						bufPtr+currentOffset, &ip[i] );
		    }
		}
                break;
            case FLOAT:
		{
	            float *fp = (float *)data;
		    for ( i = 0; i < cellCount; i++ ) {
		        currentOffset += cnvInfo->functionGet( 
						bufPtr+currentOffset, &fp[i] );
		    }
		}
                break;
            case DOUBLE:
		{
	            double *dp = (double *)data;
		    for ( i = 0; i < cellCount; i++ ) {
		        currentOffset += cnvInfo->functionGet( 
						bufPtr+currentOffset, &dp[i] );
		    }
		}
                break;
            case UNDEFINED:
            default:
                break;
        }

	/*
	 * Reset the count of delta values for this array
	 */
	resetDelta( 0 );

	return currentOffset;
}

int
Array::cbytesNeeded( const DataCharacteristics *cnvInfo ) const
{
	return( ( dimension * cnvInfo->intSize() ) + 
		( cellCount * cnvInfo->getAtomSize( type ) ) );
}

Value
Array::getCellValue( int dim0Index ) const
{

#ifdef ASSERT
	if ( cellCount == 0 ) {				// no entries 
	    MSG_OBJ.error( "getCellValue(): Array is not populated" );
	    return( Value::NOVALUE );
	}

	if ( ( dim0Index >= cellsInDim[0] ) || ( dim0Index < 0 ) ) {
	    MSG_OBJ.error( "getCellValue(): Idx 0 = %d is out of range", 
	                    dim0Index );
	    return( Value::NOVALUE );	
	}
#endif ASSERT

	int index = strides[0] * dim0Index;

#ifdef ASSERT
	if ( index >= cellCount ) {
	    MSG_OBJ.error( "getCellValue(): Index = %d is out of range", 
			    index );
	    return( Value::NOVALUE );
	}
#endif ASSERT

	DATA_P cellPtr = data + ( index * atomSize( type ) );

	switch ( type ) {
	    case CHARACTER:
	        return( Value( *( (char*)(cellPtr) ) ) );
	        //break;
	    case INTEGER:
	        return( Value( *( (int*)(cellPtr) ) ) );
	        //break;
	    case FLOAT:
	        return( Value( *( (float*)(cellPtr) ) ) );
	        //break;
	    case DOUBLE:
	        return( Value( *( (double*)(cellPtr) ) ) );
	        //break;
	    case UNDEFINED:
	    default:
	        return( Value::NOVALUE );
	        //break;
	}
} 

Value
Array::getCellValue( int dim0Index, int dim1Index ) const
{

#ifdef ASSERT
	if ( cellCount == 0 ) {				// no entries 
	    MSG_OBJ.error( "getCellValue(): Array is not populated" );
	    return( Value::NOVALUE );
	}

	if ( ( dim0Index >= cellsInDim[0] ) || ( dim0Index < 0 ) ) {
	    MSG_OBJ.error( "getCellValue(): Idx 0 = %d is out of range", 
	                    dim0Index );
	    return( Value::NOVALUE );	
	}

	if ( ( dim1Index >= cellsInDim[1] ) || ( dim1Index < 0 ) ) {
 	    MSG_OBJ.error( "getCellValue(): Idx 1 = %d is out of range", 
	    	            dim1Index );
	    return( Value::NOVALUE );	
	}
#endif ASSERT

	int index = strides[0] * dim0Index + strides[1] * dim1Index;

#ifdef ASSERT
	if ( index >= cellCount ) {
	    MSG_OBJ.error( "getCellValue(): Index = %d is out of range", 
			    index );
	    return( Value::NOVALUE );
	}
#endif ASSERT

	DATA_P cellPtr = data + ( index * atomSize( type ) );

	switch ( type ) {
	    case CHARACTER:
	        return( Value( *( (char*)(cellPtr) ) ) );
	        //break;
	    case INTEGER:
	        return( Value( *( (int*)(cellPtr) ) ) );
	        //break;
	    case FLOAT:
	        return( Value( *( (float*)(cellPtr) ) ) );
	        //break;
	    case DOUBLE:
	        return( Value( *( (double*)(cellPtr) ) ) );
	        //break;
	    case UNDEFINED:
	    default:
	        return( Value::NOVALUE );
	        //break;
	}
} 

void
Array::getDeltaValue( Value& deltaValue, int *dimVector )
{
	int	loc;
	int	i;

#ifdef ASSERT
	if (( useDelta != TRUE_ ) || ( deltaCount <= 0 )) {
	    MSG_OBJ.error( "getDeltaValue exits because there are no deltas" );
	    deltaValue = Value::NOVALUE;

	    for (i = 0; i < dimension; i++) {
	        dimVector[i] = 0;
	    }

	    return;
	}
#endif

	if ( iterateCount == deltaCount ) {
	    iterateCount = 0;
	}

	loc = deltaIndices[iterateCount];

        DATA_P cellPtr = data + ( loc * atomSize( type ) );

        switch ( type ) {
            case CHARACTER:
                deltaValue = *( (char*)(cellPtr) );
                break;
            case INTEGER:
                deltaValue =  *( (int*)(cellPtr) );
                break;
            case FLOAT:
                deltaValue = *( (float*)(cellPtr) );
                break;
            case DOUBLE:
                deltaValue = *( (double*)(cellPtr) );
                break;
	    case UNDEFINED:
            default:
                MSG_OBJ.error( "invalid type in getDeltaValue" );
                break;
        }

	dimVector[0] = loc / strides[0];
	for (i = 1; i < dimension; i++) {
	   dimVector[i] = (loc - (strides[i-1] * dimVector[i-1])) / strides[i];
	}

	iterateCount++;
}

Value
Array::getCellValue( int dim0Index, int dim1Index, int dim2Index ... ) const
{
	va_list args;
	va_start( args, dim2Index );			// varargs setup

#ifdef ASSERT
	if ( cellCount == 0 ) {				// no entries 
	    MSG_OBJ.error( "getCellValue(): Array is not populated" );
	    return( Value::NOVALUE );
	}

	if ( ( dim0Index >= cellsInDim[0] ) || ( dim0Index < 0 ) ) {
	    MSG_OBJ.error( "getCellValue(): Index 0 = %d is out of range", 
			    dim0Index );
	    return( Value::NOVALUE );	
	}

	if ( ( dim1Index >= cellsInDim[1] ) || ( dim1Index < 0 ) ) {
	    MSG_OBJ.error( "getCellValue(): Index 1 = %d is out of range", 
	    	            dim1Index );
	    return( Value::NOVALUE );	
	}

	if ( ( dim2Index >= cellsInDim[2] ) || ( dim2Index < 0 ) ) {
	    MSG_OBJ.error( "getCellValue(): Index 2 = %d is out of range", 
	    	            dim2Index );
	    return( Value::NOVALUE );	
	}
#endif ASSERT

	int index = strides[0] * dim0Index + strides[1] * dim1Index
	    	    + strides[2] * dim2Index;

	int i, dimIndex;
	for ( i = 3; i < dimension; i++ ) {
	    dimIndex = va_arg( args, int );

#ifdef ASSERT
	    if ( ( dimIndex >= cellsInDim[i] ) || ( dimIndex < 0 ) ) {
	        MSG_OBJ.error( "getCellValue(): Index %d = %d is out of range",
	                        i, dimIndex );
	        return( Value::NOVALUE );
	    }
#endif ASSERT
	    index += strides[i] * dimIndex;
	}
	va_end( args );
	
#ifdef ASSERT
	if ( index >= cellCount ) {
	    MSG_OBJ.error( "getCellValue(): Idx = %d is out of range", index );
	    return( Value::NOVALUE );
	}
#endif ASSERT

	DATA_P cellPtr = data + ( index * atomSize( type ) );

	switch ( type ) {
	    case CHARACTER:
	        return( Value( *( (char*)(cellPtr) ) ) );
	        //break;
	    case INTEGER:
	        return( Value( *( (int*)(cellPtr) ) ) );
	        //break;
	    case FLOAT:
	        return( Value( *( (float*)(cellPtr) ) ) );
	        //break;
	    case DOUBLE:
	        return( Value( *( (double*)(cellPtr) ) ) );
	        //break;
	    case UNDEFINED:
	    default:
	        return( Value::NOVALUE );
	        //break;
	}
} 

Boolean_
Array::getCellValue( Value& value, int dim0Index ) const
{

#ifdef ASSERT
	if ( cellCount == 0 ) {				// no entries 
	    MSG_OBJ.error( "getCellValue( Value& ): Array is not populated" );
	    return( FAILURE_ );
	}

	if ( ( dim0Index >= cellsInDim[0] ) || ( dim0Index < 0 ) ) {
	    MSG_OBJ.error( "getCellValue(Value&): Idx 0 = %d is out of range",
	                    dim0Index );
	    return( FAILURE_ );	
	}
#endif ASSERT

	int index = strides[0] * dim0Index;

#ifdef ASSERT
	if ( index >= cellCount ) {
	    MSG_OBJ.error( "getCellValue(Value&): Index = %d is out of range", 
	  	            index );
	    return( FAILURE_ );
	}
#endif ASSERT

	DATA_P cellPtr = data + ( index * atomSize( type ) );

	switch ( type ) {
	    case CHARACTER:
	        value = *( (char*)(cellPtr) );
	        break;
	    case INTEGER:
	        value =  *( (int*)(cellPtr) );
	        break;
	    case FLOAT:
	        value = *( (float*)(cellPtr) );
	        break;
	    case DOUBLE:
	        value = *( (double*)(cellPtr) );
	        break;
	    case UNDEFINED:
	    default:
	  	return( FAILURE_ );
	        //break;
	}

	return( SUCCESS_ );
} 

Boolean_
Array::getCellValue( Value& value, int dim0Index, int dim1Index ) const
{

#ifdef ASSERT
	if ( cellCount == 0 ) {				// no entries 
	    MSG_OBJ.error( "getCellValue(Value&): Array is not populated" );
	    return( FAILURE_ );
	}

	if ( ( dim0Index >= cellsInDim[0] ) || ( dim0Index < 0 ) ) {
	    MSG_OBJ.error( "getCellValue(Value&): Idx 0 = %d is out of range", 
	                    dim0Index );
	    return( FAILURE_ );	
	}

	if ( ( dim1Index >= cellsInDim[1] ) || ( dim1Index < 0 ) ) {
	    MSG_OBJ.error( "getCellValue(Value&): Idx 1 = %d is out of range", 
			    dim1Index );
	    return( FAILURE_ );	
	}
#endif ASSERT

	int index = strides[0] * dim0Index + strides[1] * dim1Index;

#ifdef ASSERT
	if ( index >= cellCount ) {
	    MSG_OBJ.error( "getCellValue(Value&): Index = %d is out of range", 
	         	    index );
	    return( FAILURE_ );
	}
#endif ASSERT

	DATA_P cellPtr = data + ( index * atomSize( type ) );

	switch ( type ) {
	    case INTEGER:
	        value =  *( (int*)(cellPtr) );
	        break;
	    case FLOAT:
	        value = *( (float*)(cellPtr) );
	        break;
	    case DOUBLE:
	        value = *( (double*)(cellPtr) );
	        break;
	    case CHARACTER:
                value = *( (char*)(cellPtr) );
                break;
	    case UNDEFINED:
	    default:
	  	return ( FAILURE_ );
	        //break;
	}
	return( SUCCESS_ );
} 

Boolean_
Array::getCellValue( Value& value, int dim0Index, int dim1Index,
	         		   int dim2Index ... ) const
{
	va_list args;
	va_start( args, dim2Index );			// varargs setup

#ifdef ASSERT
	if ( cellCount == 0 ) {				// no entries 
	    MSG_OBJ.error( "getCellValue( Value& ): Array is not populated" );
	    return( FAILURE_ );
	}

	if ( ( dim0Index >= cellsInDim[0] ) || ( dim0Index < 0 ) ) {
	    MSG_OBJ.error( "getCellValue(Value&): Idx 0 = %d is out of range", 
	                    dim0Index );
	    return( FAILURE_ );	
	}

	if ( ( dim1Index >= cellsInDim[1] ) || ( dim1Index < 0 ) ) {
	    MSG_OBJ.error( "getCellValue(Value&): Idx 1 = %d is out of range", 
	                    dim1Index );
	    return( FAILURE_ );	
	}

	if ( ( dim2Index >= cellsInDim[2] ) || ( dim2Index < 0 ) ) {
	    MSG_OBJ.error( "getCellValue(Value&): Idx 2 = %d is out of range", 
	                    dim2Index );
	    return( FAILURE_ );	
	}
#endif ASSERT

	int index = strides[0] * dim0Index + strides[1] * dim1Index
	    	    + strides[2] * dim2Index;

	int i, dimIndex;
	for ( i = 3; i < dimension; i++ ) {
	    dimIndex = va_arg( args, int );

#ifdef ASSERT
	    if ( ( dimIndex >= cellsInDim[i] ) || ( dimIndex < 0 ) ) {
	        MSG_OBJ.error( "getCellValue(Val&): Idx %d = %d out of range",
	    	                i, dimIndex );
	        return( FAILURE_ );
	    }
#endif ASSERT

	    index += strides[i] * dimIndex;
	}
	va_end( args );
	
#ifdef ASSERT
	if ( index >= cellCount ) {
	    MSG_OBJ.error( "getCellValue( Value& ): Index = %d out of range", 
	                    index );
	    return( FAILURE_ );
	}
#endif ASSERT

	DATA_P cellPtr = data + ( index * atomSize( type ) );

	switch ( type ) {
	    case CHARACTER:
	        value = *( (char*)(cellPtr) );
	        break;
	    case INTEGER:
	        value =  *( (int*)(cellPtr) );
	        break;
	    case FLOAT:
	        value = *( (float*)(cellPtr) );
	        break;
	    case DOUBLE:
	        value = *( (double*)(cellPtr) );
	        break;
	    case UNDEFINED:
	    default:
	        return( FAILURE_ );
	        //break;
	}

	return( SUCCESS_ );
} 

Value
Array::getTheCellValue( int index ) const
{

#ifdef ASSERT
	if ( index >= cellCount ) {			// invalid index
	    MSG_OBJ.error( "getTheCellValue(): Index = %d is out of range", 
	                    index );
	    return( Value::NOVALUE );
	}
#endif ASSERT
	
	DATA_P cellPtr = data + ( index * atomSize( type ) );

	switch ( type ) {
	    case CHARACTER:
	        return( Value( *( (char*)(cellPtr) ) ) );
	        //break;
	    case INTEGER:
	        return( Value( *( (int*)(cellPtr) ) ) );
	        //break;
	    case FLOAT:
	        return( Value( *( (float*)(cellPtr) ) ) );
	        //break;
	    case DOUBLE:
	        return( Value( *( (double*)(cellPtr) ) ) );
	        //break;
	    case UNDEFINED:
	    default:
	        return( Value::NOVALUE );
	        //break;
	}
} 

Boolean_
Array::getTheCellValue( Value& value, int index ) const
{

#ifdef ASSERT
	if ( index >= cellCount ) {			// invalid index
	    MSG_OBJ.error( "getTheCellValue(): Index = %d is out of range", 
	                    index );
	    return( FAILURE_ );
	}
#endif ASSERT
	
	DATA_P cellPtr = data + ( index * atomSize( type ) );

	switch ( type ) {
	    case CHARACTER:
	        value = *( (char*)(cellPtr) );
	        break;
	    case INTEGER:
	        value =  *( (int*)(cellPtr) );
	        break;
	    case FLOAT:
	        value = *( (float*)(cellPtr) );
	        break;
	    case DOUBLE:
	        value = *( (double*)(cellPtr) );
	        break;
	    case UNDEFINED:
	    default:
	        return( FAILURE_ );
	        //break;
	}

	return( SUCCESS_ );
} 

int
Array::objToBits( char *const bufPtr, int bufLen ) const
{
	if ( bytesNeeded() > bufLen ) {
	    return( -1 );
	} 

	int currentOffset;

	int bytesToCopy = dimension * sizeof(int);
	memcpy( bufPtr, cellsInDim, bytesToCopy ) ;
	currentOffset = bytesToCopy;

	if ( cellCount > 0 ) {
	    bytesToCopy = cellCount * atomSize(type) ;
	    memcpy( bufPtr+currentOffset, data, bytesToCopy );
	    currentOffset += bytesToCopy;
	}

	return currentOffset;
}

int
Array::objToCbits( char *const bufPtr, int bufLen,
		   const DataCharacteristics *cnvInfo ) const
{
	if ( cbytesNeeded( cnvInfo ) > bufLen ) {
	    return( -1 );
	} 

	int i;
	int bytesUsed = 0;

	/* 
	 * First we write the dimensions as integers
	 */
	for ( i = 0; i < dimension; i++ ) {
	    bytesUsed += cnvInfo->functionPut( bufPtr+bytesUsed, 
					       &cellsInDim[i] );
	}

	/*
	 * Next we write the data cells, using the appropriate
	 * conversion routines depending on the type of the array.
	 */
        switch ( type ) {
            case CHARACTER:
		{
	            char *cp = (char *)data;
		    for ( i = 0; i < cellCount; i++ ) {
		        bytesUsed += cnvInfo->functionPut( 
						bufPtr+bytesUsed, &cp[i] );
		    }
		}
                break;
            case INTEGER:
		{
	            int *ip = (int *)data;
		    for ( i = 0; i < cellCount; i++ ) {
		        bytesUsed += cnvInfo->functionPut( 
						bufPtr+bytesUsed, &ip[i] );
		    }
		}
                break;
            case FLOAT:
		{
	            float *fp = (float *)data;
		    for ( i = 0; i < cellCount; i++ ) {
		        bytesUsed += cnvInfo->functionPut( 
						bufPtr+bytesUsed, &fp[i] );
		    }
		}
                break;
            case DOUBLE:
		{
	            double *dp = (double *)data;
		    for ( i = 0; i < cellCount; i++ ) {
		        bytesUsed += cnvInfo->functionPut( 
						bufPtr+bytesUsed, &dp[i] );
		    }
		}
                break;
            case UNDEFINED:
            default:
                break;
        }

	Assert( bytesUsed <= bufLen );

	return bytesUsed;
}

void
Array::resetDelta( int countDelta )
{
	if ( countDelta == deltaCount ) {
	    return;
	}

	if ( deltaCount > 0 ) {
	    delete[] deltaIndices;
	} 

	deltaIndices = new int[countDelta];
	deltaDataSize = countDelta;
	deltaCount = countDelta;;
}

Boolean_
Array::setCellString( const CString& cstring, int dim0Index ... ) 
{
#ifdef ASSERT
	if ( type != CHARACTER ) {
	    MSG_OBJ.error( "setCellString unsupported for non-CHARACTER type" );
   	    return( FAILURE_ );
	}

	if ( cellCount == 0 ) {
	    MSG_OBJ.error( "setCellString called before dimension size(s) set");
   	    return( FAILURE_ );
	}
#endif ASSERT

	va_list args;
	va_start( args, dim0Index );	// varargs setup

#ifdef ASSERT
	if ( ( dim0Index < 0 ) || ( cellsInDim[0] <= dim0Index ) ) {
	    MSG_OBJ.error( "setCellString(): Index 0 = %d is out of range", 
	                    dim0Index );
	    return( FAILURE_ );
	}
#endif ASSERT

	int index = strides[0] * dim0Index;

	int dimIndex, i;
	for ( i = 1; i < dimension-1; i++ ) {
  	    dimIndex = va_arg( args, int );

#ifdef ASSERT
	    if ( ( dimIndex < 0 ) || ( cellsInDim[i] <= dimIndex ) ) {
	        MSG_OBJ.error( "setCellString(): Index %d = %d is out of range",
	                        i, dimIndex );
	        return( FAILURE_ );
	    }
#endif ASSERT

	    index += strides[i] * dimIndex;
	}

	va_end( args );

	if ( useDelta == TRUE_ ) {
	    setDeltaIndex( index );
	}

	char *cellPtr = data + ( index * atomSize( CHARACTER ) );
	int  maxLength = cellsInDim[ dimension - 1 ];
	int  theLength = cstring.length();

	if ( theLength > maxLength ) {
	    MSG_OBJ.warning( 
			"setCellString(): \"%s\" being truncated to %d chars.",
	                cstring.getValue(), maxLength );
	}

	for ( i = 0; i < maxLength; i++ ) {
	    if ( i < theLength ) {
	        cellPtr[i] = cstring[i];
	    } else {
	        cellPtr[i] = '\0';
	    }
	}
	    
	return( SUCCESS_ );
}

Boolean_
Array::setCellValue( const Value& value, int dim0Index ... ) 
{
	va_list args;
	va_start( args, dim0Index );	// varargs setup

#ifdef ASSERT
	if ( cellCount == 0 ) {
	    MSG_OBJ.error( "setCellValue: called before dimension size(s) set");
   	    return( FAILURE_ );
	}

	if ( ( dim0Index < 0 ) || ( cellsInDim[0] <= dim0Index ) ) {
	    MSG_OBJ.error( "setCellValue(): Idx 0 = %d is out of range", 
	                    dim0Index );
	    return( FAILURE_ );
	}
#endif ASSERT

	int index = strides[0] * dim0Index;

	int dimIndex, i;
	for ( i = 1; i < dimension; i++ ) {
  	    dimIndex = va_arg( args, int );

#ifdef ASSERT
	    if ( ( dimIndex < 0 ) || ( cellsInDim[i] <= dimIndex ) ) {
	        MSG_OBJ.error( "setCellValue(): Index %d = %d is out of range",
	                        i, dimIndex );
	    return( FAILURE_ );
	    }
#endif ASSERT

	    index += strides[i] * dimIndex;
	}

	va_end( args );

#ifdef ASSERT
	if ( index >= cellCount ) {
	    MSG_OBJ.error( "setCellValue() - Invalid index = %d", index );
	    return( FAILURE_ );
	}
#endif ASSERT

	if ( useDelta == TRUE_ ) {
	   setDeltaIndex( index );
	}

	DATA_P cellPtr = data + ( index * atomSize( type ) );

	switch ( type ) { 
	    case CHARACTER:
	        (* (char*)(cellPtr) = (char)value );
	        break;
	    case INTEGER:
	        (* (int*)(cellPtr) = (int)value );
	        break;
	    case FLOAT:
	        (* (float*)(cellPtr) = (float)value );
	        break;
	    case DOUBLE:
	        (* (double*)(cellPtr) = (double)value );
	        break;
	    case UNDEFINED:
	    default:
	        MSG_OBJ.error( "setTheCellValue(): Array has unknown type" );
	        return( FAILURE_ );
	        // break;
	}

	return( SUCCESS_ );
}

Boolean_
Array::setCellValues( const Array * array ) 
{
	int	i;

#ifdef ASSERT
	if ( cellCount == 0 ) {
	    /*
	     * If the new array has active cells, then report an error
	     * and return FAILURE_.  If the incoming array has no cells,
	     * then return SUCCESS_ and don't complain.
	     */
	    if ( array->getCellCount() != 0 ) {
	        MSG_OBJ.error( "setCellValues() called before dim sizes set");
	        return( FAILURE_ );
	    } else {
	        return( SUCCESS_ );
	    }
	} else if ( dimension != array->dimension ) {
	    MSG_OBJ.error( "setCellValues(): dimension count mismatch" );
	    return( FAILURE_ );
	} else {
	    for ( i = 0; i < dimension; i++ ) {
	        if ( cellsInDim[i] != array->cellsInDim[i] ) {
	    	    MSG_OBJ.error( "setCellValues(): dim size mismatch" );
	    	    return( FAILURE_ );
	        }
	    }
	}
#endif ASSERT

	iterateCount = 0;
	if ( array->useDelta == TRUE_ ) {
	    useDelta = TRUE_;
            resetDelta( array->deltaCount );
            memcpy( deltaIndices, array->deltaIndices, 
	    	                  sizeof(int) * array->deltaCount );
	} else {
	    useDelta = FALSE_;
	    resetDelta( 0 );
	}

	if ( type == array->type ) {
	    memcpy( data, array->data, cellCount * atomSize( type ) );
	    return( SUCCESS_ );
	}

	/* 
	 * Here the type of the new array and this one do not match. We
	 * cast the new array values to match the type of this array.
	 */
	DATA_P cellPtr = data;
	DATA_P srcPtr = array->data;
	int cellSize = atomSize( type );
	int srcSize = atomSize( array->getType() );

	switch ( type ) {
	    case CHARACTER:
	        for (i = 0; i < cellCount; i++) {
	            (* (char*)(cellPtr) = *( (char*)(srcPtr) ) );
 		    cellPtr += cellSize;
           	    srcPtr += srcSize;
	        }
	        break;

	    case INTEGER:
	        for (i = 0; i < cellCount; i++) {
	            (* (int*)(cellPtr) = *( (int*)(srcPtr) ) );
 	            cellPtr += cellSize;
           	    srcPtr += srcSize;
	        }
	        break;

	    case FLOAT:
	        for (i = 0; i < cellCount; i++) {
	            (* (float*)(cellPtr) = *( (float*)(srcPtr) ) );
 	            cellPtr += cellSize;
           	    srcPtr += srcSize;
	        }
	        break;

	    case DOUBLE:
	        for (i = 0; i < cellCount; i++) {
	            (* (double*)(cellPtr) = *( (double*)(srcPtr) ) );
                    cellPtr += cellSize;
                    srcPtr += srcSize;
	        }
	        break;

	    case UNDEFINED:
	    default:
	        MSG_OBJ.error( "setCellValues(): Array has unknown type" );
	        return( FAILURE_ );
	        //break;
	}

	return( SUCCESS_ );
} 

void
Array::setDeltaIndex( int indexValue )
{
	int 	i;
	int	*tmpIndices;
	int	tmpSize;

#define Min(a, b)    ((a) < (b) ? (a) : (b)

	for (i = 0; i < deltaCount; i++) {
	    if ( deltaIndices[i] == indexValue ) {
	        return;
	    }
	}

	if ( deltaCount >= deltaDataSize ) {
	    if ( deltaDataSize == 0 ) {
	        tmpSize = 1;
	    } else  if ( 2 * deltaDataSize > cellCount ) {
	        tmpSize = cellCount;
	    } else {
	        tmpSize = 2 * deltaDataSize;
	    }

	    tmpIndices = new int[tmpSize]; 

	    for (i = 0; i < deltaCount; i++) {
	        tmpIndices[i] = deltaIndices[i];
	    }
	   
	    tmpIndices[deltaCount] = indexValue;
	    deltaCount++;

	    delete[] deltaIndices;

	    deltaIndices = tmpIndices;
	    deltaDataSize = tmpSize;
	} else {
	    deltaIndices[deltaCount] = indexValue;
	    deltaCount++;
	}
}

void 
Array::setDimSizes( const int *intArray )
{
	/* 
	 * We try to be a bit clever and only free/realloc space if really
	 * necessary.  It is "really necessary" if the number of cells in
	 * the array has changed. Else, we can just reuse the old space.
	 * In either case, we guarantee that the data space is initialized
	 * to zeros.
	 */

	int newCellCount = 1;			// set new cell counts
	for ( int i = 0; i < dimension; i++ ) {
	    cellsInDim[i] = intArray[i];
	    newCellCount *= cellsInDim[i];
	}

	if ( newCellCount != cellCount ) {	// must resize, free and alloc
	    if ( cellCount > 0 ) {
	        free( data );
	    }
	    if ( newCellCount > 0 ) {
	        data = (DATA_P)calloc( newCellCount, atomSize( type ) );
	        Assert( data != NULL );    	// need exception handler
	    } else {
	        data = NULL;
	    }
	    cellCount = newCellCount;
	} else {				// reuse original data space
	    if ( data != NULL ) {
	        memset( data, 0, cellCount * atomSize( type ) );
	    }
	}

	    				// recalculate our strides
	if ( cellCount > 0 ) { 
	    strides[0] = cellCount / cellsInDim[0];
	    for ( i = 1; i < dimension; i++ ) {
                strides[i] = strides[i-1] / cellsInDim[i];
	    }
	} else {
	    for ( i = 0; i < dimension; i++ ) {
                strides[i] = 0;
	    }
	}

}

Boolean_
Array::setTheCellValue( int index, const Value& value ) 
{
#ifdef ASSERT
	if ( cellCount == 0 ) {
	    MSG_OBJ.error( "setTheCellValue() called before dim sizes set");
	    return( FAILURE_ );
	}

	if ( index >= cellCount) {
	    MSG_OBJ.error( "setTheCellValue() - Invalid index = %d", index );
	    return( FAILURE_ );
	}
#endif ASSERT

	DATA_P cellPtr = data + ( index * atomSize( type ) );

	if ( useDelta == TRUE_ ) {
	   setDeltaIndex( index );
	}

	switch ( type ) {
	    case CHARACTER:
	        (* (char*)(cellPtr) = (char)value );
	        break;
	    case INTEGER:
	        (* (int*)(cellPtr) = (int)value );
	        break;
	    case FLOAT:
	        (* (float*)(cellPtr) = (float)value );
	        break;
	    case DOUBLE:
	        (* (double*)(cellPtr) = (double)value );
	        break;
	    case UNDEFINED:
	    default:
	        MSG_OBJ.error( "setTheCellValue(): Array has unknown type" );
	        return( FAILURE_ );
	        // break;
	}
	return( SUCCESS_ );
} 

Array&
Array::operator=( const Array &array )
{
	/* 
	 * Note that this is the only way we can change the type and/or 
	 * dimension of an array after it has been constructed!
	 */

	int 	  i;
	Boolean_  reSize = FALSE_;

	if ( type != array.type ) {
	    reSize = TRUE_;
	} else if ( dimension != array.dimension ) {
	    reSize = TRUE_;
	} else {
	    for ( i = 0; i < dimension; i++ ) {
	        if ( cellsInDim[i] != array.cellsInDim[i] ) {
	    	reSize = TRUE_;
	    	break;
	        }
	    }
	}

	if ( reSize ) {
	    delete[] cellsInDim;	      // Free original space
	    delete[] strides;

	    if ( cellCount > 0 ) {
	        free( data );
	    }
	    data = NULL;

	    type = array.type;		     // Reset to match new 
	    dimension = array.dimension;

	    cellCount = array.cellCount;
	    cellsInDim = new int[dimension];
	    strides = new int[dimension];

	    memcpy( cellsInDim, array.cellsInDim, dimension * sizeof(int) );
	    memcpy( strides, array.strides, dimension * sizeof(int) );

	    if ( cellCount > 0 ) {
	        data = (DATA_P)malloc( cellCount * atomSize( type ) );
	        Assert( data != NULL );	    // Need Exception Handler
	    }
	}

	if ( cellCount > 0 ) {
	    memcpy( data, array.data, cellCount * atomSize( type ) );
	}

	resetDelta( array.deltaCount );
	useDelta = array.useDelta;

	if ( array.deltaCount > 0 ) {
	    memcpy( deltaIndices, array.deltaIndices, deltaCount*sizeof(int) );
	}

	return *this;
}

void
Array::printOn( ostream& os ) const
{
	int      i;
	char   *cp;
	int    *ip;
	float  *fp;
	double *dp;

	os << "<Array begin>\n";

	switch ( type ) {
	    case CHARACTER:
	        os << "type is CHARACTER\n";
	        cp = (char *)data;
	        break;
	    case INTEGER:
	        os << "type is INTEGER\n";
	        ip = (int *)data;
	        break;
	    case FLOAT:
	        os << "type is FLOAT\n";
	        fp = (float *)data;
	        break;
	    case DOUBLE:
	        os << "type is DOUBLE\n";
	        dp = (double *)data;
	        break;
	    case UNDEFINED:
	    default:
	        os << "type is UNKNOWN\n";
	        break;
	}

	if ( useDelta == TRUE_ ) {
	    os << "Delta array with " << deltaCount << " values\n";
	    os << "<Delta indices begin>";

	    for ( i = 0; i < deltaCount; i++ ) {
	        os << "\nIndex [" << i << "] = " << deltaIndices[i];
	    }
	    os << "\n<Delta indices end>\n";
	}

	os << "Dimensions: " << dimension << NL;
	os << "Cells in each dimension: " ;

	int *index = new int[ dimension ];	// To keep indicies in each dim
	for ( i = 0; i < dimension; i++ ) {	// zero index array & print 
	    index[i] = 0;
	    os << cellsInDim[i] << " " ;
	}

	int cell;
	for ( cell = 0; cell < cellCount; cell++ ) {
	    cout << "\nCell";
	    for ( i = 0; i < dimension; i++ ) {
	    	cout << "[" << index[i] << "]";
	    }
	    cout << "= ";
	    switch ( type ) {
	        case CHARACTER:
	       	    cout << *cp++ ;
	            break;
	        case INTEGER:
	       	    cout << *ip++ ;
	            break;
	        case FLOAT:
	       	    cout << *fp++ ;
	    	    break;
	        case DOUBLE:
	       	    cout << *dp++ ;
	    	    break;
	        case UNDEFINED:
	        default:
	    	    cout << " Error - unknown type" ;
	    	    break;
	    }
	    
	    for ( i = dimension - 1; i >= 0; i-- ) {
	    	if ( ++index[i] < cellsInDim[i] ) {
	    	    break;
	    	} else {
	    	    index[i] = 0;
	    	}
	    }
	}
	delete index;

	os << "\n<Array end>\n";
}
