/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * GeneralDictionary.cc: Implements GeneralDictionary.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/UsefulClasses/RCS/GeneralDictionary.C,v 1.11 1994/02/25 04:34:43 aydt Exp $
 */
#include <math.h>
#include <stdlib.h>

#include "GeneralDictionary.h"
#include "Association.h"
#include "Histogram.h"

GeneralDictionary::GeneralDictionary(int size)
{
	int i;
	
	insertHist = new Histogram(size);
	lookupHist = new Histogram(size);
	
	hashtable = new Association * [size];
	hashtableSize = size;

	for (i = 0; i < hashtableSize; i++)
		hashtable[i] = NULL;
}

GeneralDictionary::~GeneralDictionary()
{
	delete[] hashtable;
	delete insertHist;
	delete lookupHist;
}

void GeneralDictionary::insertKeyValue(const CString &key, void *value)
{
	int hval, slot;
	Association *assoc;

	assoc = privateLookup(key);

	if (assoc)
	{
		assoc->setValue(value);
		return;
	}

	hval = hash(key);

	slot = findNextEmptySlot(hval);
	if (slot == -1)	
	{
		rehash(hashtableSize * 2 + 1);
		insertKeyValue(key, value);
	}
	hashtable[slot] = new Association(key,value);
}

Association *
GeneralDictionary::privateLookup( const CString& key ) const
{
	Association *assoc;

	int hval = hash(key);
	int nProbes = 1;

	int slot;
	for ( slot = hval; (assoc = hashtable[slot]) != NULL;
	      slot = (slot + 1) % hashtableSize ) {
	    nProbes++;
	    if ( assoc->getKey() == key ) {
		break;
	    }
	}

	return assoc;
}

void *
GeneralDictionary::lookupKey( const CString& key ) const
{
	Association* assoc = privateLookup( key );

	if ( assoc == NULL ) {
	    return NULL;
	} else {
	    return assoc->getValue();
	}
}

int GeneralDictionary::findNextEmptySlot(int s)
{
	int i;
	int nProbes;
	int slot;

	nProbes = 1;

	if (hashtable[s] == NULL)
	{
		slot = s;
	}
	else
	{
		for (i = s + 1; i != s; i = (i + 1) % hashtableSize)
		{
			nProbes++;
			if (hashtable[i] == NULL)
				break;
		}
		
		if (i == s)
			slot = -1;
		else
			slot = i;
	}
	insertHist->add(nProbes);
	return slot;
}

void 
GeneralDictionary::rehash( int /* size */ )
{
	abort("GeneralDictionary overflowed and rehash not implemented");
}

int 
GeneralDictionary::hash( const CString& cstr ) const
{
	const char *str = cstr.getValue();
	const char *s;
	int hval = 0;

	for ( s = str; *s; s++ ) {
	    hval = (hval << 1) ^ (int) *s;
	}
	return ( abs( hval % hashtableSize) );
}


void 
GeneralDictionary::printOn( ostream& os ) const
{
	int i;
	Association *assoc;
	
	os << "GeneralDictionary\n";
	for ( i = 0; i < hashtableSize; i++ ) {
	    if ( (assoc = hashtable[i]) != NULL ) {
		os << form( "%d: %s %x\n", i,
				   assoc->getKey().getValue(),
				   assoc->getValue() );
	    }
	}
	os << "Lookup hist:\n" << *lookupHist;
	os << "Insert hist:\n" << *insertHist;
}

