/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author:Tara Madhyastha (tara@cs.uiuc.edu) 
 * Contributing Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * MathOperation.C: Implements simple mathematical opeations
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/UsefulClasses/RCS/MathOperation.C,v 1.12 1994/03/15 16:29:49 aydt Exp $
 *	
 */

#include "MathOperation.h"

MathOperation::MathOperation()
{
	_setClassName( MY_CLASS );

	zero = 0;
	one = 1;
	two = 2;
}

MathOperation::~MathOperation()
{
}

void
MathOperation::_binaryOp( const Value& /* lValue */, const Value& /* rValue */,
			  Value& resultValue )
{
	subclassResponsibility( "MathOperation::_binaryOp" );
	resultValue = Value::NOVALUE;
}

void
MathOperation::_unaryOp( const Value& /* dataValue */, Value& resultValue )
{
	subclassResponsibility( "MathOperation::_unaryOp" );
	resultValue = Value::NOVALUE;
}

void 				/* virtual */
MathOperation::getUnaryResult( const Value& operand, Value& result ) 
{
	//  getUnaryResult always generates a double-precision result

	if ( operand.getTraits().isScalar() ) {
	    /*
	     * If the operand is a scalar, it's a breeze 
	     */
	    operand.getValueAs( DOUBLE, _tmpValue );
	    _unaryOp( _tmpValue, result );

	} else {
	    /* 
	     * If the operand is an array, we march through the data
	     * elements and apply the function to each element. 
	     * The Array class never changes the type of the array when
	     * you setTheCellValue, so we don't have to cast to a DOUBLE
	     * here like we did with our scalar above.
	     */
	    Array *dataArrayP = (Array *) operand;
	    const int *dataDimSizes = dataArrayP->getDimSizes();
	    int dimension = operand.getTraits().getDimension();

	    Array *resultArrayP = new Array( DOUBLE, dimension );
	    resultArrayP->setDimSizes( dataDimSizes );
	    int cellCount = resultArrayP->getCellCount();

	    int i;
	    Value resultValue;

	    for ( i = 0; i < cellCount; i++ ) {
	       dataArrayP->getTheCellValue( _tmpValue, i );
	       _unaryOp( _tmpValue, resultValue );
	       resultArrayP->setTheCellValue( i, resultValue );
	    }

	    result = resultArrayP;
	    delete resultArrayP;
	  }
}

/* virtual */
void MathOperation::getBinaryResult( const Value& leftOperand,
			             const Value& rightOperand, 
			             Value& result )
{
	// the result type depends on the type of the operands

  	DataTraits leftTraits( leftOperand.getTraits() );
  	DataTraits rightTraits( rightOperand.getTraits() );

	/*
	 * If both left and right are scalars, it's a breeze.
	 */
	if ( leftTraits.isScalar() && rightTraits.isScalar() ) {
	    _binaryOp( leftOperand, rightOperand, result );

	/* 
	 * If left is an array and right is a scalar, we'll create
	 * an array having appropriate type.
	 */
	} else if ( leftTraits.isArray() && rightTraits.isScalar() ) {
	    Array *leftArrayP = (Array *) leftOperand;

	    Array *resultArrayP =
		new Array( rightOperand.getResultType( leftTraits.getType() ),
	                   leftArrayP->getDimension() );

	    resultArrayP->setDimSizes( leftArrayP->getDimSizes() );
	    int cellCount = resultArrayP->getCellCount();

	    int i;
	    Value resultValue;	

	    for ( i = 0; i < cellCount; i++ ) {
	       leftArrayP->getTheCellValue( _tmpValue, i );
		_binaryOp( _tmpValue, rightOperand, resultValue );
		resultArrayP->setTheCellValue( i, resultValue );
	    }

	    result = resultArrayP;
	    delete resultArrayP;

	/*
	 * If left is a scalar and right is an array, we'll create
	 * an array having appropriate type.
	 */
	} else if ( leftTraits.isScalar() && rightTraits.isArray() ) {
	    Array *rightArrayP = (Array *) rightOperand;

	    Array *resultArrayP =
	       new Array( leftOperand.getResultType( rightTraits.getType() ),
			  rightArrayP->getDimension() );

	    resultArrayP->setDimSizes( rightArrayP->getDimSizes() );
	    int cellCount = resultArrayP->getCellCount();

	    int i;
	    Value resultValue;

	    for ( i = 0; i < cellCount; i++ ) {
		rightArrayP->getTheCellValue( _tmpValue, i );
		_binaryOp( leftOperand, _tmpValue, resultValue );
		resultArrayP->setTheCellValue( i, resultValue );
	    }

	    result = resultArrayP;
	    delete resultArrayP;

	/* 
	 * If both left and right are arrays, we do some extra checking
	 * to make sure they "match."  The caller should have already
	 * verified that the dimensions are the same so we just double-check
	 * that with an Assert statement.  We make sure that the size of
	 * each dimension for each array is the same.  If that is true,
	 * then set the result type, dimSizes,  and output an array.
	 * If the sizes do not match, then print a warning and output a Value 
	 * of type UNDEFINED.
	 */
	} else {	// Both rightOperand and leftOperand are Arrays
	    Assert( leftTraits.getDimension() == rightTraits.getDimension() );

	    Array *leftArrayP = (Array *) leftOperand;
	    const int *leftDimSizes = leftArrayP->getDimSizes();

	    Array *rightArrayP = (Array *) rightOperand;
	    const int *rightDimSizes = rightArrayP->getDimSizes();

	    int dimension = leftTraits.getDimension();
	    int d = 0;

	    while ( d < dimension ) {
	        if ( leftDimSizes[d] == rightDimSizes[d] ) {
	            d++;
	        } else {
	            break;
	        }
	    }

	    if ( d == dimension ) {		// dimSizes all matched
	        Array *resultArrayP = new Array(
			   leftOperand.getResultType( rightTraits.getType() ),
			   dimension );

	        resultArrayP->setDimSizes( leftDimSizes );
	      	int cellCount = resultArrayP->getCellCount();

		int i;
		Value rCellValue;
		Value resultValue;

	    	for ( i = 0; i < cellCount; i++ ) {
		    
		    leftArrayP->getTheCellValue( _tmpValue, i );
		    rightArrayP->getTheCellValue( rCellValue, i );
		    _binaryOp( _tmpValue, rCellValue, resultValue );
		    resultArrayP->setTheCellValue( i, resultValue );
	        }

	        result = resultArrayP;
		delete resultArrayP;
	    } else {			// dimSizes mismatch
	   	warning( 
		     "Unable to process arrays with unequal dimension sizes." );
		result = Value::NOVALUE;
	    }
	  }
}

void 			/* virtual */
MathOperation::getReductionResult( const Value& operand, Value& result ) 
{
	Assert( operand.getTraits().isArray() );  // Only arrays okay

	_unaryOp( operand, _tmpValue );
	_tmpValue.getValueAs( DOUBLE, result );	  // force result to be a DOUBLE
}

void 			/* virtual */
MathOperation::getHistoryResult( const Value& operand, const Value& oldValue, 
				 Value& result ) 
{
	Assert( oldValue.getTraits().isScalar() );

	if ( operand.getTraits().isScalar() ) {
	    /*
	     * If the operand is a scalar, it's a breeze.
	     */
	    _binaryOp( oldValue, operand, result );

	} else {
	    /* 
	     * If the operand is an array, we reduce the array using 
	     * each element and the "history" value as operands.  When
	     * we finish, result contains the correct value.
	     */
	    Array *operandP = (Array *) operand;
	    int cellCount = operandP->getCellCount();

	    int i;
	    Value cellValue;
	    _tmpValue = oldValue;

	    for ( i = 0; i < cellCount; i++ ) {
		operandP->getTheCellValue( cellValue, i );
		_binaryOp( _tmpValue, cellValue, result );
	    }
	}
}

void 
MathOperation::setBase( double newBase )
{
	base = newBase;
}

void 
MathOperation::printOn( ostream& os ) const
{
	os << "MathOperation\n";
}

/*
 *      Initialize the static data.
 */
const char *const MathOperation::MY_CLASS = "MathOperation";


