/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Robert Olson (olson@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * FUWrapper.cc: Implements a Functional Unit Wrapper.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Wrapper/RCS/FUWrapper.C,v 1.26 1994/03/15 19:38:49 aydt Exp $
 */

#include "FUWrapper.h"

#include "FunctionalUnit.h"
#include "InputPipeSocket.h"
#include "InputPort.h"
#include "InterfaceClass.h"
#include "OutputPort.h"
#include "Pablo.h"
#include "RecordDictionary.h"
#include "RecordDictionaryPIterator.h"
#include "SystemErrors.h"

static InputPort  *NullInputPortPtr = NULL;
static OutputPort *NullOutputPortPtr = NULL;
static InputPipeSocket *NullSocketPtr = NULL;

FUWrapper::FUWrapper( FunctionalUnit *fu, const CString& className ) 
	  : Wrapper()
{
	_setClassName( MY_CLASS );

	functionalUnit = fu;
	fuClassName = className;
	functionalUnit->setWrapper( this );

	inputPorts = NULL;
	inputPortCnt = 0;
	outputPorts = NULL;
	outputPortCnt = 0;
	outputDossier = NULL;
	outputBindings = NULL;
	outputBindCnt = 0;
	inputSockets = NULL;
	inputSocketCnt = 0;
	userInputs = NULL;
	userInputCnt = 0;
}

FUWrapper::~FUWrapper()
{
	if ( inputPorts != NULL ) {
	    delete[] inputPorts;
	}

	if ( outputPorts != 0 ) {
	    delete[] outputPorts;
	}

	if ( outputDossier != NULL ) {
	    delete outputDossier;
	}

	if ( outputBindCnt != 0 ) {
	    delete[] outputBindings;
	}

	if ( inputSocketCnt != 0 ) {
	    for ( int i = 0; i < inputSocketCnt; i++ ) {
		delete inputSockets[i];
	    }
	    delete[] inputSockets;
	}

	if ( userInputCnt != 0 ) {
	    for ( int i=0; i < userInputCnt; i++ ) {
		delete userInputs[i];
	    }
	    delete[] userInputs;
	}

}

Boolean_			/* virtual */
FUWrapper::addInputPipe( StreamPipe *pipe, ModuleId *source )
{
	/*
	 * This is called by the graph editor... currently there is
	 * no way to delete an input pipe -- once it's added it's there
	 * for good.  The inputSockets are re-initialized whenever
	 * init() is called.
	 */

	InputPipeSocket* *ipsl = inputSockets;

	inputSockets = new InputPipeSocket* [ inputSocketCnt + 1 ];

	if ( inputSocketCnt != 0 ) {
	    int i;
	    for ( i = 0; i < inputSocketCnt; i++ ) {
		inputSockets[i] = ipsl[i];
	    }
	    delete[] ipsl;
	}

	inputSockets[ inputSocketCnt ] = new InputPipeSocket( pipe, this, 
								    source );
	inputSockets[ inputSocketCnt ]->init( inputSocketCnt );
	inputSocketCnt++;

	return SUCCESS_;
}

void 
FUWrapper::addInputPort( InputPort *port )
{
	/*
	 * This is called by the Functional Unit.  The list of input ports
	 * is cleared whenever ::init() is called but individual ports cannot
	 * be removed once they have been added.
	 */

	InputPort* *ippl = inputPorts;

	inputPorts = new InputPort* [ inputPortCnt + 1 ];

	if ( inputPortCnt != 0 ) {
	    int i;
	    for ( i = 0; i < inputPortCnt; i++ ) {
		inputPorts[i] = ippl[i];
	    }
	    delete[] ippl;
	}

	inputPorts[ inputPortCnt ] = port;
	inputPortCnt++;
}

void 
FUWrapper::addOutputPort( OutputPort *port )
{
	/*
	 * This is called by the Functional Unit.  The list of output ports
	 * is cleared whenever ::init() is called but individual ports cannot
	 * be removed once they have been added.
	 */

	OutputPort* *oppl = outputPorts;

	outputPorts = new OutputPort* [ outputPortCnt + 1 ];

	if ( outputPortCnt != 0 ) {
	    int i;
	    for ( i = 0; i < outputPortCnt; i++ ) {
		outputPorts[i] = oppl[i];
	    }
	    delete[] oppl;
	}

	outputPorts[ outputPortCnt ] = port;
	outputPortCnt++;
}

void
FUWrapper::addUserInput( const Value &value, int inPort, int outField )
{
	/*
	 * This is called by the InterfaceClass in the configuration 
	 * process of the module.  ::init() and ::resetConfiguration()
	 * both clear all userInputs.  ::loadConfigurationFromDir()
	 * sets userInputs directly.
	 */

	UserInputData* *uil = userInputs;

	userInputs = new UserInputData* [ userInputCnt + 1 ];

	if ( userInputCnt != 0 ) {
	    int i;
	    for ( i = 0; i < userInputCnt; i++ ) {
		userInputs[i] = uil[i];
	    }
	    delete[] uil;
	}

	userInputs[ userInputCnt ] = new UserInputData( value, inPort, 
							       outField );
	if ( inPort != -1 ) {
	    inputPorts[inPort]->setSocketBinding( inputSocketCnt, 
						  NullSocketPtr );
	}

	Assert( outField < outputBindCnt );

	userInputCnt++;
}

void
FUWrapper::bindOutputPort( int oport, int ofield )
{
	Assert( oport >= 0 );
	Assert( oport < outputPortCnt );
	Assert( ofield >= 0 );
	Assert( ofield < outputBindCnt );
	Assert( outputPorts[oport]->getTraits() == 
		outputDossier->getField( ofield )->getTraits() );

	outputBindings[ ofield ] = oport;
}
	
void 				/* virtual */
FUWrapper::configure( Boolean_ singleModule )
{
	/*  
	 * This is called by the infrastructure manager for system-wide
	 * configuration, or by the main graph interface for configuration
	 * of an individual module.  Expect the module to have at least
	 * one input pipe or it cannot be configured.  Expect the "source
	 * modules for all input pipes to have been configured if this 
	 * is a single module configuration. Configuration  consists of:
	 *   1) Call each inputSocket configure() method to load any
	 *      new descriptors into the dictionaries.  If there aren't
	 *      any dictionary entries on any socket, return.
	 *   2) Call the FU::configureOperation() method to select
	 *      the operation to be performed.
	 *   3) Call the User Interface to select actions for each
	 *      record on each pipe; to bind record fields to input ports and
	 *      output fields for 'field-extract' records; to create an
	 *      output record; to bind output ports to output record fields.
	 *   4) Call FU::configure() to configure display parameters, etc.
	 *   5) Call flushOutputDictionary() to send all entries downstream.
	 */

	if ( inputSocketCnt == 0 ) {
	    info( "No input pipes in configure" );
	    return;
	} 

	int i;

	if ( singleModule ) {
	    Boolean_ configureUpstream = FALSE_;
	    Wrapper *upstreamWrapper;
	    for ( i = 0; i < inputSocketCnt; i++ ) {
		upstreamWrapper = inputSockets[i]->getModuleId()->getWrapper();
		if ( ! upstreamWrapper->isConfigured() ) {
		    info( "Module %s must be configured prior to configuring \
			   \nthis module ( %s ).", 
				     upstreamWrapper->getName().getValue(),
				     getName().getValue() );
		    configureUpstream = TRUE_;
		}
	    }
	    if ( configureUpstream ) {
		return;
	    }
	}

	int inputRecordCnt = 0;
	for ( i = 0; i < inputSocketCnt; i++ ) {
	    inputSockets[i]->configure( singleModule );
	    inputRecordCnt += inputSockets[i]->entriesInDictionary();
	}
	if ( inputRecordCnt == 0 ) {
	    error ( "Module %s does not have any records on it's input pipe(s) \
		    \n and cannot be configured.", getName().getValue() );
	    return;
	}

	functionalUnit->configureOperation();

	InterfaceClass *ic = new InterfaceClass( this );
	ic->runDialog();
	delete ic;

	functionalUnit->configure();

	flushOutputDictionary();	

	needsConfiguration = FALSE_;
}

void 				/* virtual */
FUWrapper::configureParams( )
{
	if ( needsConfiguration ) {
	    info( "Module needs complete configuration, \
		   \nnot just parameter update\n" );
	    configure( TRUE_ );
	    return;
	} else {
	    functionalUnit->updateParameters();
	}
}


void				/* virtual */
FUWrapper::flushOutputDictionary()
{
	if ( binaryOutput == NULL ) {
	    return;
	}

	/* The outputDictionary is never actually used for this class.
	 * Instead, each time this method is called, we go through the 
	 * RecordDictionary associated with each  input pipe.  For each 
	 * entry in each of these dictionaries, we see if the action for 
	 * that entry is "Copy". If it is, write the corresponding 
	 * StructureDescriptor to the binary output pipe(s).
	 * Finally, if there is an active outputDossier, write the
	 * StructureDescriptor for it to the binary output pipe(s).
	 */

	int	            i; 
	InputPipeSocket     *socket;
	RecordDossier       *doss;
	int		    tag;

	for ( i = 0; i < inputSocketCnt; i++ ) {
 	    socket = inputSockets[ i ];
	    RecordDictionaryPIterator rdIter( socket->getDictionary() );

	    for ( doss = rdIter.first(); doss != NULL; doss = rdIter.next() ) {
		tag = doss->getTag();
		if ( ActionDoesCopy( socket->getAction( tag ) ) ) {
		    const StructureDescriptor &descr = 
					Pablo::StructureDict()->fetch( tag );
		    writeDescriptor( descr, tag );
		}
	    }

	}
		
	if ( outputDossier != NULL ) {
	    tag = outputDossier->getTag();
	    const StructureDescriptor &descr = 
					Pablo::StructureDict()->fetch( tag );
	    writeDescriptor( descr, tag );
	}
}

FunctionalUnit *		/* virtual */
FUWrapper::getFunctionalUnit() const
{
	return functionalUnit;
}


InputPort *
FUWrapper::getInputPort( const CString &pname ) const
{
	int i;
	for ( i = 0; i < inputPortCnt; i++ ) {
	    if ( inputPorts[i]->getName() == pname ) {
		return inputPorts[i];
	    }
	}
	warning( "No Input Port named %s", (const char *)pname );
	return NullInputPortPtr;
}

OutputPort *
FUWrapper::getOutputPort( const CString &pname ) const
{
	int i;
	for ( i = 0; i < outputPortCnt; i++ ) {
	    if ( outputPorts[i]->getName() == pname ) {
		return outputPorts[i];
	    }
	}
	warning( "No Output Port named %s", (const char *)pname );
	return NullOutputPortPtr;
}

void 				/* virtual */
FUWrapper::init()
{
	/*
	 * The initialize method resets the state of the module.
	 * It is called after the constructor and before saved configurations
	 * are reloaded. It
	 *    1) Resets runMode to SYNCHRONOUS_ and sets name
	 *    2) Resets list of input ports to NULL.
	 *    3) Resets list of output ports to NULL.
	 *    4) Resets output record to NULL.
	 *    5) Resets list of bindings from output ports to output record.
	 *    6) Resets list of user inputs
	 *    7) Calls inputSockets::init() to Clear dictionaries on all 
	 *       input pipes and reset binding tables.
	 *    8) Calls FU::init()
	 */
	Wrapper::init();

	runMode = SYNCHRONOUS_;
	_setObjectName( getName().getValue() );
	
	if ( inputPortCnt != 0 ) {
	    delete[] inputPorts;
  	    inputPortCnt = 0;
	    inputPorts = NULL;
	}

	if ( outputPortCnt != 0 ) {
	    delete[] outputPorts;
	    outputPortCnt = 0;
	    outputPorts = NULL;
	}

	if ( outputDossier != NULL ) {
	    delete outputDossier;
	    outputDossier = NULL;
	}

	if ( outputBindCnt != 0 ) {
	    delete[] outputBindings;
	    outputBindCnt = 0;
	    outputBindings = NULL;
	}

	if ( userInputCnt != 0 ) {
	    for ( int i=0; i < userInputCnt; i++ ) {
		delete userInputs[i];
	    }
	    delete[] userInputs;
	    userInputCnt = 0;
	    userInputs = NULL;
	}

	for ( int i = 0; i < inputSocketCnt; i++ ) {
	     inputSockets[i]->init( i );
	}

	functionalUnit->init();
}

Boolean_ 			/* virtual */
FUWrapper::loadConfigurationFromDir( const CString &dir, int moduleIndex )
{
	Assert( moduleIndex == moduleIdentifier );
	/* 
	 * We do:
	 *  1) Call init() to reset state. 
	 *  2) Locally reload the run mode, the traits for output ports, 
	 *     the output dossier, and outputPort->outputDossier field bindings.
	 *  3) Call FU::loadConfigFromFile to reset the FU specific details.
	 *  4) Load the userInput constant information. Must be after 3 so that
	 *     traits are set on input ports by FU for binding check.
	 *  5) For each Input Socket, call loadConfigurationFromDir() to 
	 *     reset the dictionary for the socket, set the Action table, 
	 *     and update binding info for input fields to input ports and
	 *     output Dossier fields.
	 *  6) Call flushOutputDictionary() to send entries downstream.
	 */
	static char buf[1024];
	int    i, number, rc;

	/* 
	 * On second thought, I don't think we need an init() here. The order
	 * in which this gets calls insures that the init() was done before
	 * we got to this point (the wrapper was created & then init() called.
	 * I'll just comment it out for now in case I'm wrong.... Ruth.
	 */
	/* init(); */

	sprintf( buf, "%s/module-%d.%s", dir.getValue() , moduleIndex,
						getName().getValue() );
	FILE *fp = fopen( buf, "r" );
	if ( fp == NULL ) {
	    warning( "loadConfigurationFromDir: Unable to open %s: %s", buf, 
							   errorString() );
	    configure( FALSE_ );
	    return SUCCESS_;
	}

	fscanf( fp, "%*[^\n]\n" );		// Comment line
	fscanf( fp, "%d\n", &number );		// runMode; 0=synch, 1=asynch
	if ( number == 1 ) {
	    runMode = ASYNCHRONOUS_;
	}

	fscanf( fp, "%*[^\n]\n" );		// Comment line
	fscanf( fp, "%d\n", &number );		// number of output ports
	if (  number != outputPortCnt ) {
	    warning( "Mismatch on number of output ports" );
	    fclose( fp );
	    configure( FALSE_ );
	    return SUCCESS_;
	}

	DataTraits *dt;
	for ( i = 0; i < outputPortCnt; i++ ) {
	    dt = DataTraits::loadConfigurationFromFP( fp );
	    outputPorts[i]->setTraits( *dt );
	    delete dt;
	}

	fscanf( fp, "%*[^\n]\n" );		// Comment line
	fscanf( fp, "%d\n", &number );		// tag of output dossier
	if ( number != -1 ) {
	    setOutputRecord( number );
	    fscanf( fp, "%*[^\n]\n" );		// Comment line
	    for ( i = 0; i < outputBindCnt; i++ ) {
		fscanf( fp, "%d", &outputBindings[i] );	
	    }
	    fscanf( fp, "\n" );
	}

	sprintf( buf, "%s/%s", dir.getValue(), getName().getValue() );
	RETURN_ON_FAILURE( functionalUnit->loadConfigFromFile( buf ) );

	/* Return to 'our' saved file to load user input info */

	fscanf( fp, "%*[^\n]\n" );		// Comment line
	fscanf( fp, "%d\n", &number );		// Number of User Constants
	if ( number != 0 ) {
	    userInputCnt = number;
	    userInputs = new UserInputData* [ userInputCnt ];

	    int inPort, outField;
	    Value value;

	    for ( i = 0; i < userInputCnt; i++ ) {
	        dt = DataTraits::loadConfigurationFromFP( fp );
		Assert( dt->isScalar() );

		switch( dt->getType() ) {
		    case CHARACTER:
			char c;
			fscanf( fp, "%c\n", &c );
			value = c;
			break;
		    case INTEGER:
			int n;
			fscanf( fp, "%d\n", &n );
			value = n;
			break;
		    case FLOAT:
			float f;
			fscanf( fp, "%f\n", &f );
			value = f;
			break;
		    case DOUBLE:
			double d;
			fscanf( fp, "%lf\n", &d );
			value = d;
			break;
		    case UNDEFINED:
			break;
		}

		fscanf( fp, "%d %d\n", &inPort, &outField );
	        userInputs[ i ] = new UserInputData( value, inPort, outField );

		if ( inPort != -1 ) {
		    inputPorts[inPort]->setSocketBinding( inputSocketCnt, 
							  NullSocketPtr );
		    rc = ( inputPorts[inPort]->constrainTraits( *dt ) );
		    Assert( rc );
		}
		delete dt;
	    }

	}

	fclose( fp );
		
	for ( i = 0; i < inputSocketCnt; i++ ) {
	    RETURN_ON_FAILURE( 
		inputSockets[i]->loadConfigurationFromDir( dir, moduleIndex ) );
	}

	flushOutputDictionary();
	needsConfiguration = FALSE_;

	return SUCCESS_;
}

Boolean_ 			/* virtual */
FUWrapper::ready()
{
	if ( inputSocketCnt == 0 ) {
	    return FALSE_;
	}

	if ( needsConfiguration ) {
	    return FALSE_;
	}

	/* 
	 * 1) Process data on each of our input pipes through the
	 *    inputPipeSocket interface.  This updates the values in the
	 *    InputPorts and sets the status there to NEW_.  Attribute
	 *    and Command packets are passed through.  Descriptor packets
	 *    update the dictionary for this module & if "Copy Thru" are also
	 *    passed down.
	 * 2) If none of the pipes have new data then they are all empty 
	 *    and we are NOT ready to run.
	 * 3) Update input port and output port values with user input data
	 *    if appropriate. (Resets port status to NEW for input ports)
	 * 4) If we are in SYNCHRONOUS_ mode, then if there is any input
	 *    port that does not have NEW_ data, we can't run.
	 * 5) If we are in ASYNCHRONOUS_ mode, then if there is any input
	 *    port that has INVALID_ data, we can't run.  Test (2) above
	 *    insured that there was new data on at least 1 of the ports.
	 */

	int i;
	Boolean_ anyData = FALSE_;		// Does any socket have data?

	for ( i = 0; i < inputSocketCnt; i++ ) {
	    Boolean_ hasData = inputSockets[i]->processData();
	    anyData = CnvToBoolean_( anyData || hasData );
	}

	if ( !anyData ) {			// No new data anywhere!
	    return FALSE_;
	}

	if ( userInputCnt > 0 ) {
	    /* 
	     * This isn't a very efficient way of doing this.  supposedly
	     * these values will never change so we could just set them once,
	     * taking care that the status on the inputPort(s) gets reset to
	     * NEW_ so that the checks below will work.  
	     */
	    for ( i = 0; i < userInputCnt; i++ ) {
		int inPort = userInputs[i]->inputPort;
		int outField = userInputs[i]->outputField;
		if ( inPort != -1 ) {
		    inputPorts[inPort]->setValueP( &userInputs[i]->dataValue, 
					 	   -1 );
		}
		if ( outField != -1 ) {
		    outputDossier->setValue( outField, 
					     userInputs[i]->dataValue );
		}
	    }
	}
		    
	if ( runMode == SYNCHRONOUS_ ) {	// Needs new data everywhere
	    for ( i = 0; i < inputPortCnt; i++ ) {
	        if ( inputPorts[i]->valueStatus() != NEW_ ) {
		   return FALSE_;
		}
	    }
	} else {				// Needs valid data everywhere
	    for ( i = 0; i < inputPortCnt; i++ ) {
		if ( inputPorts[i]->valueStatus() == INVALID_ ) {
		    return FALSE_;
		}
	    }
	}

	return TRUE_;
}

void
FUWrapper::resetConfiguration()
{
	/* 
	 * Called by the InterfaceClass to clear out 'old' configuration
	 * information before updating with 'new' configuration information.
	 * Eventually we may change our outputDossier, but not now.
	 */
	int i;

	for ( i = 0; i < outputBindCnt; i++ ) {
	     outputBindings[i] = -1;
	}

	if ( userInputCnt != 0 ) {
	    for ( i = 0; i < userInputCnt; i++ ) {
		delete userInputs[i];
	    }
	    delete [] userInputs;
	    userInputCnt = 0;
	    userInputs = NULL;
	}
}

void				/* virtual */
FUWrapper::restart()
{
	functionalUnit->restart();
}
	    
void 				/* virtual */
FUWrapper::run( Boolean_& errorFlag )
{
	functionalUnit->run( errorFlag );

	/* 
	 * We do this to insure that none of the input ports are left with
	 * a NEW status -- that could hang up the inputPipeSocket::processData
	 * routine.  The only way that should happen is if the FU doesn't
	 * do a getValue() or getValueP() for each of it's input ports.
	 * We may want to make the FU call markValueSeen() itself on those
	 * unused ports so we don't have to do it here for all of them!
	 */
	int i;
	for ( i = 0; i < inputPortCnt; i++ ) {
	    inputPorts[i]->markValueSeen();
	}
}

Boolean_ 			/* virtual */
FUWrapper::saveConfigurationToDir( const CString &dir, int moduleIndex ) const
{
	/* 
	 * We do:
	 *  1) Locally save the run mode, the traits for output ports, 
	 *     the output dossier, the outputPort->outputDossier field bindings,
	 *     and the user input constant bindings.
	 *  2) For each Input Socket, call saveConfigurationToDir() to save
	 *     the dictionary(?) for the socket, save the Action table,
	 *     and save binding info for input fields to input ports and
	 *     output Dossier fields.
	 *  3) Call FU::saveConfigToFile to save the FU specific details.
	 */
	static char buf[1024];
	int    i;

	sprintf( buf, "%s/module-%d.%s", dir.getValue() , moduleIndex, 
						getName().getValue() );
	FILE *fp = fopen( buf, "w" );
	if ( fp == NULL ) {
	    warning( "saveConfigToDir::Unable to open %s: %s", buf, 
							  errorString() );
	    return FAILURE_;
	}

	fprintf( fp, "# Run Mode: 0=synchronous; 1=asynchronous\n");
	fprintf( fp, "%d\n", runMode );	

	fprintf( fp, "# Number of output ports and their traits\n" );
	fprintf( fp, "%d\n", outputPortCnt );
	for ( i = 0; i < outputPortCnt; i++ ) {
	    RETURN_ON_FAILURE( outputPorts[i]->
				      getTraits().saveConfigurationToFP( fp ) );
	}

	fprintf( fp, "# Output Record Tag (-1 => none)\n" );
	if ( outputDossier != NULL ) {
	    fprintf( fp, "%d\n", outputDossier->getTag() );
	    fprintf( fp, "# For each output field, output port bound to it\n");
	    for ( i = 0; i < outputBindCnt; i++ ) {
		fprintf( fp, "%d ", outputBindings[i] );	
	    }
	    fprintf( fp, "\n" );
	} else {
	    fprintf( fp, "-1\n" );
	}

	fprintf( fp, "# User-input constants and bindings (count first)\n" );	
	fprintf( fp, "%d\n", userInputCnt );	
	if ( userInputCnt != 0 ) {
	    for ( i = 0; i < userInputCnt; i++ ) {
		Value *vp = &userInputs[i]->dataValue;
		RETURN_ON_FAILURE( vp->getTraits().saveConfigurationToFP(fp) );
		Assert( vp->getTraits().isScalar() );
		switch( vp->getTraits().getType () ) {
		    case CHARACTER:
			fprintf( fp, "%c\n", vp->operator char() );
			break;
		    case INTEGER:
			fprintf( fp, "%d\n", vp->operator int() );
			break;
		    case FLOAT:
			fprintf( fp, "%f\n", vp->operator float() );
			break;
		    case DOUBLE:
			fprintf( fp, "%lf\n", vp->operator double() );
			break;
		    case UNDEFINED:
			break;
		}
		fprintf( fp, "%d %d\n", userInputs[i]->inputPort,
				        userInputs[i]->outputField );
	    }
	}
	fclose( fp );
		
	for ( i = 0; i < inputSocketCnt; i++ ) {
	    RETURN_ON_FAILURE( 
		inputSockets[i]->saveConfigurationToDir( dir, moduleIndex ) );
	}

	sprintf( buf, "%s/%s", dir.getValue(), getName().getValue() );
	RETURN_ON_FAILURE( functionalUnit->saveConfigToFile( buf ) );

	return SUCCESS_;
}

Boolean_
FUWrapper::setOutputRecord( int sysTag )
{
	/* 
	 * Look up the system tag in the global structure dictionary.
	 * If there is an entry, we're in business...
	 *   1) Delete active outputDossier (if one exists) and
	 *      free space for previous outputBindings. 
	 *   2) Create new outputDossier instance with given system tag and
	 *	corresponding structure descriptor.
	 *   3) Create new outputBindings table with one slot for each
	 *      field in the outputDossier -- initialize all slots to -1 
	 *      indicating no output fields are yet satisfied by output ports.
	 *   4) Return SUCCESS_
	 * Else...
	 *   1) Print a warning message 
	 *   2) Return FAILLURE_
	 */

	Assert( outputPortCnt != 0 );	// don"t expect output record if no
					// output ports!

	const StructureDescriptor& descr = 
					Pablo::StructureDict()->fetch( sysTag );

	if ( descr.notNull() ) {

 	    Pablo::TagMappingDict()->insert( moduleIdentifier, 0, sysTag );

	    int i;
	    if ( outputBindCnt != 0 ) {	
	    	delete outputDossier;
	    	delete[] outputBindings;
	    }

	    outputDossier = new RecordDossier( sysTag, descr );
	    outputBindCnt = outputDossier->entryCount();
	    outputBindings = new int [outputBindCnt];

	    for ( i = 0; i < outputBindCnt; i++ ) {
		outputBindings[i] = -1;
	    }
	    return SUCCESS_;

	} else {

	    warning( "setOutputRecord(%d): not in global dictionary", sysTag );
	    return FAILURE_;

	}
}

void 				/* virtual */
FUWrapper::writeLayoutToFP( FILE *fp ) const
{
	fprintf( fp, "F %s %s\n", fuClassName.getValue(), 
				  getName().getValue() );
}

void 
FUWrapper::writeOutputRecord()
{
	/*
	 * Make sure we have an output pipe!
	 */
	if ( binaryOutput == NULL ) {
	    return;
	}

	/* 
 	 * This is called from a Functional Unit run:: method when it does 
	 * a _writeOutput().  We first update any fields in the outputDossier 
	 * that are satisfied by the FU output ports, and then write the 
	 * outputDossier to the output pipe(s).  Note that the outputDossier 
	 * fields that are filled by fields from the input records are not 
	 * updated here -- they were updated when the field extract was 
	 * performed on the input record.
	 */

	int i, oport;

	for ( i = 0; i < outputBindCnt; i++ ) {
	    if ( ( oport = outputBindings[i] ) >= 0 ) {
		outputDossier->setValue( i, outputPorts[oport]->getValue() );
	    }
	}
	writeData( *outputDossier );
}

/*
 * Misc helper methods
 */
void 
FUWrapper::printOn( ostream &os ) const 
{
	int i;
	
	os << form( "FUWrapper: this=0x%x identifier=%d FU=0x%x\n",
		     this, moduleIdentifier, functionalUnit );

	os << "Input ports:\n";
	for ( i = 0; i < inputPortCnt; i++ ) {
	    os << i << ": " << *inputPorts[i] << NL;
	}

	os << "Output ports:\n";
	for ( i = 0; i < outputPortCnt; i++ ) {
	    os << i << ": " << *outputPorts[i] << NL;
	}

	os << "Input Pipe Sockets:\n";
	for ( i = 0; i < inputSocketCnt; i++ ) {
		os << i << ": " << *inputSockets[i] << NL;
	}
}

/*
 * Initialize the static data
 */
const char *const FUWrapper::MY_CLASS = "FUWrapper";
