/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * InterfaceClass.cc: Implements a Functional Unit Wrapper.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Wrapper/RCS/InterfaceClass.C,v 1.34 1994/03/15 19:39:04 aydt Exp $
 */

#include "InterfaceClass.h"

#include "ConfigBoard.h"
#include "ConstantsDialog.h"
#include "ExecutionModeDialog.h"
#include "OutputSpecificationDialog.h"
#include "PipePortBindingDialog.h"
#include "RecordBindingDialog.h"

#include "FunctionalUnit.h"
#include "InputPipeSocket.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "Pablo.h"
#include "RecordDictionaryPIterator.h"
#include "WidgetBase.h"

static const char* UserInput = "User Input";

InterfaceClass::InterfaceClass( FUWrapper *fuw )
{
	/*
	 * g++ isn't correctly freeing all memory that is allocated when
	 * structures are allocated via new, so I'm preallocating lots
	 * of space...
	 */
	_setClassName( MY_CLASS );

	fuWrapper = fuw;
	notInitialized = TRUE_;
	dictionaryLookup = 
		     new StructureDictionaryInquirer(*Pablo::StructureDict() );

	inputPipeCnt = fuWrapper->inputSocketCount();
	inputPipeInfo = new inputPipeStruct [ inputPipeCnt + 1 ];

	inputPortCnt = fuWrapper->inputPortCount();
	if ( inputPortCnt > MAX_NUMBER_INPUT_PORTS ) {
	    abort( "Maximum number of input ports (%d) exceeded.",
		    MAX_NUMBER_INPUT_PORTS );
	}

	outputPortCnt = fuWrapper->outputPortCount();
	if ( outputPortCnt > MAX_NUMBER_OUTPUT_PORTS ) {
	    abort( "Maximum number of output ports (%d) exceeded.",
	            MAX_NUMBER_OUTPUT_PORTS );
	}

	fieldBindingCnt = 0;
	userInputCnt = 0;
	outputFieldCnt = 0;
}

InterfaceClass::~InterfaceClass()
{
	int i, j;

	for ( i = 0; i < inputPipeCnt; i++ ) {
	    for ( j = 0; j < inputPipeInfo[i].recordCnt; j++ ) {
	        delete inputPipeInfo[i].record[j];
	    }
	    delete[] inputPipeInfo[i].record;
	}
	
	
	delete[] inputPipeInfo;

	delete dictionaryLookup;
}

int
InterfaceClass::_addStructureToSystem()
{
	// Note that if this field was satisfied from an output port it"s
	// possible that the field type isn"t set yet.  So, check that!

	StructureDescriptorResolver resolver;
	Attributes           	    attributes;
	FieldDescriptor     	    *fieldP;

	attributes.insert( "created by", getModuleName() );
	StructureDescriptor structure( outputRecordName, attributes );

	for ( int i = 0; i < outputFieldCnt; i++ ) {
	    outputFieldStruct& oField = outputFieldInfo[i];
	    if ( oField.source == Output_Port ) {
		oField.traits = 
			fuWrapper->getOutputPort(oField.srcIndex)->getTraits();
	    }
	    Assert( oField.traits.isDefined() );

	    fieldP = new FieldDescriptor( oField.name );
	    fieldP->setTraits( oField.traits );
	    structure.insert( *fieldP );
	    delete fieldP;
	}

	int sysTag = resolver.determineTag( fuWrapper->getIdentifier(), 0, 
								   structure );
	if ( sysTag == FAILURE_ ) {
	    abort( "Error inserting new Descriptor %s in Dictionary\n",
		    structure.getName().getValue() );
	}
	return sysTag;
}

int
InterfaceClass::_getRealInputPortNum( int pipe, int portIdx ) const
{
	int     cnt = -1;
	Sources src;
	int     srcIdx;

	for ( int i = 0; i < inputPortCnt; i++ ) {
	    src = inputPortInfo[i].source;
	    srcIdx = inputPortInfo[i].srcIndex;

	    if ( ( src == User_Input) && ( pipe == inputPipeCnt ) ) {
	        cnt++;
	    } else if ( ( src == Input_Pipe ) && ( pipe == srcIdx ) ) {
	        cnt++;
	    }

	    if ( cnt == portIdx ) {
		return( i );
	    }
	}
	return -1;
}

int
InterfaceClass::_getRealOutputFieldNum( int pipe, int fieldIdx ) const
{
	int 	cnt = -1;
	Sources src;
	int 	srcIdx;

	for ( int i = 0; i < outputFieldCnt; i++ ) {
	    src = outputFieldInfo[i].source;
	    srcIdx = outputFieldInfo[i].srcIndex;

	    if ( ( src == Output_Port ) && ( pipe == -1 ) ) {
	        cnt++;
	    } else if ( ( src == User_Input ) && ( pipe == inputPipeCnt ) ) {
	        cnt++;
	    } else if ( ( src == Input_Pipe ) && ( pipe == srcIdx ) ) {
	        cnt++;
	    }

	    if ( cnt == fieldIdx ) {
		return( i );
	    }
	}
	return -1;
}

Boolean_
InterfaceClass::_inputPortSetValid() 
{
	int i,o;
	Boolean_ allBound = TRUE_;
	for ( i = 0; i < inputPortCnt; i++ ) {
	    allBound = CnvToBoolean_( allBound && 
		       (fuWrapper->getInputPort(i)->getConstraintsCnt() > 0) );
	}

	if ( ! allBound ) {		// too soon to tell if invalid!
	    return TRUE_;
	}
		    
        if ( fuWrapper->getFunctionalUnit()->inputTraitsValid() ) {
	    // Valid, so set traits of output fields set from output ports
	    for ( i = 0; i < outputFieldCnt; i++ ) {
		if ( outputFieldInfo[i].source == Output_Port ) {
		    if ( outputFieldInfo[i].bindCnt > 0 ) {
		        o = outputFieldInfo[i].srcIndex;
			Assert( ( 0 <= o )  && ( o < outputPortCnt ) );
			outputFieldInfo[i].traits = 
				fuWrapper->getOutputPort(o)->getTraits();
		    }
		}
	    }

            return TRUE_;		// set is okay!

        } else {
            CString mesg;
            mesg = "FU will not accept this set of input types:\n";

            for ( i = 0; i < inputPortCnt; i++ ) {
                DataTraits traits = fuWrapper->getInputPort(i)->getTraits();
                mesg = mesg + "    ";
                mesg = mesg + inputPortInfo[i].name;
                mesg = mesg + " ";
                mesg = mesg + traits.getTypeName();
                mesg = mesg + " ";
                mesg = mesg + form( "%d", traits.getDimension() );
                mesg = mesg + "\n";
            }

            info( mesg.getValue() );
            return FALSE_;
        }
}

int
InterfaceClass::_nextFieldBindingIdx()
{
	if ( ++fieldBindingCnt == MAX_NUMBER_FIELD_BINDINGS ) {
	    abort( "Maximum number of field bindings (%d) exceeded",
		    MAX_NUMBER_FIELD_BINDINGS );
	    return -1;		// not reached
	} else {
	    return( fieldBindingCnt - 1 );
	}
}

int
InterfaceClass::_nextOutputFieldIdx()
{
	if ( ++outputFieldCnt == MAX_NUMBER_OUTPUT_FIELDS ) {
	    abort( "Maximum number of output fields (%d) exceeded",
		    MAX_NUMBER_OUTPUT_FIELDS );
	    return -1;		// not reached
	} else {
	    return( outputFieldCnt - 1 );
	}
}

int
InterfaceClass::_nextUserInputIdx()
{
	if ( ++userInputCnt == MAX_NUMBER_USER_INPUTS ) {
	    abort( "Maximum number of field bindings (%d) exceeded",
		    MAX_NUMBER_USER_INPUTS );
	    return -1;		// not reached
	} else {
	    return( userInputCnt - 1 );
	}
}

void
InterfaceClass::_setValueFromString( char *string, Value& value,
				     const DataTraits& traits )
{
	char   	      c;
	int    	      i;
	float	      f;
	double 	      lf;
	int	      cnt;

	if ( traits.isUndefined() ) {

	    /*
	     * Scan the input.  If it's numeric, treat floats as doubles.
	     */

	    cnt = sscanf( string, "%c", &c );
	    if ( cnt == 1 ) {
		value = c;
	    }
	    
	    cnt = sscanf( string, "%lf", &lf );
	    if ( cnt == 1 ) {
		value = lf;
	    }

	    cnt = sscanf( string, "%d%c", &i, &c );
	    if ( (cnt == 1) || ( (cnt == 2) && (c != '.') ) ){
		value = i;
	    }
	    

	} else {

            Assert( traits.isScalar() );

            switch( traits.getType() ) {
                case CHARACTER:
                    cnt = sscanf( string, "%c", &c );
		    if ( cnt > 0 ) {
                        value = c;
		    } else {
			value = ' ';
		    }
                    break;
                case INTEGER:
                    cnt = sscanf( string, "%d", &i );
		    if ( cnt > 0 ) {
                        value = i;
		    } else {
			value = 0;
		    }
                    break;
                case FLOAT:
                    cnt = sscanf( string, "%f", &f );
		    if ( cnt > 0 ) {
                        value = f;
		    } else {
			value = 0.0;
		    }
                    break;
                case DOUBLE:
                    cnt = sscanf( string, "%lf", &lf );
		    if ( cnt > 0 ) {
                        value = lf;
		    } else {
			value = 0.0;
		    }
                    break;
		case UNDEFINED:
		    break;
	    }
	}
}

void
InterfaceClass::_showInputPortBindings()
{
	configBoard->clearAllPipePortExtract();
	configBoard->clearAllPipeOutputExtract();

	int i, in, out, pipe, record, field;
	RecordDossier* dossier;

	for ( i = 0; i < fieldBindingCnt; i++ ) {
	    in = fieldBindingInfo[i].inPort;
	    out = fieldBindingInfo[i].outField;
	    if ( ( in != -1 ) || ( out != -1 ) ) {
	        pipe = fieldBindingInfo[i].pipe;
	        record = fieldBindingInfo[i].record;
	        field = fieldBindingInfo[i].field;
	        dossier = inputPipeInfo[pipe].record[record]->dossier;

	        if ( in != -1 ) {
		    configBoard->fieldExtractPipePort( 
			      in,
			      dossier->getName().getValue(),
			      dossier->getField(field)->getName().getValue() );
		}

	        if ( out != -1 ) {
		    configBoard->fieldExtractPipeField( 
			     outputFieldInfo[out].name.getValue(),
			     dossier->getName().getValue(),
			     dossier->getField(field)->getName().getValue() );
		}
	    }
	}
}

void
InterfaceClass::_showModuleResultBindings() 
{
	configBoard->clearAllOutputPortFieldBindings();

	int i, outPort;

	for ( i = 0; i < outputFieldCnt; i++ ) {
	    if ( outputFieldInfo[i].source == Output_Port ) {
	        outPort = outputFieldInfo[i].srcIndex;
		if ( outPort != -1 ) {
	            Assert( outPort < outputPortCnt );
	            configBoard->addOutputPortFieldBinding( outPort,
	        		outputFieldInfo[i].name.getValue() );
		}
             }
	}
}

const char *
InterfaceClass::_showValueAsString( const Value& value ) const
{
	static char buf[128];
	
	if ( value.getTraits().isUndefined() ) {
	    return NULLCHARSTRING;
	} else {
	    Assert( value.getTraits().isScalar() );

            switch( value.getTraits().getType () ) {
                case CHARACTER:
                    sprintf( buf, "%c", value.operator char() );
                    break;
                case INTEGER:
                    sprintf( buf, "%d", value.operator int() );
                    break;
                case FLOAT:
                    sprintf( buf, "%f", value.operator float() );
                    break;
                case DOUBLE:
                    sprintf( buf, "%lf", value.operator double() );
                    break;
		case UNDEFINED:		// keep compiler happy
		    break;
            }
	    return buf;
	}
}

/*********** General Methods used by the FUWrapper and Binding Dialog ********/

const char *
InterfaceClass::getModuleName() const
{
	return ( fuWrapper->getName().getValue() );
}

void
InterfaceClass::initialize()
{
	int i, j, f, p, in, out;

	previouslyConfigured = FALSE_;
	fieldBindingCnt = 0;

	/* Input Pipe information */
	for ( i = 0; i < inputPipeCnt; i++ ) {

	    inputPipeStruct& pipeInfo = inputPipeInfo[i];
	    InputPipeSocket *isocket = fuWrapper->getInputSocket( i );
	    Assert( isocket->getPipeIndex() == i );

	    if ( notInitialized ) {
	        pipeInfo.name = isocket->getModuleId()->getName().getValue();

		RecordDictionaryPIterator iterator( isocket->getDictionary() );
	    	pipeInfo.recordCnt = isocket->getDictionary().entryCount();
	    	pipeInfo.record = new recordStruct* [ pipeInfo.recordCnt ];

	        for ( j = 0; j < pipeInfo.recordCnt; j++ ) {
		    pipeInfo.record[j] = new recordStruct;
		    pipeInfo.record[j]->dossier = iterator.next();
		    pipeInfo.record[j]->tag = 
				  pipeInfo.record[j]->dossier->getTag();
		}
		
	    } 

	    pipeInfo.inputPortsFilled = isocket->inputPortsFilled();
	    pipeInfo.outputFieldsFilled = isocket->outputFieldsFilled();

	    /* Fill input port source and source index info */
	    for ( j = 0; j < pipeInfo.inputPortsFilled; j++ ){
		f = isocket->getInputPort(j);
		inputPortInfo[f].source = Input_Pipe;
		inputPortInfo[f].srcIndex = i;
	    }

	    /* Fill output field source and source index info */
	    for ( j = 0; j < pipeInfo.outputFieldsFilled; j++ ){
		f = isocket->getOutputField(j);
		outputFieldInfo[f].source = Input_Pipe;
		outputFieldInfo[f].srcIndex = i;
		outputFieldInfo[f].bindCnt = 0;	    // incr below when field bnd
	    }
	    
	    /* Fill record action and field binding info */
	    for ( j = 0; j < pipeInfo.recordCnt; j++ ) {
	        pipeInfo.record[j]->action =
			          isocket->getAction( pipeInfo.record[j]->tag );
		
		if ( ActionDoesExtraction( pipeInfo.record[j]->action ) ) {
		    for ( p = 0; p < pipeInfo.inputPortsFilled; p++ ) {
		        f = _nextFieldBindingIdx();
			fieldBindingInfo[f].pipe = i;
			fieldBindingInfo[f].record = j;
			fieldBindingInfo[f].field = 
						isocket->getFieldToInputPort( 
			  		           pipeInfo.record[j]->tag, p );
			fieldBindingInfo[f].inPort = isocket->getInputPort( p );
			fieldBindingInfo[f].outField = -1;
		    }
		    for ( p = 0; p < pipeInfo.outputFieldsFilled; p++ ) {
		        f = _nextFieldBindingIdx();
			fieldBindingInfo[f].pipe = i;
			fieldBindingInfo[f].record = j;
			fieldBindingInfo[f].field = 
				                isocket->getFieldToOutputField( 
			  		           pipeInfo.record[j]->tag, p );
			fieldBindingInfo[f].inPort = -1;
			fieldBindingInfo[f].outField = 
					        isocket->getOutputField( p );
			outputFieldInfo[fieldBindingInfo[f].outField].bindCnt++;
		    }
		}
	    }

	}

	/* User Input information */
	in = 0;
	out = 0;

	userInputCnt = fuWrapper->userInputCount();
	for ( i = 0; i < userInputCnt; i++ ) {
	    FUWrapper::UserInputData *userIn = fuWrapper->getUserInput( i );
	    userInputInfo[i].dataValue = userIn->dataValue;
	    userInputInfo[i].inputPort = userIn->inputPort;
	    userInputInfo[i].outputField = userIn->outputField;

	    /* Fill input port source and source index info */
	    p = userInputInfo[i].inputPort;
	    if ( p != -1 ) {
		inputPortInfo[p].source = User_Input;
		inputPortInfo[p].srcIndex = i;
		in++;
	    }
	    /* Fill output field source and source index info */
	    f = userInputInfo[i].outputField;
	    if ( f != -1 ) {
		outputFieldInfo[f].source = User_Input;
		outputFieldInfo[f].srcIndex = i;
		outputFieldInfo[f].bindCnt = 1;
		out++;
	    }
	}
	/* Fill appropriate slot in inputPipeInfo array */
	inputPipeInfo[inputPipeCnt].name = UserInput;
	inputPipeInfo[inputPipeCnt].inputPortsFilled = in;
	inputPipeInfo[inputPipeCnt].outputFieldsFilled = out;


	/* Input Port information */
	for ( i = 0; i < inputPortCnt; i++ ) {
	    InputPort *iport = fuWrapper->getInputPort( i );

	    if ( notInitialized ) {
	        inputPortInfo[i].name = iport->getName().getValue();
	        inputPortInfo[i].initialTraits = iport->getTraits();
	        inputPortInfo[i].initialConstraintsCnt = 
						 iport->getConstraintsCnt();
	    } else {
		iport->resetTraits( inputPortInfo[i].initialTraits,
				    inputPortInfo[i].initialConstraintsCnt );
	    }

	    j = iport->getSocketNumber();
	    if ( j == inputPipeCnt ) {
		Assert( inputPortInfo[i].source == User_Input );
	    } else if ( ( 0 <= j ) && ( j < inputPipeCnt ) ) {
		Assert( inputPortInfo[i].source == Input_Pipe );
	        Assert( inputPortInfo[i].srcIndex == j );
	    } else if ( j == -1 ) {
		inputPortInfo[i].source = Unbound;
	    } else {
		abort( "invalid socket number %d", j );
	    }

	    if ( inputPortInfo[i].source != Unbound ) {
		previouslyConfigured = TRUE_;
		Assert( inputPortInfo[i].initialTraits.isDefined() );
		Assert( inputPortInfo[i].initialConstraintsCnt > 0 );
	    }
	}

	/* Output Port information */
	for ( i = 0; i < outputPortCnt; i++ ) {
	    OutputPort *oport = fuWrapper->getOutputPort( i );
	    if ( notInitialized ) {
	        outputPortInfo[i].name = oport->getName().getValue();
		outputPortInfo[i].initialTraits = oport->getTraits();
	    } else {
		oport->setTraits( outputPortInfo[i].initialTraits );
	    }
	}

	/* Output Field information */
	if ( outputPortCnt > 0 ) {
	    outputDossier = fuWrapper->getDossier();

	    if ( outputDossier == NULL ) {
		Assert( ! previouslyConfigured );

		strcpy( outputRecordName, "<<" );
		strcat( outputRecordName, getModuleName() );
		strcat( outputRecordName, ">>" );
		
		outputFieldCnt = outputPortCnt;
		for ( i = 0; i < outputFieldCnt; i++ ) {
		    outputFieldInfo[i].name = outputPortInfo[i].name;
		    outputFieldInfo[i].source = Output_Port;
		    outputFieldInfo[i].srcIndex = i;
		    outputFieldInfo[i].traits = outputPortInfo[i].initialTraits;
		    outputFieldInfo[i].bindCnt = 1;
		}
	    } else {
		Assert( previouslyConfigured );
		strcpy( outputRecordName, outputDossier->getName().getValue() );
		outputFieldCnt = outputDossier->entryCount();
		for ( i = 0; i < outputFieldCnt; i++ ) {
		    FieldDescriptor *field = outputDossier->getField( i );
		    outputFieldInfo[i].name = field->getName();
		    outputFieldInfo[i].traits = field->getTraits();
		}

		// find output fields filled from output ports 
		for ( i = 0; i < outputFieldCnt; i++ ) {
		    j = fuWrapper->getOutputPortBindingForField( i );
		    if ( j != -1 ) {
			outputFieldInfo[i].source = Output_Port;
			outputFieldInfo[i].srcIndex = j;
			outputFieldInfo[i].bindCnt = 1;
		    }
		}
	    }
	} 
		    
	/*
	 * Currently we don't let people change the type of the input
	 * port or output fields on reconfiguration.  To enforce this,
	 * we artifically increase the number of constraints on the
	 * input port and output fields traits.
	 */
	if ( previouslyConfigured ) {
	    for ( i = 0; i < inputPortCnt; i++ ) {
	        fuWrapper->getInputPort(i)->lockTraits();
	    }
	    for ( i = 0; i < outputFieldCnt; i++ ) {
		outputFieldInfo[i].bindCnt++;
	    }
	}

	/*
	 * Go through the output field information and save the values
	 * we have at this point -- used if restoreOriginalFieldBindings()
	 * is called.
	 */
	for ( i = 0; i < outputFieldCnt; i++ ) {
	    outputFieldInfo[i].origIndex = outputFieldInfo[i].srcIndex;
	    outputFieldInfo[i].origTraits = outputFieldInfo[i].traits;
	    outputFieldInfo[i].origCnt = outputFieldInfo[i].bindCnt;
	}

	runMode = fuWrapper->getExecutionMode();
	notInitialized = FALSE_;
}

Boolean_ 
InterfaceClass::moduleConfiguredPreviously()
{
	return previouslyConfigured;
}

void
InterfaceClass::runDialog()
{
	int i; 

	/* initialize our state from FUWrapper and associated classes */
	initialize();

	Widget topLevel = Pablo::TopLevel()->getWidget();

	configBoard = new ConfigBoard( XtDisplay(topLevel), this );
        configBoard->registerInputPipes();
        configBoard->registerInputPorts();
        configBoard->registerOutputPorts();
        configBoard->realize();

	if ( ! previouslyConfigured ) {
	    /* 
	     * If we haven't been configured before, call 
	     * Screen1: Pipe to InputPort Binding Dialog
	     * Screen2: Output Record Specification Dialog
	     */

	    if ( inputPortCnt != 0 ) {
	        if ( ( inputPipeCnt > 1 ) || ( inputPortCnt > 1 ) ) {
	            PipePortBindingDialog *PPdialog = 
		       new PipePortBindingDialog( topLevel, this, configBoard );
	            PPdialog->run();
	            delete PPdialog;
	        } else {
		    bindPipeToInputPort( 0, 0 );     // no choice so don"t ask
		    configBoard->addInputPipePortBinding( 0, 0 );
		}
	    }

            if ( outputPortCnt > 0 ) {
		OutputSpecificationDialog *OSdialog = 
		   new OutputSpecificationDialog( topLevel, this, configBoard );
		OSdialog->run();
		delete OSdialog;
	    }

	} else {
	    /* 
	     * Bring status board up to date if we don't use screens one
	     * and two.  
	     * - Show output fields
	     * - Show binding from input pipes to input ports and output fields
	     * - Show bindings from user inputs to input ports and output fields
	     */
	    int pipe, inPort, outField;

	    for ( i = 0; i < outputFieldCnt; i++ ) {
	        configBoard->addOutputRecordField( 
	    			outputFieldInfo[i].name.getValue() );
	    }

	    for ( i = 0; i < fieldBindingCnt; i++ ) {
		pipe = fieldBindingInfo[i].pipe;

		inPort = fieldBindingInfo[i].inPort;
		if ( inPort != -1 ) {
		    configBoard->addInputPipePortBinding( pipe, inPort );
		}

		outField = fieldBindingInfo[i].outField;
		if ( outField != -1 ) {
		    configBoard->addInputPipeOutputFieldBinding( pipe,
				outputFieldInfo[outField].name.getValue() );
		}
	    }

	    pipe = inputPipeCnt;	// signals user input
	    for ( i = 0; i < userInputCnt; i++ ) {
		inPort = userInputInfo[i].inputPort;
		if ( inPort != -1 ) {
		    configBoard->addInputPipePortBinding( pipe, inPort );
		}

		outField = userInputInfo[i].outputField;
		if ( outField != -1 ) {
		    configBoard->addInputPipeOutputFieldBinding( pipe,
				 outputFieldInfo[outField].name.getValue() );
		}
	    }
	}

	_showModuleResultBindings();
	_showInputPortBindings();
		
	/*
	 * If we have either input ports or output fields that are
	 * satisfied by user-input constant values, then call
	 * Screen3: Constants Dialog.  We save the original User Input
	 * values so that we can restore them if reset hit in Screen 3.
	 */
	if ( userInputCnt > 0 ) {
	    shadowConstants = new Value* [userInputCnt];
	    int constantsCnt = 0;

	    for ( i = 0; i < userInputCnt; i++ ) {
		shadowConstants[i] = new Value( userInputInfo[i].dataValue );
	        if ( userInputInfo[i].inputPort != -1 ) {
		    constantsCnt++;
	        }
	        if ( userInputInfo[i].outputField != -1 ) {
		    constantsCnt++;
	        }
	    }

	    if ( constantsCnt > 0 ) {
	        ConstantsDialog *ConstDialog = 
		       new ConstantsDialog( topLevel, this, configBoard );
	        ConstDialog->run();
	        delete ConstDialog;
	    }

	    for ( i = 0; i < userInputCnt; i++ ) {
		delete shadowConstants[i];
	    }
	    delete[] shadowConstants;

	}

	/*
	 * Always call 
	 * Screen4: Record Binding Dialog
	 */
	RecordBindingDialog *RBdialog = 
		   new RecordBindingDialog( topLevel, this, configBoard );
	
	RBdialog->run();
	delete RBdialog;

	/*
	 * If we have multiple input ports which are filled from multiple
	 * input pipes, then call
	 * Screen5: Module Execution Dialog
	 */
	if ( (inputPortCnt > 1) && (inputPipeCnt > 1) ) {
	    int pipeToPortCnt = 0;
	    for ( int i = 0; i < inputPipeCnt; i++ ) {
		if ( inputPipeInfo[i].inputPortsFilled > 0 ) {
		    pipeToPortCnt++;
		}
	    }
	    if ( pipeToPortCnt > 1 ) {
		ExecutionModeDialog *exModeDialog = 
		    new ExecutionModeDialog( topLevel, this, configBoard );
		exModeDialog->run();
		delete exModeDialog;
	    } else {
		runMode = SYNCHRONOUS_;
	    }
	} else {
	    runMode = SYNCHRONOUS_;
	}

	delete configBoard;

	/*
	 * Undo our locked traits on the input ports. No need to
	 * worry about output fields.
	 */
	if ( previouslyConfigured ) {
	    for ( i = 0; i < inputPortCnt; i++ ) {
	        fuWrapper->getInputPort(i)->loosenTraits();
	    }
	}

	updateConfiguration();
}

void
InterfaceClass::updateConfiguration()
{
	int i, j, in, out, tag;
	/* 
	 * First, take care of FUWrapper things  -- 
	 *    output record
	 *    output port to output field bindings
	 *    user input constant values
	 *    execution mode
	 */
	fuWrapper->resetConfiguration();

	if ( outputPortCnt > 0 ) {
	    Assert( outputFieldCnt > 0 );

	    if ( outputDossier == NULL ) {
		Assert( ! previouslyConfigured );

		if ( ! fuWrapper->setOutputRecord( _addStructureToSystem() ) ) {
		    abort( "Unable to add output record\n" );
		}
	    }

	    for ( i = 0; i < outputFieldCnt; i++ ) {
		if ( outputFieldInfo[i].source == Output_Port ) {
		    fuWrapper->bindOutputPort( outputFieldInfo[i].srcIndex, i );
		}
	    }
	}

	for ( i = 0; i < userInputCnt; i++ ) {
	    in = userInputInfo[i].inputPort;
	    out = userInputInfo[i].outputField;
	    if ( ( in != -1 ) || ( out != -1 ) ) {
		fuWrapper->addUserInput( userInputInfo[i].dataValue, in, out );
	    }
	}

	fuWrapper->setExecutionMode( runMode );

	/* 
	 * Next, take care of Input Pipe Socket things  -- 
	 *    input ports bound
	 *    output fields bound
	 *    actions for each record type
	 */
	int size  = inputPortCnt + outputFieldCnt;	// plenty big
	int *intArray = new int[size];

	for ( i = 0; i < inputPipeCnt; i++ ) {

	    InputPipeSocket *isocket = fuWrapper->getInputSocket( i );
	    isocket->resetConfiguration();

	    in = 0;
	    for ( j = 0; j < inputPortCnt; j++ ) {
		if ( ( inputPortInfo[j].source == Input_Pipe ) &&
		     ( inputPortInfo[j].srcIndex == i ) ) {

		    intArray[in++] = j;
		}
	    }
	    Assert( in == inputPipeInfo[i].inputPortsFilled );
	    isocket->setInputPorts( in, intArray );

	    out = 0;
	    for ( j = 0; j < outputFieldCnt; j++ ) {
		if ( ( outputFieldInfo[j].source == Input_Pipe ) &&
		     ( outputFieldInfo[j].srcIndex == i ) ) {

		    intArray[out++] = j;
		}
	    }
	    Assert( out == inputPipeInfo[i].outputFieldsFilled );
	    isocket->setOutputFields( out, intArray );

	    for ( j = 0; j < inputPipeInfo[i].recordCnt; j++ ) {
		isocket->setActionType( inputPipeInfo[i].record[j]->tag,
					inputPipeInfo[i].record[j]->action );
	    }

	    for ( j = 0; j < fieldBindingCnt; j++ ) {

		if ( fieldBindingInfo[j].pipe == i ) {

		    fieldBindingStruct& FB = fieldBindingInfo[j];
		    tag = inputPipeInfo[i].record[ FB.record ]->tag;

		    if ( FB.inPort != -1 ) {
			isocket->bindFieldToInputPort( tag, FB.field,
			                  inputPortInfo[ FB.inPort ].name );
		    }

		    if ( FB.outField != -1 ) {
			isocket->bindFieldToOutputField( tag, FB.field,
			               outputFieldInfo[ FB.outField ].name );
		    }
		}
	    }
	}
	delete intArray;

	/* 
	 * Finally, if this is a reconfiguration make sure that any values
	 * marked valid on our input ports are still being satified
	 * by the same input pipe/record/field.  If they aren't then change
	 * their status to invalid.
	 */
	if ( previouslyConfigured ) {
	    for ( i = 0; i < inputPortCnt; i++ ) {
		fuWrapper->getInputPort(i)->verifyStatus();
	    }
	}
		

	
}

/**** Methods used first by Screen1 (PipePortBinding) of the Binding Dialog ***/

void
InterfaceClass::bindPipeToInputPort( int pipe, int port )
{
	Assert( ! previouslyConfigured );

	int srcIdx;

	if ( pipe < inputPipeCnt ) {		   // A real input pipe
	    if ( inputPortInfo[port].source == User_Input ) {
		// This used to be a user input value, mark user input
		// info that it is no longer
		int srcIdx = inputPortInfo[port].srcIndex;
		Assert( userInputInfo[srcIdx].inputPort == port );
		userInputInfo[srcIdx].inputPort = -1;
	    }
	    inputPortInfo[port].source = Input_Pipe;
	    inputPortInfo[port].srcIndex = pipe;
	} else {				   // User input
	    if ( inputPortInfo[port].source == User_Input ) {
		// This has always been user input value, verify port
		srcIdx = inputPortInfo[port].srcIndex;
		Assert( userInputInfo[srcIdx].inputPort == port );
	    } else {	
		// This is a first-time user input, use next userInputInfo slot
		srcIdx = _nextUserInputIdx();
		userInputInfo[srcIdx].inputPort = port;
		userInputInfo[srcIdx].outputField = -1;

	        inputPortInfo[port].source = User_Input;
		inputPortInfo[port].srcIndex = srcIdx;
	    }
	}

	inputPipeInfo[pipe].inputPortsFilled++;
}

void
InterfaceClass::clearPipeToInputPortBindings()
{
	Assert( ! previouslyConfigured );

	for ( int i = 0; i < inputPipeCnt + 1; i++ ) {
	     inputPipeInfo[i].inputPortsFilled = 0;
	}
}

const char *
InterfaceClass::getInputPipeName( int pipe ) const
{
	Assert( ( pipe >= 0 ) && ( pipe <= inputPipeCnt ) );
	return inputPipeInfo[pipe].name;
}

const char *
InterfaceClass::getInputPortName( int port ) const
{
	Assert( ( port >= 0 ) && ( port < inputPortCnt ) );
	return inputPortInfo[port].name;
}

int
InterfaceClass::getNumberInputPipes() const
{
	return inputPipeCnt;
}

int
InterfaceClass::getNumberInputPorts() const
{
	return inputPortCnt;
}

int
InterfaceClass::getPipeBoundToInputPort( int port ) const
{
	Assert( port < inputPortCnt );
	if ( inputPortInfo[port].source == User_Input ) {
	    return inputPipeCnt;
	} else if ( inputPortInfo[port].source == Input_Pipe ) {
	    Assert( ( inputPortInfo[port].srcIndex >= 0 ) &&
		    ( inputPortInfo[port].srcIndex < inputPipeCnt ) );
	    return inputPortInfo[port].srcIndex;
	} else if ( inputPortInfo[port].source == Unbound ) {
	    return -1;
	} else {
	    error( "Invalid binding to input port %d -- source is %d\n", 
				  port, inputPortInfo[port].source );
	    return -1;
	}
}


/** Methods used first by Screen2 (OutputSpecification) of the Binding Dialog */

void
InterfaceClass::addOutputField( char *fieldName, int pipe )
{
	Assert( outputDossier == NULL );	// for now
	Assert( ! previouslyConfigured );

	int idx = _nextOutputFieldIdx();
	outputFieldInfo[idx].name = fieldName;

	if ( pipe == -1 ) {			// Output Port
	    outputFieldInfo[idx].source = Output_Port;
	    outputFieldInfo[idx].srcIndex = -1;
	    outputFieldInfo[idx].origIndex = -1;
	    outputFieldInfo[idx].traits.clear();
	    outputFieldInfo[idx].origTraits.clear();
	    outputFieldInfo[idx].bindCnt = 0;
	    outputFieldInfo[idx].origCnt = 0;

	    // look for default bindings from output ports 
	    for ( int i = 0; i < outputPortCnt; i++ ) {
		if ( ( outputPortCnt == 1 ) || 
		     ( outputPortInfo[i].name == outputFieldInfo[idx].name ) ) {
		    outputFieldInfo[idx].srcIndex = i;
		    outputFieldInfo[idx].origIndex = i;
		    outputFieldInfo[idx].traits = 
						outputPortInfo[i].initialTraits;
		    outputFieldInfo[idx].origTraits = 
						outputFieldInfo[idx].traits; 
		    outputFieldInfo[idx].bindCnt = 1;
		    outputFieldInfo[idx].origCnt = 1;
		    break;
		}
	    }

	} else if ( pipe == inputPipeCnt ) {	// User Input    
	    // This is a first-time user input, use next userInputInfo slot
	    int srcIdx = _nextUserInputIdx();
	    userInputInfo[srcIdx].inputPort = -1;
	    userInputInfo[srcIdx].outputField = idx;

	    outputFieldInfo[idx].source = User_Input;
	    outputFieldInfo[idx].srcIndex = srcIdx;
	    outputFieldInfo[idx].traits.clear();
	    outputFieldInfo[idx].origTraits.clear();
	    outputFieldInfo[idx].bindCnt = 0;
	    outputFieldInfo[idx].origCnt = 0;

	    inputPipeInfo[pipe].outputFieldsFilled++;

	} else {				// A real input pipe
	    outputFieldInfo[idx].source = Input_Pipe;
	    outputFieldInfo[idx].srcIndex = pipe;
	    outputFieldInfo[idx].traits.clear();
	    outputFieldInfo[idx].origTraits.clear();
	    outputFieldInfo[idx].bindCnt = 0;
	    outputFieldInfo[idx].origCnt = 0;

	    inputPipeInfo[pipe].outputFieldsFilled++;
	}
}

Boolean_
InterfaceClass::createOutputRecord( char *name )
{
	Assert( ! previouslyConfigured );

	dictionaryLookup->resetQuery();
	if ( dictionaryLookup->nextMatch( name ) ) {
            info( 
		"Record Name is %s already in use - resetting to default.", 
		name );
	    return FAILURE_;
	} else {
	    strcpy( outputRecordName, name );
	    return SUCCESS_;
	}
}

const char *
InterfaceClass::getNameOfOutputFieldBoundToInputPipe( int pipe, 
						      int fieldIdx ) const
{
	int 	cnt = -1;
	Sources src;
	int 	srcIdx;

	for ( int i = 0; i < outputFieldCnt; i++ ) {
	    src = outputFieldInfo[i].source;
	    srcIdx = outputFieldInfo[i].srcIndex;

	    if ( ( src == Output_Port ) && ( pipe == -1 ) ) {
	        cnt++;
	    } else if ( ( src == User_Input ) && ( pipe == inputPipeCnt ) ) {
	        cnt++;
	    } else if ( ( src == Input_Pipe ) && ( pipe == srcIdx ) ) {
	        cnt++;
	    }

	    if ( cnt == fieldIdx ) {
		return( outputFieldInfo[i].name.getValue() );
	    }
	}
	return NULLCHARSTRING;
}

int
InterfaceClass::getNumberInputPortsBoundToInputPipe( int pipe ) const
{
	return inputPipeInfo[pipe].inputPortsFilled;
}

int
InterfaceClass::getNumberOutputFieldsBoundToInputPipe( int pipe ) const
{
	if ( pipe == -1 ) {        // return number bound to output ports 
	    int cnt = 0;
	    for ( int i = 0; i < outputFieldCnt; i++ ) {
		if ( outputFieldInfo[i].source == Output_Port ) {
		    cnt++;
		}
	    }
	    return cnt;
	} else {		   // an input pipe or user input
	    return inputPipeInfo[pipe].outputFieldsFilled;
	}
}
		
const char * 
InterfaceClass::getOutputRecordName() 
{
	return outputRecordName;
}

Boolean_
InterfaceClass::isPipeBound( int pipe ) const
{
  	return( CnvToBoolean_( inputPipeInfo[pipe].inputPortsFilled > 0 ) );
}

void
InterfaceClass::removeAllOutputFields()
{
	Assert( outputDossier == NULL );		// for now...
	Assert( ! previouslyConfigured );

	int i;
	for ( i = 0; i < inputPipeCnt + 1; i++ ) {
	    inputPipeInfo[i].outputFieldsFilled = 0;
	}

	for ( i = 0; i < outputFieldCnt; i++ ) {
	    if ( outputFieldInfo[i].source == User_Input ) {
		// This used to be a user input value, mark user input
		// info that it is no longer
		int srcIdx = outputFieldInfo[i].srcIndex;
		Assert( userInputInfo[srcIdx].outputField == i );
		userInputInfo[srcIdx].outputField = -1;
	    }
	}
	outputFieldCnt = 0;
}

/**** Methods used first by Screen3 (Constants Input) of the Binding Dialog ***/

const char *
InterfaceClass::getInputPortValue( int portNumber ) const
{
	Assert( portNumber < inputPortCnt );
	Assert( inputPortInfo[portNumber].source == User_Input );

	int idx = inputPortInfo[portNumber].srcIndex;
	Assert( userInputInfo[idx].inputPort == portNumber );
	
	return( _showValueAsString( userInputInfo[idx].dataValue ) );
}

int
InterfaceClass::getNumberOutputFields() const
{
	return outputFieldCnt;
}

const char *
InterfaceClass::getOutputFieldName( int fieldNumber ) const
{
	Assert( fieldNumber < outputFieldCnt );
	return( outputFieldInfo[fieldNumber].name.getValue() );
}



const char *
InterfaceClass::getOutputFieldValue( int fieldNumber ) const
{
	Assert( fieldNumber < outputFieldCnt );
	Assert( outputFieldInfo[fieldNumber].source == User_Input );

	int idx = outputFieldInfo[fieldNumber].srcIndex;
	Assert( userInputInfo[idx].outputField == fieldNumber );
	
	return( _showValueAsString( userInputInfo[idx].dataValue ) );
}

Boolean_
InterfaceClass::isInputPortBoundToConstant( int portNumber ) const
{
	if ( inputPortInfo[portNumber].source == User_Input ) {
	    Assert( userInputInfo[inputPortInfo[portNumber].srcIndex].inputPort 
		    == portNumber );
	    return TRUE_;
	} else {
	    return FALSE_;
	}
}

Boolean_
InterfaceClass::isOutputFieldBoundToConstant( int fieldNumber ) const
{
	if ( outputFieldInfo[fieldNumber].source == User_Input ) {
	    Assert( fieldNumber ==
	      userInputInfo[outputFieldInfo[fieldNumber].srcIndex].outputField);
	    return TRUE_;
	} else {
	    return FALSE_;
	}
}

void
InterfaceClass::restoreOriginalConstants()
{
	int i;

	/* Data values in the User Input Info array */
	for ( i = 0; i < userInputCnt; i++ ) {
	    userInputInfo[i].dataValue = *shadowConstants[i];
	}

	/* Input Port traits */
	for ( i = 0; i < inputPortCnt; i++ ) {
	    if ( inputPortInfo[i].source == User_Input ) {
	        InputPort *iport = fuWrapper->getInputPort( i );
	        iport->resetTraits( inputPortInfo[i].initialTraits,
			            inputPortInfo[i].initialConstraintsCnt );
		if ( previouslyConfigured ) {
		    iport->lockTraits();
		}
	    }
	}

	/* OutputField traits and bindCnt  */
	for ( i = 0; i < outputFieldCnt; i++ ) {
	    if ( outputFieldInfo[i].source == User_Input ) {
	        outputFieldInfo[i].traits = outputFieldInfo[i].origTraits;
	        outputFieldInfo[i].bindCnt = outputFieldInfo[i].origCnt;
		if ( previouslyConfigured ) {
		    outputFieldInfo[i].bindCnt++;
		}
	    }
	}
}

const char *
InterfaceClass::setInputPortValue( int portNumber, char *userValue )
{
	Assert( inputPortInfo[portNumber].source == User_Input );

	InputPort *iport = fuWrapper->getInputPort( portNumber );

	if ( !previouslyConfigured ) {	
	    // If this is the first time we"re configuring this module,
	    // then don"t constrain traits on input port by a previous
	    // constant value that may have been entered... 
	    iport->clearTraits();
	}
	DataTraits traits = iport->getTraits();

	Value value;
	_setValueFromString( userValue, value, traits );

	if ( iport->constrainTraits( value.getTraits() )  == FAILURE_ ) {
            CString mesg;
            mesg = form( "Value %s cannot bind to port %s\n",
				  userValue,
                                  inputPortInfo[portNumber].name );
            mesg = mesg + form( "Value: type = %s dimension = %d\n\n",
				 value.getTraits().getTypeName(),
                                 value.getTraits().getDimension() );

            mesg = mesg + "Allowed traits:\n";
	    iport->concatValidTraits( mesg );
            info( mesg.getValue() );
	
	    return NULLCHARSTRING;
	} 

	if ( previouslyConfigured ) {
	    // If we don"t do this and keep changing our input constant, we"ll
	    // get some outrageous number of bindings.  Don"t worry about
	    // resetting for !previouslyConfigured since we clear above.
	    iport->resetTraits( value.getTraits(), 2 );
	}

	int userIndex = inputPortInfo[portNumber].srcIndex;
	Assert( userInputInfo[userIndex].inputPort == portNumber );

	userInputInfo[userIndex].dataValue = value;
	return( _showValueAsString( value ) );
}

const char *
InterfaceClass::setOutputFieldValue( int fieldNumber, char *userValue )
{
	Assert( outputFieldInfo[fieldNumber].source == User_Input );

	if ( !previouslyConfigured ) {
	    // If this is the first time we"re configuring this module,
	    // then don"t constrain traits on output field by a previous
	    // constant value that may have been entered... 
	    outputFieldInfo[fieldNumber].traits.clear();
	    outputFieldInfo[fieldNumber].bindCnt = 0;
	}

	DataTraits traits = outputFieldInfo[fieldNumber].traits;

	Value value;
	_setValueFromString( userValue, value, traits );

        if ( outputFieldInfo[fieldNumber].bindCnt > 0 ) {
	    if ( outputFieldInfo[fieldNumber].traits != value.getTraits() ) {
                CString mesg;
                mesg = form( "Value %s cannot bind to output field %s\n",
                                userValue,
                                outputFieldInfo[fieldNumber].name.getValue() );
                mesg = mesg + form( "Value:   Type = %s;  Dimension = %d;\n",
                                value.getTraits().getTypeName(),
                                value.getTraits().getDimension() );
                mesg = mesg + form( "Required:   Type = %s;  Dimension = %d;\n",
                  typeNameArray[ outputFieldInfo[fieldNumber].traits.getType()],
                  outputFieldInfo[fieldNumber].traits.getDimension() );
                info( mesg.getValue() );
	
	        return NULLCHARSTRING;
	    }
	} else {
	    outputFieldInfo[fieldNumber].traits = value.getTraits();
	}

	if ( previouslyConfigured ) {
	    outputFieldInfo[fieldNumber].bindCnt = 2;
	} else {
	    outputFieldInfo[fieldNumber].bindCnt = 1;
	}

   	int userIndex = outputFieldInfo[fieldNumber].srcIndex;
	Assert( userInputInfo[userIndex].outputField == fieldNumber );

	userInputInfo[userIndex].dataValue = value;
	return( _showValueAsString( value ) );
}


/**** Methods used first by Screen4 (Record Bindings) of the Binding Dialog ***/

Boolean_
InterfaceClass::allOutputPortTraitsSet()
{
	Boolean_ allSet = TRUE_;

	for ( int i = 0; i < outputPortCnt; i++ ) {
	    allSet = ( CnvToBoolean_( allSet &&
		       fuWrapper->getOutputPort(i)->getTraits().isDefined() ) );
	}
	return allSet;
}

const char *
InterfaceClass::getNameOfFieldInInputPipeRecord( int pipe, int record, 
							   int field ) const
{
	RecordDossier *dossier = inputPipeInfo[pipe].record[record]->dossier;
	return( dossier->getField(field)->getName().getValue() );
}

const char *
InterfaceClass::getNameOfInputPortBoundToInputPipe( int pipe, int port ) const
{
	int     cnt = -1;
	Sources src;
	int     srcIdx;

	for ( int i = 0; i < inputPortCnt; i++ ) {
	    src = inputPortInfo[i].source;
	    srcIdx = inputPortInfo[i].srcIndex;

	    if ( ( src == User_Input) && ( pipe == inputPipeCnt ) ) {
	        cnt++;
	    } else if ( ( src == Input_Pipe ) && ( pipe == srcIdx ) ) {
	        cnt++;
	    }

	    if ( cnt == port ) {
		return( inputPortInfo[i].name );
	    }
	}
	return NULLCHARSTRING;
}

const char *
InterfaceClass::getNameOfRecordOnInputPipe( int pipe, int record ) const
{
	RecordDossier *dossier = inputPipeInfo[pipe].record[record]->dossier;
	return dossier->getName().getValue();
}

int
InterfaceClass::getNumberFieldsInInputPipeRecord( int pipe, int record ) const
{
	return inputPipeInfo[pipe].record[record]->dossier->entryCount();
}

int
InterfaceClass::getNumberOutputPorts() const
{
	return outputPortCnt;
}

int
InterfaceClass::getNumberRecordsOnInputPipe( int pipe ) const
{
	return inputPipeInfo[pipe].recordCnt;
}

const char *
InterfaceClass::getOutputPortName( int oport ) const
{
	return outputPortInfo[oport].name;
}

ActionType
InterfaceClass::getStatusOfRecordOnInputPipe( int pipe, int record ) const
{
	return inputPipeInfo[pipe].record[record]->action;
}

Boolean_
InterfaceClass::isInputRecordFieldBoundToInputPort( int pipe, int record,
						    int field, int port ) const
{
	int realPort = _getRealInputPortNum( pipe, port );
	if ( realPort == -1 ) {
	    return FALSE_;
	}

	for ( int i = 0; i < fieldBindingCnt; i++ ) {
	    const fieldBindingStruct& fbinding = fieldBindingInfo[i];
	    if ( ( fbinding.pipe == pipe ) 
		 && ( fbinding.record == record ) 
		 && ( fbinding.field == field ) 
		 && ( fbinding.inPort == realPort ) ) {
		return TRUE_;
	    }
	}
	return FALSE_;
}

Boolean_
InterfaceClass::isInputRecordFieldBoundToOutputField( int pipe, int record,
						  int field, int ofield ) const
{
	int realOfield = _getRealOutputFieldNum( pipe, ofield );
	if ( realOfield == -1 ) {
	    return FALSE_;
	}

	for ( int i = 0; i < fieldBindingCnt; i++ ) {
	    const fieldBindingStruct& fbinding = fieldBindingInfo[i];
	    if ( ( fbinding.pipe == pipe ) 
		 && ( fbinding.record == record ) 
		 && ( fbinding.field == field ) 
		 && ( fbinding.outField == realOfield ) ) {
		return TRUE_;
	    }
	}
	return FALSE_;
}

Boolean_
InterfaceClass::isOutputPortBoundToOutputField( int oport, int ofield ) const
{
	int realOfield = _getRealOutputFieldNum( -1, ofield );
	if ( realOfield == -1 ) {
	    return FALSE_;
	}

	if ( outputFieldInfo[realOfield].srcIndex == oport ) {
	    return TRUE_;
	} else {
	    return FALSE_;
	}
}

Boolean_
InterfaceClass::linkInputRecordFieldToInputPort( int pipe, int record,
						 int field, int port )
{
	int realPort = _getRealInputPortNum( pipe, port );
	if ( realPort == -1 ) {
	    return FALSE_;
	}

	InputPort *iport = fuWrapper->getInputPort( realPort );
	FieldDescriptor *fieldD = 
		inputPipeInfo[pipe].record[record]->dossier->getField(field);

	if ( iport->constrainTraits( fieldD->getTraits() )  == FAILURE_ ) {
            CString mesg;
            mesg = form( "Field %s cannot bind to port %s\n",
                                  fieldD->getName().getValue(),
                                  inputPortInfo[realPort].name );
            mesg = mesg + form( "Field: type = %s dimension = %d\n\n",
                                 typeNameArray[ fieldD->getTraits().getType() ],
                                 fieldD->getTraits().getDimension() );

            mesg = mesg + "Allowed traits:\n";
	    iport->concatValidTraits( mesg );
            info( mesg.getValue() );
	
	    return FALSE_;
	} else {
	    if ( iport->getConstraintsCnt() == 1 ) {  // perhaps last one!
		if ( ! _inputPortSetValid() ) {
		    iport->loosenTraits();
		    return FALSE_;
		}
	    }
	    int f = _nextFieldBindingIdx();
	    fieldBindingInfo[f].pipe = pipe;
	    fieldBindingInfo[f].record = record;
	    fieldBindingInfo[f].field = field;
	    fieldBindingInfo[f].inPort = realPort;
	    fieldBindingInfo[f].outField = -1;

	    /* Update Config board; Arguments are input port number, 	*
	     * input record name, and input field name.			*/
	    configBoard->fieldExtractPipePort( realPort,
	      inputPipeInfo[pipe].record[record]->dossier->getName().getValue(),
	      fieldD->getName().getValue() );

	    return TRUE_;
        }
}

Boolean_
InterfaceClass::linkInputRecordFieldToOutputField( int pipe, int record,
						   int field, int ofield )
{
	int o = _getRealOutputFieldNum( pipe, ofield );
	if ( o == -1 ) {
	    return FALSE_;
	}

	FieldDescriptor *fieldD = 
		inputPipeInfo[pipe].record[record]->dossier->getField(field);

        if ( outputFieldInfo[o].bindCnt > 0 ) {
	    if ( outputFieldInfo[o].traits != fieldD->getTraits() ) {
                CString mesg;
                mesg = form( "Field %s cannot bind to output field %s\n",
                                  fieldD->getName().getValue(),
                                  outputFieldInfo[o].name.getValue() );
                mesg = mesg + form( "Field: type = %s dimension = %d\n\n",
                                 typeNameArray[ fieldD->getTraits().getType() ],
                                 fieldD->getTraits().getDimension() );
                mesg = mesg + form( "Required: type = %s dimension = %d\n",
                           typeNameArray[ outputFieldInfo[o].traits.getType() ],
                           outputFieldInfo[o].traits.getDimension() );
                info( mesg.getValue() );
	
	        return FALSE_;
	    } else {
		outputFieldInfo[o].bindCnt++;
	    }
	} else {
	    outputFieldInfo[o].bindCnt = 1;
	    outputFieldInfo[o].traits = fieldD->getTraits();
	}

	int f = _nextFieldBindingIdx();
	fieldBindingInfo[f].pipe = pipe;
	fieldBindingInfo[f].record = record;
	fieldBindingInfo[f].field = field;
	fieldBindingInfo[f].inPort = -1;
	fieldBindingInfo[f].outField = o;

	/* Update Config board; Arguments are output field name, 	*
	 * input record name, and input field name.			*/
	configBoard->fieldExtractPipeField( outputFieldInfo[o].name.getValue(),
	    inputPipeInfo[pipe].record[record]->dossier->getName().getValue(),
	    fieldD->getName().getValue() );

	return TRUE_;
}

Boolean_
InterfaceClass::linkOutputPortToOutputField( int oport, int ofield )
{
	int o = _getRealOutputFieldNum( -1, ofield );
	if ( o == -1 ) {
	    return FALSE_;
	}

	OutputPort *oPort = fuWrapper->getOutputPort( oport );

	if ( outputFieldInfo[o].bindCnt > 0 ) {
	    if ( outputFieldInfo[o].traits != oPort->getTraits() ) {
                CString mesg;
                mesg = form( "Port %s cannot bind to output field %s\n",
                                  outputPortInfo[oport].name, 
				  outputFieldInfo[o].name.getValue() );
                mesg = mesg + form( "Port: type = %s dimension = %d\n\n",
                                 typeNameArray[ oPort->getTraits().getType() ],
                                 oPort->getTraits().getDimension() );
                mesg = mesg + form( "Required: type = %s dimension = %d\n",
                           typeNameArray[ outputFieldInfo[o].traits.getType() ],
                           outputFieldInfo[o].traits.getDimension() );
                info( mesg.getValue() );
	
	        return FALSE_;
	    } 
	} 
	outputFieldInfo[o].bindCnt++;
	outputFieldInfo[o].traits = oPort->getTraits();
	outputFieldInfo[o].srcIndex = oport;

	configBoard->addOutputPortFieldBinding( oport, 
					   outputFieldInfo[o].name.getValue() );
	return TRUE_;
}

void
InterfaceClass::restoreOriginalFieldBindings()
{
	/*
	 * Notice that we don't restore information for Input Ports and
	 * Output Fields that are bound to User Inputs - those are set
	 * prior to this screen.
	 */
	int i, j, p, f;

	/* Input Port traits */
	for ( i = 0; i < inputPortCnt; i++ ) {
	    if ( inputPortInfo[i].source != User_Input ) {
	        InputPort *iport = fuWrapper->getInputPort( i );
	        iport->resetTraits( inputPortInfo[i].initialTraits,
			            inputPortInfo[i].initialConstraintsCnt );
		if ( previouslyConfigured ) {
		    iport->lockTraits();
		}
	    }
	}

	/* Output Port traits */
	for ( i = 0; i < outputPortCnt; i++ ) {
	    OutputPort *oport = fuWrapper->getOutputPort( i );
	    oport->setTraits( outputPortInfo[i].initialTraits );
	}

	/* Record Action and Field Binding Table */
	fieldBindingCnt = 0;

	for ( i = 0; i < inputPipeCnt; i++ ) {
	    inputPipeStruct& pipeInfo = inputPipeInfo[i];
	    InputPipeSocket *isocket = fuWrapper->getInputSocket( i );

	    for ( j = 0; j < pipeInfo.recordCnt; j++ ) {
	        pipeInfo.record[j]->action =
			          isocket->getAction( pipeInfo.record[j]->tag );
		
		if ( ActionDoesExtraction( pipeInfo.record[j]->action ) ) {
		    for ( p = 0; p < pipeInfo.inputPortsFilled; p++ ) {
		        f = _nextFieldBindingIdx();
			fieldBindingInfo[f].pipe = i;
			fieldBindingInfo[f].record = j;
			fieldBindingInfo[f].field = 
						isocket->getFieldToInputPort( 
			  		           pipeInfo.record[j]->tag, p );
			fieldBindingInfo[f].inPort = isocket->getInputPort( p );
			fieldBindingInfo[f].outField = -1;
		    }
		    for ( p = 0; p < pipeInfo.outputFieldsFilled; p++ ) {
		        f = _nextFieldBindingIdx();
			fieldBindingInfo[f].pipe = i;
			fieldBindingInfo[f].record = j;
			fieldBindingInfo[f].field = 
				                isocket->getFieldToOutputField( 
			  		           pipeInfo.record[j]->tag, p );
			fieldBindingInfo[f].inPort = -1;
			fieldBindingInfo[f].outField = 
					        isocket->getOutputField( p );
		    }
		}
	    }
	 }

	/* OutputField traits, bindCnt and srcIndex (if bound to Output_Port) */
	for ( i = 0; i < outputFieldCnt; i++ ) {
	    if ( outputFieldInfo[i].source != User_Input ) {
	        outputFieldInfo[i].traits = outputFieldInfo[i].origTraits;
	        outputFieldInfo[i].bindCnt = outputFieldInfo[i].origCnt;
		if ( previouslyConfigured ) {
		    outputFieldInfo[i].bindCnt++;
		}
	        if ( outputFieldInfo[i].source == Output_Port ) {
		    outputFieldInfo[i].srcIndex = outputFieldInfo[i].origIndex;
	        } 
	    }
	}

	_showModuleResultBindings();
	_showInputPortBindings();
}

void
InterfaceClass::setStatusOfRecordOnInputPipe( int pipe, int record, 
							ActionType status )
{
	inputPipeInfo[pipe].record[record]->action = status;
}

void
InterfaceClass::unlinkInputRecordFieldFromInputPort( int pipe, int record,
						     int field, int port )
{
	int realPort = _getRealInputPortNum( pipe, port );
	if ( realPort == -1 ) {
	    return;
	}

	/* Field Binding Information */
	for ( int f = 0; f < fieldBindingCnt; f++ ) {
	    if ( ( fieldBindingInfo[f].pipe == pipe )
	         && ( fieldBindingInfo[f].record == record )
	         && ( fieldBindingInfo[f].field == field )
	         && ( fieldBindingInfo[f].inPort == realPort ) ) {
		fieldBindingInfo[f].inPort = -1;
	    }
	}

	/* Update Config board; Arguments are input port number,         *
	 * input record name, and input field name.			 */
	configBoard->clearSinglePipePortExtract( realPort,
	    inputPipeInfo[pipe].record[record]->dossier->getName().getValue() );

	/* Input Port Constraints */
	InputPort *iport = fuWrapper->getInputPort( realPort );
	iport->loosenTraits(); 

	/* Output Port and Output Field traits may have been constrained  *
	 * by this binding 						  */
	if ( iport->getTraits().isUndefined() ) {
	    int i, o;

	    for ( i = 0; i < outputPortCnt; i++ ) {
	        OutputPort *oport = fuWrapper->getOutputPort( i );
	        oport->setTraits( outputPortInfo[i].initialTraits );
	    }

	    for ( i = 0; i < outputFieldCnt; i++ ) {
		if ( outputFieldInfo[i].source == Output_Port ) {
		    if ( outputFieldInfo[i].bindCnt > 0 ) {
		        o = outputFieldInfo[i].srcIndex;
			Assert( ( 0 <= o )  && ( o < outputPortCnt ) );
			outputFieldInfo[i].traits = 
				fuWrapper->getOutputPort(o)->getTraits();
		    }
		}
	    }

	}

}

void
InterfaceClass::unlinkInputRecordFieldFromOutputField( int pipe, int record,
						       int field, int ofield )
{
	int realField = _getRealOutputFieldNum( pipe, ofield );
	if ( realField == -1 ) {
	    return;
	}

	/* Field Binding Information */
	for ( int f = 0; f < fieldBindingCnt; f++ ) {
	    if ( ( fieldBindingInfo[f].pipe == pipe )
	         && ( fieldBindingInfo[f].record == record )
	         && ( fieldBindingInfo[f].field == field )
	         && ( fieldBindingInfo[f].outField == realField ) ) {
		fieldBindingInfo[f].outField = -1;
	    }
	}

	/* Output Field Constraints */
	outputFieldInfo[realField].bindCnt--;

	/* Update Config board; Arguments are output field name, *
	 * input record name, and input field name.	         */
	configBoard->clearSinglePipeOutputExtract( 
	    outputFieldInfo[realField].name.getValue(),
	    inputPipeInfo[pipe].record[record]->dossier->getName().getValue() );
}

void
InterfaceClass::unlinkOutputPortFromOutputField( int oport, int ofield )
{
	int o = _getRealOutputFieldNum( -1, ofield );
	if ( o == -1 ) {
	    return;
	}

	outputFieldInfo[o].bindCnt--;
	configBoard->clearSingleOutputPortFieldBinding( oport, 
					   outputFieldInfo[o].name.getValue() );
}

/***** Methods used first by Screen5 (Execution mode) of the Binding Dialog ***/

ExecutionMode_
InterfaceClass::getModuleExecutionMode() const
{
	return runMode;
}

void
InterfaceClass::setModuleExecutionMode( ExecutionMode_ mode ) 
{
	runMode = mode;
}

/********* Methods used by the Scoreboard ***********************************/

const char *
InterfaceClass::getTraitsForInputPort( int portNumber ) const
{
	static char buf[2048];
	CString cstr;

	InputPort *iport = fuWrapper->getInputPort( portNumber );
	iport->concatValidTraits( cstr );
	strcpy( buf, cstr.getValue() );

	return buf;
}

const char*
InterfaceClass::getTraitsForOutputPort( int portNumber ) const
{
	static char buf[1024];

	const DataTraits& dt = 
			     fuWrapper->getOutputPort(portNumber)->getTraits();

	if ( dt.getDimension() == -1 ) {
	    sprintf( buf, "   Type = %s;  Dimension = ANY;\n", 
			      typeNameArray[ dt.getType() ] );
	} else {
	    sprintf( buf, "   Type = %s;  Dimension = %d;\n", 
			      typeNameArray[ dt.getType() ] ,
			      dt.getDimension() );
        }
	return buf;
}

const char*
InterfaceClass::getTraitsForOutputField( const char* fieldName ) const
{
	static char buf[1024];

	for ( int i = 0; i < outputFieldCnt; i++ ) {
	    if ( fieldName == outputFieldInfo[i].name ) {
		break;
	    }
	}

	if ( i < outputFieldCnt ) {
	    const DataTraits& dt = outputFieldInfo[i].traits;

	    if (  dt.getDimension() == -1 ) {
	        sprintf( buf, "   Type = %s;  Dimension = ANY;\n", 
	    		          typeNameArray[ dt.getType() ] );
	    } else {
	        sprintf( buf, "   Type = %s;  Dimension = %d;\n", 
			          typeNameArray[ dt.getType() ] ,
			          dt.getDimension() );
            }
	} else {
	    sprintf( buf, "Type information not set for field `%s'\n", 
				  fieldName );
	}
	return buf;
}


/*
 * Initialize the static data
 */
const char *const InterfaceClass::MY_CLASS = "InterfaceClass";
