/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author:  Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * LedFormWrapper.cc:  Class wrapper for the LED performance widgets
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/Wrappers/RCS/LedFormWrapper.C,v 1.17 1994/02/25 04:44:33 aydt Exp $
 */

#include "LedFormWrapper.h"

extern "C" {
#include "Led.h"
}


#include <Xm/Form.h>	// Form widget class name definition
#include "XArgs.h"	// defines the X widget argument structure type

#define ADJ_LED_WIDTH_VEC 0.50
#define ADJ_LED_HEIGHT_VEC 0.50
#define ADJ_LED_WIDTH_ARY 0.20
#define ADJ_LED_HEIGHT_ARY 0.20

#define max(a,b)        ( (a) > (b) ? (a):(b) )

LedFormWrapper::LedFormWrapper( WidgetBase *par, LedFU *ledFUPtr, 
				const XArgs& args, const CString& name ) 
	       : PerfWidget( par )
{
	_setClassName( MY_CLASS );
   	_setType( "LedFormWrapper" );
   	ledFUthisPtr = ledFUPtr;

   	// Create the Motif Form widget
   	widget = XtCreateManagedWidget( name.getValue(), xmFormWidgetClass, 
				        par->getWidget(), args.getArgs(), 
				        args.getNArgs() );
   	Arg arg;		
   	XtSetArg( arg, XmNfractionBase, (XtArgVal)10000 );
   	XtSetValues( widget, &arg, 1 );

	// Set initial number of columns and rows
   	formCols = 1;  
   	formRows = 1;

   	// Create a single LED widget to be the initial child of the Form
   	childList = new Widget[1];
   	childList[0] = XtCreateManagedWidget( "Led:0,0", ledWidgetClass, 
					      widget, NULL, 0 );
   	labels.createMinMaxLabel( widget, "LedMinMaxLabel" );
	setLedLabel( L_LEFT_VERT, NULLCHARSTRING );
	setLedLabel( L_BOTTOM_HORIZ, NULLCHARSTRING );

   	addCallback( childList[0], XtNselect, &Callback::callback1, 
						this, NULL );
	setPerfWidgetColors();

}                       

LedFormWrapper::~LedFormWrapper()
{
	XtDestroyWidget( widget );
	delete [] childList;
}

void 
LedFormWrapper::_adjustLayout( int nCols, int nRows )
{
       	// Unmanage all the child widgets before resize
       	XtUnmanageChildren( childList, (formRows * formCols) );

	// The plan here is to determine the current height of "any" led
        // (we use the first led as our canonical led), then set the new
        // width and height (again for a single led).

	Dimension ledWidth, ledHeight;
	Dimension oldSingleLedWidth, oldSingleLedHeight;
	XtVaGetValues( childList[0],
		       XmNwidth, &oldSingleLedWidth,
		       XmNheight, &oldSingleLedHeight,
		          NULL );
	ledWidth = (oldSingleLedWidth*formCols) / nCols;
        ledHeight = (oldSingleLedHeight*formRows) / nRows;

       	_genChildLayout( nCols, nRows );
	setLedAttr( XmNwidth, ledWidth );
	setLedAttr( XmNheight, ledHeight );

        // Make visible all the new repositioned stuff 
	XtManageChildren( childList, (nRows * nCols) );
        XFlush( XtDisplay( widget ) );
}        

Widget 
LedFormWrapper::_createChild( char *newWidgetName, int iCol, int iRow,  
                         int ledLayoutXOrg, int ledLayoutYOrg,
                         int ledWidthProportion, int ledHeightProportion )
{
   	int i = 0;
   	Arg arglist[4];

   	// Attach the widget to the Form for RESIZING
   	XtArgVal leftPos = ledLayoutXOrg + iCol * ledWidthProportion;
   	XtArgVal rightPos = leftPos + ledWidthProportion;
   	XtArgVal topPos =  ledLayoutYOrg + iRow * ledHeightProportion;
   	XtArgVal bottomPos = topPos +  ledHeightProportion;
   	XtSetArg( arglist[i], XmNleftPosition, leftPos ); i++;
   	XtSetArg( arglist[i], XmNrightPosition, rightPos ); i++;
   	XtSetArg( arglist[i], XmNtopPosition, topPos ); i++;
   	XtSetArg( arglist[i], XmNbottomPosition, bottomPos ); i++;

   	Widget newLed = XtCreateWidget( newWidgetName, ledWidgetClass, 
					widget, arglist, i );
   	addCallback( newLed, XtNselect, &Callback::callback1, this, NULL );
	setPerfWidgetColors( newLed );
   	return( newLed );
}

void 
LedFormWrapper::_findLedsFormPosition( int nColElements, int nRowElements, 
			    int *ledLayoutXOrg, int *ledLayoutYOrg,
		            int *ledWidthProportion, int *ledHeightProportion )
{

  	int iFractionBase, lLeft, widthP, lBottom, lTop, heightP;
  	getLedFormAttr( XmNfractionBase, (XtArgVal) &iFractionBase );

  	if ( labels.isALabel( L_LEFT_VERT ) ) {
      	    lLeft = labels.getLabelHeight( L_LEFT_VERT );
      	    widthP = (int)( ( iFractionBase - (2 * lLeft) ) / nColElements );
  	} else if ( ( nRowElements > 1 ) && ( formRows == 1 ) ) {
            // If resizing to an array, the labelling layout is fixed 
      	    lLeft = labels.getLabelHeight( L_LEFT_VERT );
            widthP = (int)( ( iFractionBase - (2 * lLeft) ) / nColElements );
   	} else {
            lLeft = 0;
      	    widthP = (int)( iFractionBase/nColElements );
	}
	*ledLayoutXOrg = lLeft;
	*ledWidthProportion = widthP;

  	if ( labels.isALabel( L_BOTTOM_HORIZ ) ) {
	    lBottom = labels.getLabelHeight( L_BOTTOM_HORIZ );
	    lTop = labels.getLabelHeight( L_TOP_HORIZ );
	    heightP = (int)( (iFractionBase - lBottom - lTop) / nRowElements );
  	} else if ( ( nRowElements > 1 ) && ( formRows == 1 ) ) {
	    lBottom = labels.getLabelHeight( L_BOTTOM_HORIZ );
	    lTop = labels.getLabelHeight( L_TOP_HORIZ );
	    heightP = (int)( (iFractionBase - lBottom - lTop) / nRowElements );
  	} else {
	    lTop = labels.getLabelHeight( L_TOP_HORIZ );
	    heightP = (int)( (iFractionBase - lTop) / nRowElements );
  	} 
  	*ledLayoutYOrg = labels.getLabelHeight( L_TOP_HORIZ );
	*ledHeightProportion = heightP;
}

void 
LedFormWrapper::_genChildLayout( int nNewCols, int nNewRows )
{
  	// Calculate the new size and create a new WidgetList to hold children
  	int newSize = nNewRows * nNewCols;
	WidgetList newChildList = new Widget[ newSize ];

	// Figure out where leds should be in new display
  	int ledLayoutXOrg, ledLayoutYOrg;
	int ledWidthProportion, ledHeightProportion;
  	_findLedsFormPosition( nNewCols, nNewRows, 
		               &ledLayoutXOrg, &ledLayoutYOrg, 
		               &ledWidthProportion, &ledHeightProportion ); 

	// Now, for every element in the new and/or old display...
	int maxRow = max( formRows, nNewRows );
	int maxCol = max( formCols, nNewCols );

  	int iRow, iCol;
  	int iEleNew, iEleOld;
  	char newWidgetName[MAX_WIDGET_NAME_LEN];

    	for ( iRow = 0; iRow < maxRow; iRow++ ) {
      	    for ( iCol = 0; iCol< maxCol; iCol++) { 

		// Find new and old positons ( Note: both are not always valid )
       		iEleNew = iRow * nNewCols + iCol;	
       		iEleOld = iRow * formCols + iCol;

        	if ( (iRow < nNewRows) && (iCol < nNewCols) ) {
		    //  This (iRow, iCol) is in New Display
           	    if ( (iRow < formRows) && (iCol < formCols)){
           	        // Also in Old Display - reuse the Widget.
               		_setALedFormPosition( iCol, iRow,
                                 ledLayoutXOrg, ledLayoutYOrg,
				 ledWidthProportion, ledHeightProportion );
			newChildList[ iEleNew ] = childList[ iEleOld ];
           	    } else {
	       		// Not in Old Display - Create new Led Widget
               		sprintf( newWidgetName, "Led:%d,%d", iRow, iCol ); 
			newChildList [ iEleNew ] = _createChild( newWidgetName, 
						     iCol, iRow, ledLayoutXOrg,
						     ledLayoutYOrg, 
						     ledWidthProportion, 
						     ledHeightProportion );
           	    }
        	} else {  
		    // This (iRow, iCol) is not in New Display
		    if ( (iRow < formRows) && (iCol < formCols) ) {
		        // This (iRow, iCol) in Old Display; Destroy the Widget
   	   	        XtDestroyWidget( childList[ iEleOld ] );
		    }
        	}
      	    }
    	}

  	// The new list is now complete. Clear out the old space & adjust ptrs
  	delete [] childList; 
	childList = newChildList;
  	formRows = nNewRows;
  	formCols = nNewCols;
}


void 
LedFormWrapper::_setALedFormPosition( int iCol, int iRow, 
			      int ledLayoutXOrg, int ledLayoutYOrg,
                              int ledWidthProportion, int ledHeightProportion )
{
     	Arg arglist[4];
     	XtSetArg( arglist[0], XmNleftPosition,
	          (XtArgVal)( ledLayoutXOrg + iCol*ledWidthProportion ) ); 
     	XtSetArg( arglist[1], XmNrightPosition,
		 (XtArgVal)( ledLayoutXOrg + (iCol+1)*ledWidthProportion ) );
     	XtSetArg( arglist[2], XmNtopPosition, 
		  (XtArgVal)( ledLayoutYOrg + iRow*ledHeightProportion ) );
     	XtSetArg( arglist[3], XmNbottomPosition, 
	          (XtArgVal)( ledLayoutYOrg + (iRow+1)*ledHeightProportion ) );

     	int iEle = iRow * formCols + iCol;
     	XtSetValues( childList[iEle], arglist, 4 );
}

void 				/* virtual */
LedFormWrapper::callback1( Widget callbackWidget, XtPointer /* ptr1 */, 
				XtPointer /* ptr2 */ )
{
    	int iRow, iCol;
    	sscanf( XtName(callbackWidget), "Led:%d,%d", &iRow, &iCol );
    	if ( ( iRow >=0 ) && ( iCol >=0) ) {
       	    ledFUthisPtr->fuCallback( iRow, iCol ); 
    	}
}

void
LedFormWrapper::getLedAttr( String name, XtArgVal value ) const
{
   	Arg arg;
   	XtSetArg( arg, name, value );
   	XtGetValues( childList[0], &arg, 1 );
}

void 
LedFormWrapper::getLedFormAttr( String name, XtArgVal value ) const
{
  	Arg arg;
  	XtSetArg( arg, name, value );
  	XtGetValues( widget, &arg, 1 );
}

void 
LedFormWrapper::getLedLabel( int whichLabel, 
				       const char **aLabel ) const
{
     	labels.getLabel( whichLabel, aLabel );
} 

void
LedFormWrapper::setDisplayValues( int value ) 
{
	Assert( (formRows == 1) && (formCols == 1) );

   	LedSetValue( (LedWidget)childList[0], value );
}

void
LedFormWrapper::setDisplayValues( int nCols, int *values ) 
{
	if ( ( nCols != formCols ) || ( formRows != 1 ) ) {
	    _adjustLayout( nCols, 1 );
	}

	int i;
   	for ( i = 0; i < nCols; i++ ) {
     	    LedSetValue( (LedWidget)childList[i], values[i] );
   	}
}

void
LedFormWrapper::setDisplayValues( int nCols, int nRows, int *values ) 
{
	if ( ( nCols != formCols ) || ( nRows != formRows ) ) {
	    _adjustLayout( nCols, nRows );
	}

	int nLeds = nCols * nRows;
	int i;
      	for ( i = 0; i < nLeds; i++ ) {
            LedSetValue( (LedWidget)childList[i], values[i] );
        }
}

void
LedFormWrapper::setLedAttr( String name, XtArgVal value )
{
	Arg arg;
	XtSetArg( arg, name, value );

	for ( int i = 0; i < (formCols * formRows); i++ ) {
       	    XtSetValues( childList[i], &arg, 1 );
	}
}

void 
LedFormWrapper::setLedFormAttr( String name, XtArgVal value )
{
  	Arg arg;
  	XtSetArg( arg, name, value );
  	XtSetValues( widget, &arg, 1 );
}

void 
LedFormWrapper::setLedLabel( double fuMin, double fuMax )
{
     labels.setLabel( fuMin, fuMax );
}

void 
LedFormWrapper::setLedLabel( int whichLabel, const char *stringLabel )
{
     	Boolean_ layoutChange; 
     	if ( whichLabel == L_LEFT_VERT ) {
            layoutChange = labels.setLabel( widget, "LedLabelVert",
					    whichLabel, stringLabel );
     	} else if ( whichLabel == L_BOTTOM_HORIZ ) {
            layoutChange = labels.setLabel( widget, "LedLabelHoriz",
					    whichLabel, stringLabel );
     	}

     	if ( layoutChange && ( formRows == 1 ) ) {
            int ledLayoutXOrg, ledLayoutYOrg;
            int ledWidthProportion, ledHeightProportion;
            _findLedsFormPosition( formCols, formRows,
     		                   &ledLayoutXOrg, &ledLayoutYOrg,
			           &ledWidthProportion, &ledHeightProportion );
            for ( int iCol=0; iCol < formCols; iCol++ ) {
                for ( int iRow=0; iRow < formRows; iRow++ ) {
                    _setALedFormPosition( iCol, iRow, ledLayoutXOrg, 
					  ledLayoutYOrg, ledWidthProportion, 
					  ledHeightProportion );
	        }
            }
      	}
}

void
LedFormWrapper::setPerfWidgetColors()
{
        colorTable.gencolor( getDisplay(), getScreen() );
        LedSetColors( (LedWidget)childList[0], colorTable.getNumColors(),
                      colorTable.getColormapEntries() );
}

void 
LedFormWrapper::setPerfWidgetColors( Widget led)
{
        LedSetColors( (LedWidget)led, colorTable.getNumColors(),
                      colorTable.getColormapEntries() );
}

void 
LedFormWrapper::setPerfWidgetPixmaps( int pixmapCount, Pixmap* pixmaps )
{
        LedSetPixmaps( (LedWidget)childList[0], pixmapCount, pixmaps );
}

void
LedFormWrapper::setSingleValue( int iCol, int iRow, int value )
{
	if ( ( iCol >= formCols ) || ( iRow >= formRows ) ) {
	    warning( "setSingleValue: Index (%d, %d) out of range: (%d, %d)",
		      iCol, iRow, formCols-1, formRows-1 );
	    return;
	}

       	int iEle = ( iRow * formCols ) + iCol;
        LedSetValue( (LedWidget)childList[iEle], value );
}
	
void 
LedFormWrapper::updateDisplayLayout( int nCols, int nRows )
{
    	if ( ( nCols != formCols ) || ( nRows != formRows ) ) {
	    _adjustLayout( nCols, nRows );
	}
}        

/*
 *      Initialize the static data.   Only executed once.
 */
const char *const LedFormWrapper::MY_CLASS = "LedFormWrapper";
