/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Allen D. Malony (malony@uicsrd.csrd.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, and by a grant
 * from the Digital Equipment Corporation External Research Program.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/bargraph/RCS/Bargraph.c,v 1.10 1994/02/25 04:45:18 aydt Exp $
 */
#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/Xresource.h>
#include <X11/StringDefs.h>
#include "BargraphP.h"

#define Offset(field) XtOffset(BargraphWidget, field)
#define Norm(w, v)	(((float)v-(float)w->bargraph.min_value) /	\
			 ((float)w->bargraph.max_value -		\
			  (float)w->bargraph.min_value))

/**********************/
/* private procedures */
/**********************/
static void	ClassInitialize(), Initialize();
static void	Resize(), Redisplay(), Notify(), Destroy();
static Boolean	SetValues();

static void	FillArea(), ForceFillArea(), PaintBar(), DrawSticky();
static void	BargraphSelect(), BargraphNotify();

/*********************/
/* public procedures */
/*********************/
extern void	BargraphSetValue();
extern void	BargraphSetColor(), BargraphSetPixmap();

/*************************/
/* bargraph translations */
/*************************/
static char bargraph_translations[] = "	\
  <Btn1Down>:	BargraphSelect()\n	\
  <Btn1Up>:	BargraphNotify()	\
";

/********************/
/* bargraph actions */
/********************/
static XtActionsRec	bargraph_actions[] = {
  {"BargraphSelect",	(XtActionProc) BargraphSelect	},
  {"BargraphNotify",	(XtActionProc) BargraphNotify	},
};

/**********************/
/* bargraph resources */
/**********************/
static XtResource resources[] = {
  /******************/
  /* core resources */
  /******************/

  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     Offset(core.width), XtRString, "100"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     Offset(core.height), XtRString, "200"},

  /**********************/
  /* bargraph resources */
  /**********************/
  {XtNstickyHold, XtCStickyHold, XtRInt, sizeof(int),
     Offset(bargraph.sticky_hold), XtRString, "5"},
  {XtNorientation, XtCOrientation, XtROrientation, sizeof(XtOrientation),
     Offset(bargraph.orientation), XtRString, "vertical"},
  {XtNfillColor, XtCFillColor, XtRPixel, sizeof(Pixel),
     Offset(bargraph.fill_color), XtRString, "blue"},
  {XtNstickyColor, XtCStickyColor, XtRPixel, sizeof(Pixel),
     Offset(bargraph.sticky_color), XtRString, "red"},
  {XtNfillPixmap, XtCFillPixmap, XtRPixmap, sizeof(Pixmap),
     Offset(bargraph.fill_pixmap), XtRPixmap, None},
  {XtNminValue, XtCMinValue, XtRInt, sizeof(int),
     Offset(bargraph.min_value), XtRString, "0"},
  {XtNmaxValue, XtCMaxValue, XtRInt, sizeof(int),
     Offset(bargraph.max_value), XtRString, "100"},
  {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     Offset(bargraph.select), XtRCallback, (caddr_t) NULL},
};
#undef Offset

/*************************/
/* bargraph class record */
/*************************/
BargraphClassRec bargraphClassRec = {
  (WidgetClass) &widgetClassRec,	/* superclass			*/
  "Bargraph",				/* class_name			*/
  sizeof(BargraphRec),			/* size				*/
  ClassInitialize,			/* class_initialize		*/
  NULL,					/* class_part_initialize	*/
  FALSE,				/* class_inited			*/
  Initialize,				/* initialize			*/
  NULL,					/* initialize_hook		*/
  XtInheritRealize,			/* realize			*/
  bargraph_actions,			/* actions			*/
  XtNumber(bargraph_actions),		/* num_actions			*/
  resources,				/* resources			*/
  XtNumber(resources),			/* resource_count		*/
  NULLQUARK,				/* xrm_class			*/
  TRUE,					/* compress_motion		*/
  TRUE,					/* compress_exposure		*/
  TRUE,					/* compress_enterleave		*/
  FALSE,				/* visible_interest		*/
  Destroy,				/* destroy			*/
  Resize,				/* resize			*/
  Redisplay,				/* expose			*/
  SetValues,				/* set_values			*/
  NULL,					/* set_values_hook		*/
  XtInheritSetValuesAlmost,		/* set_values_almost		*/
  NULL,					/* get_values_hook		*/
  NULL,					/* accept_focus			*/
  XtVersion,				/* version			*/
  NULL,					/* callback_private		*/
  bargraph_translations,		/* tm_table			*/
  XtInheritQueryGeometry,		/* query_geometry		*/
  XtInheritDisplayAccelerator,		/* display_accelerator		*/
};

WidgetClass bargraphWidgetClass = (WidgetClass)&bargraphClassRec;

static XrmQuark  XtQEhorizontal;
static XrmQuark  XtQEvertical;

/************************************************************************/
/*                          PRIVATE PROCEDURES                          */
/************************************************************************/

/************************************************************************/
/* Destroy()								*/
/************************************************************************/
static void Destroy(bgw)
Widget bgw;
{
  BargraphWidget w = (BargraphWidget) bgw;

#ifdef DEBUG
printf("Destroy\n");
#endif
  XtDestroyGC(w->bargraph.fill_colorGC);
  XtDestroyGC(w->bargraph.fill_pixmapGC);
  XtDestroyGC(w->bargraph.stickyGC);
  XtDestroyGC(w->bargraph.eraseGC);
  XtRemoveAllCallbacks( bgw, XtNselect);

} /* destroy */

/************************************************************************/
/* CvtStringToOrientation()						*/
/************************************************************************/
static void CvtStringToOrientation(args, num_args, fromVal, toVal)
XrmValuePtr	*args;		/* unused */
Cardinal	*num_args;	/* unused */
XrmValuePtr	fromVal;
XrmValuePtr	toVal;
{
  static XtOrientation	orient;
  XrmQuark		q;
  char			lowerName[1000];

  strcpy( lowerName, fromVal->addr );
  q = XrmStringToQuark(lowerName);
  if (q == XtQEhorizontal) {
    orient = XtorientHorizontal;
    (*toVal).size = sizeof(XtOrientation);
    (*toVal).addr = (caddr_t) &orient;
    return;
  }
  if (q == XtQEvertical) {
    orient = XtorientVertical;
    (*toVal).size = sizeof(XtOrientation);
    (*toVal).addr = (caddr_t) &orient;
    return;
  }

} /* CvtStringToOrientation */

/************************************************************************/
/* ClassInitialize()							*/
/************************************************************************/
static void ClassInitialize()
{
#ifdef DEBUG
printf("ClassInitialize\n");
#endif
  XtQEhorizontal = XrmStringToQuark(XtEhorizontal);
  XtQEvertical   = XrmStringToQuark(XtEvertical);
  XtAddConverter(XtRString, XtROrientation, CvtStringToOrientation,
		 NULL, (Cardinal)0);

} /* ClassInitialize */

/************************************************************************/
/* Initialize()								*/
/************************************************************************/
static void Initialize(request, new)
Widget	request;
Widget	new;
{
  BargraphWidget	w = (BargraphWidget) new;
  XGCValues		gcValues;
  XtGCMask		gcMask;

#ifdef DEBUG
printf("Initialize\n");
#endif
  /* initialize erase GC */
  gcValues.foreground = w->core.background_pixel;
  gcMask = GCForeground;
  w->bargraph.eraseGC = XtGetGC(new, gcMask, &gcValues);

  /* initialize sticky GC */
  gcValues.foreground = w->bargraph.sticky_color;
  gcMask = GCForeground;
  w->bargraph.stickyGC = XtGetGC(new, gcMask, &gcValues);

  /* initialize fill color GC */
  gcValues.foreground = w->bargraph.fill_color;
  gcValues.fill_style = FillSolid;
  gcMask = GCForeground | GCFillStyle;
  w->bargraph.fill_colorGC = XtGetGC(new, gcMask, &gcValues);

  /* initialize fill pixmap GC */

  gcValues.background = w->core.background_pixel;
  gcValues.foreground = w->bargraph.fill_color;
  gcMask = GCBackground | GCForeground | GCFillStyle;
  if (w->bargraph.fill_pixmap != None) {
    gcValues.fill_style = FillTiled;
    gcValues.tile = w->bargraph.fill_pixmap;
    gcMask = gcMask | GCTile;
  }
  w->bargraph.fill_pixmapGC = XtGetGC(new, gcMask, &gcValues);

  if (w->bargraph.orientation == XtorientVertical) {
    if (w->core.height <= 0)
      w->bargraph.length = DEFAULT_LENGTH;
    else
      w->bargraph.length = w->core.height;
    if (w->core.width <= 0)
      w->bargraph.thickness = DEFAULT_THICKNESS;
    else
      w->bargraph.thickness = w->core.width;
  }
  else {
    if (w->core.width <= 0)
      w->bargraph.length = DEFAULT_LENGTH;
    else
      w->bargraph.length = w->core.width;
    if (w->core.height <= 0)
      w->bargraph.thickness = DEFAULT_THICKNESS;
    else
      w->bargraph.thickness = w->core.height;
  }

  w->bargraph.value = w->bargraph.min_value;
  w->bargraph.old_value = w->bargraph.min_value;
  w->bargraph.sticky_value = w->bargraph.min_value;
  w->bargraph.sticky_held = w->bargraph.min_value;

} /* Initialize */

/************************************************************************/
/* SetValues()								*/
/************************************************************************/
static Boolean SetValues(gcurrent, grequest, gnew)
Widget	gcurrent;
Widget	grequest;
Widget	gnew;
{
  BargraphWidget	current = (BargraphWidget) gcurrent;
  BargraphWidget	new = (BargraphWidget) gnew;
  XGCValues		gcValues;
  XtGCMask		gcMask;
  Boolean		redraw = FALSE;
  Boolean               realized = XtIsRealized( gcurrent );

#ifdef DEBUG
printf("SetValues\n");
#endif
  /* change of erase color */
  if (new->core.background_pixel != current->core.background_pixel) {
    XtDestroyGC(current->bargraph.eraseGC);
    gcValues.foreground = new->core.background_pixel;
    gcMask = GCForeground;
    new->bargraph.eraseGC = XtGetGC( gcurrent, gcMask, &gcValues);
    redraw = TRUE;
  }

  /* change of sticky color */
  if (new->bargraph.sticky_color != current->bargraph.sticky_color) {
    XtDestroyGC(current->bargraph.stickyGC);
    gcValues.foreground = new->bargraph.sticky_color;
    gcMask = GCForeground;
    new->bargraph.stickyGC = XtGetGC( gcurrent, gcMask, &gcValues);
    redraw = TRUE;
  }

  /* change of fill color */
  if (new->bargraph.fill_color != current->bargraph.fill_color) {
    XtDestroyGC(current->bargraph.fill_colorGC);
    gcValues.foreground = new->bargraph.fill_color;
    gcValues.fill_style = FillSolid;
    gcMask = GCForeground | GCFillStyle;
    new->bargraph.fill_colorGC = XtGetGC( gcurrent, gcMask, &gcValues);
    redraw = TRUE;
  }

  /* change of fill pixmap */
  if (new->bargraph.fill_pixmap != current->bargraph.fill_pixmap) {
    XtDestroyGC(current->bargraph.fill_pixmapGC);
    gcValues.background = new->core.background_pixel;
    gcValues.foreground = new->bargraph.fill_color;
    gcValues.fill_style = FillTiled;
    gcValues.tile = new->bargraph.fill_pixmap;
    gcMask = GCBackground | GCForeground | GCFillStyle | GCTile;
    new->bargraph.fill_pixmapGC = XtGetGC( gcurrent, gcMask, &gcValues);
    redraw = TRUE;
  }

  if (redraw && realized) {
    Redisplay(new);
    redraw = FALSE;
  }

  /* change of height */
  if ((new->core.height != current->core.height)
      || (new->core.width != current->core.width)) {
    if (current->bargraph.orientation == XtorientVertical) {
      new->bargraph.length = new->core.height;
      new->bargraph.thickness = new->core.width;
    }
    else {
      new->bargraph.length = new->core.width;
      new->bargraph.thickness = new->core.height;
    }
    redraw = TRUE;
  }

  return(redraw);

} /* SetValues */

/************************************************************************/
/* Resize()								*/
/************************************************************************/
static void Resize(gw)
Widget	gw;
{
  BargraphWidget	w = (BargraphWidget) gw;

#ifdef DEBUG
printf("Resize\n");
#endif
  if (w->bargraph.orientation == XtorientVertical) {
    w->bargraph.length = w->core.height;
    w->bargraph.thickness = w->core.width;
  }
  else {
    w->bargraph.length = w->core.width;
    w->bargraph.thickness = w->core.height;
  }
  w->bargraph.old_value = w->bargraph.min_value;
  Redisplay(gw);

} /* Resize */

/************************************************************************/
/* Redisplay()								*/
/************************************************************************/
static void Redisplay(gw, event, region)
Widget	gw;
XEvent	*event;
Region	region;
{
  BargraphWidget	w = (BargraphWidget) gw;

#ifdef DEBUG
printf("Redisplay\n");
#endif
  if (XtIsRealized((Widget) w)) {
     PaintBar(w, EVENT_REDISPLAY); 
     DrawSticky(w);
  }

} /* Redisplay */

/************************************************************************/
/* FillArea()								*/
/************************************************************************/
static void FillArea(w, value, filltype)
BargraphWidget	w;
int		value;
int		filltype;
{
  Dimension	length, old_length;

#ifdef DEBUG
printf("FillArea\n");
#endif

  /* PCR: 4/20/93: Changed FillArea to make bargraph clears unnecessary. */
  /* Instead, each FillArea call checks whether the value has lowered.   */
  /* If so, FillArea clears only the area at the top of the bargraph     */
  /* which needs to be cleared, rather than redrawing the whole thing.   */
  /* So, filltype == 0 should no longer be used.			 */
  if( w->bargraph.old_value > w->bargraph.value ) {
	if( w->bargraph.orientation == XtorientHorizontal ) {
		length = (int) (Norm(w, value) * (float) (w->core.width-2));
		old_length = (int) (Norm(w, w->bargraph.old_value) * 
						(float) (w->core.width-2));
		XClearArea( XtDisplay(w), XtWindow(w),
			length + 1, 1,
			old_length, w->core.height-2,
			True );
	}
	else {
		length = (int) (Norm(w,value) * (float) (w->core.height-2));
		old_length = (int) (Norm(w,w->bargraph.old_value) * 
						(float) (w->core.height-2));
		XClearArea(XtDisplay(w), XtWindow(w),
			1, w->core.height-old_length-1,
			w->core.width-2, old_length - length,
			True );
	}
  }
  else if( w->bargraph.old_value < w->bargraph.value ) {

	switch(filltype) {


	case 1:	/* fill solid */
	  if (w->bargraph.orientation == XtorientHorizontal) {
	    length = (int) (Norm(w, value) * (float) (w->core.width-2));
	    XFillRectangle(XtDisplay(w), XtWindow(w), w->bargraph.fill_colorGC,
			   1, 1,
			   length, w->core.height-2);
	  }
	  else {
	    length = (int) (Norm(w, value) * (float) (w->core.height-2));
	    XFillRectangle(XtDisplay(w), XtWindow(w), w->bargraph.fill_colorGC,
			   1, w->core.height-length-1,
			   w->core.width-2, length);
	  }
	  break;
	case 2:	/* fill pixmap */
	  if (w->bargraph.orientation == XtorientHorizontal) {
	    length = (int) (Norm(w, value) * (float) (w->core.width-2));
	    XFillRectangle(XtDisplay(w), XtWindow(w), w->bargraph.fill_pixmapGC,
	  		 1, 1, length, w->core.height-2);
	  }
	  else {
	    length = (int) (Norm(w, value) * (float) (w->core.height-2));
	    XFillRectangle(XtDisplay(w), XtWindow(w), w->bargraph.fill_pixmapGC,
	  		 1, w->core.height-length-1,
	  		 w->core.width-2, length);
	  }
	  break;
	}  
  }

} /* FillArea */



/* PCR: 6/3/93:	Added a ForceFillArea function so that redraws occur */
/* correctly after the widget receives Expose events from the X server */

/************************************************************************/
/* ForceFillArea()							*/
/************************************************************************/
static void ForceFillArea(w, value, filltype)
BargraphWidget	w;
int		value;
int		filltype;
{
  Dimension	length, old_length;

#ifdef DEBUG
printf("ForceFillArea\n");
#endif


  switch(filltype) {

	case 0:	
	  if (w->bargraph.orientation == XtorientHorizontal) {
	    length = (int) (Norm(w, value) * (float) (w->core.width-2));
	    XClearArea(XtDisplay(w), XtWindow(w),
		 1, 1,
		 length, w->core.height-2,
		 FALSE);
	  }
	  else {
	    length = (int) (Norm(w,value) * (float) (w->core.height-2));
	    XClearArea(XtDisplay(w), XtWindow(w),
		 1, w->core.height-length-1,
		 w->core.width-2, length,
		 FALSE);
	  }
	  break;

	case 1:	/* fill solid */
	  if (w->bargraph.orientation == XtorientHorizontal) {
	    length = (int) (Norm(w, value) * (float) (w->core.width-2));
	    XFillRectangle(XtDisplay(w), XtWindow(w), w->bargraph.fill_colorGC,
			   1, 1,
			   length, w->core.height-2);
	  }
	  else {
	    length = (int) (Norm(w, value) * (float) (w->core.height-2));
	    XFillRectangle(XtDisplay(w), XtWindow(w), w->bargraph.fill_colorGC,
			   1, w->core.height-length-1,
			   w->core.width-2, length);
	  }
	  break;

	case 2:	/* fill pixmap */
	  if (w->bargraph.orientation == XtorientHorizontal) {
	    length = (int) (Norm(w, value) * (float) (w->core.width-2));
	    XFillRectangle(XtDisplay(w), XtWindow(w), w->bargraph.fill_pixmapGC,
	  		 1, 1, length, w->core.height-2);
	  }
	  else {
	    length = (int) (Norm(w, value) * (float) (w->core.height-2));
	    XFillRectangle(XtDisplay(w), XtWindow(w), w->bargraph.fill_pixmapGC,
	  		 1, w->core.height-length-1,
	  		 w->core.width-2, length);
	  }
	  break;
  }

  XFlush( XtDisplay(w) );

} /* ForceFillArea */




/************************************************************************/
/* PaintBar()								*/
/************************************************************************/
static void PaintBar(w, eventType)
BargraphWidget	w;
int eventType;
{
  int	filltype;

#ifdef DEBUG
printf("PaintBar\n");
#endif
  if (XtIsRealized((Widget) w)) {
    if (w->bargraph.fill_pixmap == None)
      filltype = 1;
    else
      filltype = 2;

    /* PCR: 6/28/93: changed the ForceFillArea from after all this	*/
    /* to before it, so that Redisplays are forced to draw the whole	*/
    /* bar.								*/
    if( eventType == EVENT_REDISPLAY ) {
      ForceFillArea(w, w->bargraph.value, filltype);
    }
    else if( w->bargraph.value != w->bargraph.old_value ) {
      /* PCR: 9/9/93:  Modified to remove some redudancies - changes in  */
      /* bargraph are made only if value has changed      	         */
      FillArea(w, w->bargraph.value, filltype);
    }
  }
} /* PaintBar */


/************************************************************************/
/* DrawSticky()								*/
/************************************************************************/
static void DrawSticky(w)
BargraphWidget	w;
{
  int		value = w->bargraph.value;
  Dimension	length;

#ifdef DEBUG
printf("DrawSticky\n");
#endif
  if (!(XtIsRealized((Widget) w)))
     return;

  if (w->bargraph.sticky_hold == -1)
    return;
  if (w->bargraph.sticky_hold == 0) {
    if (w->bargraph.orientation == XtorientHorizontal) {
      length = (int) (Norm(w, value) * (float) (w->core.width-2));
      if (length == 0)
	length = 1;
      XDrawLine(XtDisplay(w), XtWindow(w), w->bargraph.stickyGC,
		length, 1, length, w->core.height-2);
    }
    else {
      length = (int) (Norm(w, value) * (float) (w->core.height-2));
      if (length == 0)
	length = 1;
      XDrawLine(XtDisplay(w), XtWindow(w), w->bargraph.stickyGC,
		1, w->core.height-length-1,
		w->core.width-2, w->core.height-length-1);
    }
  }
  else {
    if (value > w->bargraph.sticky_value) {
      w->bargraph.sticky_value = value;
      w->bargraph.sticky_held = w->bargraph.sticky_hold;
    }
    else if ((value < w->bargraph.sticky_value) &&
	     (w->bargraph.sticky_held <= 0)) {
      if (w->bargraph.orientation == XtorientHorizontal) {
	length = (int) (Norm(w, w->bargraph.sticky_value) *
			(float) (w->core.width-2));
	if (length == 0)
	  length = 1;
	XDrawLine(XtDisplay(w), XtWindow(w), w->bargraph.eraseGC,
		  length, 1, length, w->core.height-2);
      }
      else {
	length = (int) (Norm(w, w->bargraph.sticky_value) *
			(float) (w->core.height-2));
	if (length == 0)
	  length = 1;
	XDrawLine(XtDisplay(w), XtWindow(w), w->bargraph.eraseGC,
		  1, w->core.height-length-1,
		  w->core.width-2, w->core.height-length-1);
      }
      w->bargraph.sticky_value = value;
      w->bargraph.sticky_held = w->bargraph.sticky_hold;
    }
    else {
      w->bargraph.sticky_held--;
      value = w->bargraph.sticky_value;
    }
    if (w->bargraph.orientation == XtorientHorizontal) {
      length = (int) (Norm(w, value) * (float) (w->core.width-2));
      if (length == 0)
	length = 1;
      XDrawLine(XtDisplay(w), XtWindow(w), w->bargraph.stickyGC,
		length, 1, length, w->core.height-2);
    }
    else {
      length = (int) (Norm(w, value) * (float) (w->core.height-2));
      if (length == 0)
	length = 1;
      XDrawLine(XtDisplay(w), XtWindow(w), w->bargraph.stickyGC,
		1, w->core.height-length-1,
		w->core.width-2, w->core.height-length-1);
    }
  }

} /* DrawSticky */

/************************************************************************/
/* BargraphSelect() processes button down event.			*/
/************************************************************************/
static void BargraphSelect(w, event)
BargraphWidget		w;
XButtonPressedEvent	*event;
{
#ifdef DEBUG
printf("BargraphSelect\n");
#endif
  w->bargraph.value_select = w->bargraph.value;

} /* BargraphSelect */

/************************************************************************/
/* BargraphNotify() processes button up event.				*/
/************************************************************************/
static void BargraphNotify(w, event)
BargraphWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("BargraphNotify\n");
#endif
  XtCallCallbacks( (Widget)w, XtNselect, (caddr_t) w->bargraph.value_select);

} /* BargraphNotify */

/************************************************************************/
/*                          PUBLIC PROCEDURES                           */
/************************************************************************/

/************************************************************************/
/* BargraphSetValue() sets the current value of the bargraph widget	*/
/* to value.								*/
/************************************************************************/
extern void BargraphSetValue(w, value)
BargraphWidget	w;
int		value;
{
#ifdef DEBUG
printf("BargraphSetValue\n");
#endif
  w->bargraph.old_value = w->bargraph.value;
  if (value < w->bargraph.min_value) {
    printf("BargraphSetValue: tried to set a value (%d) ", value);
    printf("less than min (%d)\n", w->bargraph.min_value);
    w->bargraph.value = w->bargraph.min_value;
  }
  else if (value > w->bargraph.max_value) {
    printf("BargraphSetValue: tried to set a value (%d) ", value);
    printf("greater than max (%d)\n", w->bargraph.max_value);
    w->bargraph.value = w->bargraph.max_value;
  }
  else
    w->bargraph.value = value;
  PaintBar(w, EVENT_VALUE_UPDATE);
  DrawSticky(w);

} /* BargraphSetValue */

/************************************************************************/
/* BargraphSetColor() sets the bargraph fill color to color.		*/
/************************************************************************/
extern void BargraphSetColor(w, color)
BargraphWidget	w;
Pixel		color;
{
  XGCValues		gcValues;
  XtGCMask		gcMask;

#ifdef DEBUG
printf("BargraphSetColor\n");
#endif
  XtDestroyGC(w->bargraph.fill_colorGC);
  gcValues.foreground = w->bargraph.fill_color;
  gcValues.fill_style = FillSolid;
  gcMask = GCForeground | GCFillStyle;
  w->bargraph.fill_colorGC = XtGetGC( (Widget)w, gcMask, &gcValues);
  PaintBar(w, EVENT_COLOR_UPDATE);
  DrawSticky(w);

} /* BargraphSetColor */

/************************************************************************/
/* BargraphSetPixmap() sets the bargraph fill pixmap to color.		*/
/************************************************************************/
extern void BargraphSetPixmap(w, pixmap)
BargraphWidget	w;
Pixmap		pixmap;
{
  XGCValues		gcValues;
  XtGCMask		gcMask;

#ifdef DEBUG
printf("BargraphSetPixmap\n");
#endif
  if (w->bargraph.fill_pixmap == None)
    XtDestroyGC(w->bargraph.fill_pixmapGC);
  gcValues.background = w->core.background_pixel;
  gcValues.foreground = w->bargraph.fill_color;
  gcMask = GCBackground | GCForeground | GCFillStyle;
  if (w->bargraph.fill_pixmap != None) {
    gcValues.fill_style = FillTiled;
    gcValues.tile = w->bargraph.fill_pixmap;
    gcMask = gcMask | GCTile;
  }
  w->bargraph.fill_pixmapGC = XtGetGC( (Widget)w, gcMask, &gcValues);
  PaintBar(w, EVENT_PIXMAP_UPDATE);
  DrawSticky(w);

} /* BargraphSetPixmap */
