/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Authors: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *                       Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/bubble/RCS/Bubble.c,v 1.10 1994/02/25 04:45:37 aydt Exp $
 */

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include <X11/keysymdef.h>
#include "BubbleP.h"
#include <stdio.h>
#include <stdlib.h>

#define offset(field)	XtOffset(BubbleWidget,field)
#define norm(w, v)	(((float)v-(float)w->bubble.min_value) /	\
			 ((float)w->bubble.max_value -			\
			  (float)w->bubble.min_value + 1))

static void	BubbleSelect(), BubbleNotify();
static char bubble_translations[] = "	\
  <Btn1Down>:	BubbleSelect()\n	\
  <Btn1Up>:	BubbleNotify()		\
";

static XtActionsRec	bubble_actions[] = {
  {"BubbleSelect",	(XtActionProc) BubbleSelect	},
  {"BubbleNotify",	(XtActionProc) BubbleNotify	},
};

static XtResource resources[] = {
  /* core resources */
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     offset(core.width), XtRString, "200"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     offset(core.height), XtRString, "200"},
  {XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
     offset(core.background_pixel), XtRString, "white"},
  /* bubble resources */
  {XtNcolors, XtCColors, XtRPointer, sizeof(Pixel *),
     offset(bubble.colors), XtRString, NULL},
  {XtNnumColors, XtCNumColors, XtRInt, sizeof(int),
     offset(bubble.numcolors), XtRString, "0"},
  {XtNpixmaps, XtCPixmaps, XtRPointer, sizeof(Pixel *),
     offset(bubble.pixmaps), XtRString, NULL},
  {XtNnumPixmaps, XtCNumPixmaps, XtRInt, sizeof(int),
     offset(bubble.numpixmaps), XtRString, "0"},
  {XtNgridColor, XtCGridColor, XtRPixel, sizeof(Pixel),
     offset(bubble.grid_color), XtRString, "black"},
  {XtNshowGrid, XtCShowGrid, XtRBoolean, sizeof (Boolean),
     offset(bubble.show_grid), XtRString, "FALSE"},
  {XtNrows, XtCRows, XtRInt, sizeof(int), 
     offset(bubble.rows), XtRString, "10" },
  {XtNcolumns, XtCColumns, XtRInt, sizeof(int), 
     offset(bubble.columns), XtRString, "10" },
  {XtNvalues, XtCValues, XtRPointer, sizeof(bubbleValue *),
     offset(bubble.values), XtRString, NULL},
  {XtNminValue, XtCMinValue, XtRInt, sizeof(int),
     offset(bubble.min_value), XtRString, "0"},
  {XtNmaxValue, XtCMaxValue, XtRInt, sizeof(int),
     offset(bubble.max_value), XtRString, "100"},
  {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     offset(bubble.select), XtRCallback, (caddr_t) NULL},
};

#undef offset

static void	ClassInitialize(), Initialize();
static void	Realize(), Destroy(), Resize(), Redisplay();
static Boolean	SetValues();

static void	CreateColorGCs(), CreatePixmapGCs();
static void	DestroyColorGCs(), DestroyPixmapGCs();
static void	DrawBubble();

static void	Realloc();
	
/* public procedures */
extern void	BubbleSetValues();
extern void	BubbleSetColors(), BubbleSetPixmaps();
extern void	BubbleGetRowCol();

BubbleClassRec bubbleClassRec = {
  &widgetClassRec,		/* superclass			*/
  "Bubble",			/* class_name			*/
  sizeof(BubbleRec),		/* widget_size			*/
  ClassInitialize,		/* class_initialize		*/
  NULL,				/* class_part_initialize	*/
  FALSE,			/* class_inited			*/
  Initialize,			/* initialize			*/
  NULL,				/* initialize_hook		*/
  Realize,			/* realize			*/
  bubble_actions,		/* actions			*/
  XtNumber(bubble_actions),	/* num_actions			*/
  resources,			/* resources			*/
  XtNumber(resources),		/* resource_count		*/
  NULL,				/* xrm_class			*/
  TRUE,				/* compress_motion		*/
  TRUE,				/* compress_exposure		*/
  TRUE,				/* compress_enterleave		*/
  FALSE,			/* visible_interest		*/
  Destroy,			/* destroy			*/
  Resize,			/* resize			*/
  Redisplay,			/* expose			*/
  SetValues,			/* set_values			*/
  NULL,				/* set_values_hook		*/
  XtInheritSetValuesAlmost,	/* set_values_almost		*/
  NULL,				/* get_values_hook		*/
  NULL,				/* accept_focus			*/
  XtVersion,			/* version			*/
  NULL,				/* callback_private		*/
  bubble_translations,		/* tm_table			*/
  XtInheritQueryGeometry,	/* query_geometry          	*/
  XtInheritDisplayAccelerator,	/* display_accelerator    	*/
  NULL				/* extension               	*/
};

WidgetClass bubbleWidgetClass = (WidgetClass) &bubbleClassRec;

static void ClassInitialize()
{
  XtAddConverter(XtRString, XtRBackingStore, XmuCvtStringToBackingStore,
		 NULL, 0);
}

/* ARGSUSED */
static void Initialize(request, new)
Widget request, new;
{
  int		i;
  BubbleWidget	w = (BubbleWidget)new;
  XtGCMask	GCmask;
  XGCValues	GCvalues;

#ifdef DEBUG
printf("Initialize\n");
#endif
  if (w->core.width == 0)
    w->core.width = MINSIZE;
  if (w->core.height == 0)
    w->core.height = MINSIZE;

  w->bubble.values = NULL;

  GCmask = GCForeground | GCBackground;
  GCvalues.background = w->core.background_pixel;
  GCvalues.foreground = w->bubble.grid_color;
  w->bubble.gridGC = XtGetGC((Widget)w, GCmask, &GCvalues);

  GCmask = GCForeground;
  GCvalues.foreground = w->core.background_pixel;
  w->bubble.eraseGC = XtGetGC((Widget)w, GCmask, &GCvalues);
  w->bubble.newSize = FALSE;

} /* Initialize */

/* ARGSUSED */
static Boolean SetValues(gcurrent, grequest, gnew)
Widget	gcurrent, grequest, gnew;
{
  int			i;
  register BubbleWidget	current = (BubbleWidget) gcurrent;
  register BubbleWidget	new = (BubbleWidget) gnew;
  Boolean		redraw = FALSE;
  XtGCMask		GCmask;
  XGCValues		GCvalues;
  Boolean               realized = XtIsRealized(gcurrent);

  Boolean		reAlloc = FALSE;

#ifdef DEBUG
printf("SetValues\n");
#endif
  /* change of width or height */
  if ((new->core.height != current->core.height)
      || (new->core.width != current->core.width)) {
    Resize(new);
    redraw = TRUE;
  }

  /* change grid color */
  if ((new->bubble.grid_color != current->bubble.grid_color)
      || (new->core.background_pixel != current->core.background_pixel)) {
    GCmask = GCForeground | GCBackground;
    GCvalues.foreground = new->bubble.grid_color;
    GCvalues.background = new->core.background_pixel;
    XtDestroyGC(current->bubble.gridGC);
    new->bubble.gridGC = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }

  /* change number of rows */
  if (new->bubble.rows != current->bubble.rows) {
    if (new->bubble.rows > MAXROWS)
      new->bubble.rows = MAXROWS;
    reAlloc = TRUE;
    new->bubble.newSize = TRUE;       /* Can't resize now - no values        */
  }

  /* change number of columns */
  if (new->bubble.columns != current->bubble.columns) {
    if (new->bubble.columns > MAXCOLUMNS)
      new->bubble.columns = MAXCOLUMNS;
    reAlloc = TRUE;
    new->bubble.newSize = TRUE;       /* Can't resize now - no values        */
  }

 /* change erase color */
  if(new->core.background_pixel != current->core.background_pixel) {
    GCmask = GCForeground;
    GCvalues.foreground = new->core.background_pixel;
    XtDestroyGC(current->bubble.eraseGC);
    new->bubble.eraseGC = XtGetGC((Widget)current, GCmask, &GCvalues);
    redraw = TRUE;
  }

  /* change colors */
  if (new->bubble.colors != current->bubble.colors) {
    DestroyColorGCs(current);
    CreateColorGCs(new);
    redraw = TRUE;
  }

  /* change of pixmaps */
  if (new->bubble.pixmaps != current->bubble.pixmaps) {
    DestroyPixmapGCs(current);
    CreatePixmapGCs(new);
    redraw = TRUE;
  }

        /* reallocate array space, resize, and redraw as necessary */
        if ( reAlloc ) {
                Realloc( current, new );
		redraw = TRUE;
	}

  if (redraw && realized) {
    Redisplay(new);
  }

  return(FALSE);
  
} /* SetValues */

static void Realize(gw, GCmask, attrs)
Widget			gw;
XtGCMask		*GCmask;
XSetWindowAttributes	*attrs;
{
  BubbleWidget	w = (BubbleWidget) gw;

#ifdef DEBUG
printf("Realize\n");
#endif
  *GCmask |= CWBitGravity;
  attrs->bit_gravity = ForgetGravity;
  switch(w->bubble.backing_store) {
  case Always:
  case NotUseful:
  case WhenMapped:
    *GCmask |=CWBackingStore;
    attrs->backing_store = w->bubble.backing_store;
    break;
  }
  XtCreateWindow(gw, InputOutput,(Visual *)CopyFromParent,
		 *GCmask, attrs);
  Resize(gw);

} /* Realize */

static void Destroy(gw)
Widget gw;
{
  int		i;
  BubbleWidget	w = (BubbleWidget) gw;

#ifdef DEBUG
printf("Destroy\n");
#endif
  DestroyColorGCs(w);
  DestroyPixmapGCs(w);
  XtDestroyGC(w->bubble.eraseGC);

  XtRemoveAllCallbacks(w, XtNselect);

  if (w->bubble.values != NULL)
     XtFree(w->bubble.values);

} /* Destroy */

static void Resize(gw) 
Widget	gw;
{
  BubbleWidget	w = (BubbleWidget) gw;
  int		width;
  int		height;

#ifdef DEBUG
printf("Resize\n");
#endif
  if (XtIsRealized(gw)) {
    if ((w->bubble.columns > 0) && (w->bubble.rows > 0)) {
      width = w->core.width - 2*PADDING;
      w->bubble.offsety = 0;
      height = w->core.height - w->bubble.offsety - PADDING;
      w->bubble.elew = width / w->bubble.columns;
      w->bubble.eleh = height / w->bubble.rows;
      w->bubble.offsetx = PADDING;
      w->bubble.offsetx += (width - w->bubble.columns*w->bubble.elew) / 2;
      w->bubble.offsety += (height - w->bubble.rows*w->bubble.eleh) / 2;
      w->bubble.newSize = FALSE;
    }
  }
} /* Resize */

/* ARGSUSED */
static void Redisplay(gw, event, region)
Widget	gw;
XEvent	*event;				/* unused */
Region	region;				/* unused */
{
  BubbleWidget	w = (BubbleWidget) gw;
  
#ifdef DEBUG
printf("Redisplay\n");
#endif

  if (XtIsRealized(w)) {
     if ( w->bubble.newSize ) {
	Resize( w );
     }
     DrawBubble(w);
  }

} /* Redisplay */

static void CreateColorGCs(w)
BubbleWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreateColorGCs\n");
#endif
  gcValues.fill_style = FillSolid;
  gcMask = GCForeground | GCFillStyle;
  for (i=0; i<w->bubble.numcolors; i++) {
    gcValues.foreground = w->bubble.colors[i];
    w->bubble.colorGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreateColorGCs */

static void CreatePixmapGCs(w)
BubbleWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreatePixmapGCs\n");
#endif
  gcValues.background = w->core.background_pixel;
  gcValues.fill_style = FillTiled;
  gcValues.foreground = w->bubble.grid_color;
  gcMask = GCBackground | GCForeground | GCFillStyle | GCTile;
  for (i=0; i<w->bubble.numpixmaps; i++) {
    gcValues.tile = w->bubble.pixmaps[i];
    w->bubble.pixmapGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreatePixmapGCs */

static void DestroyColorGCs(w)
BubbleWidget	w;
{
  register	i;
  
#ifdef DEBUG
printf("DestroyColorGCs\n");
#endif
  if (w->bubble.colors != NULL)
    for (i=0; i<w->bubble.numcolors; i++)
      XtDestroyGC(w->bubble.colorGCs[i]);

} /* DestroyColorGCs */

static void DestroyPixmapGCs(w)
BubbleWidget	w;
{
  register	i;
  
#ifdef DEBUG
printf("DestroyPixmapGCs\n");
#endif
  if (w->bubble.pixmaps != NULL)
    for (i=0; i<w->bubble.numpixmaps; i++)
      XtDestroyGC(w->bubble.pixmapGCs[i]);

} /* DestroyPixmapGCs */

/************************************************************************/
/* MapXYToRowCol() takes x,y coordinates in the bubble widget window	*/
/* and maps them to bubble row,column values.  A -1 in either the row	*/
/* or column value indicate the x or y coordinate, respectively, was	*/
/* outside the bubble part.						*/
/************************************************************************/
static void MapXYToRowCol(w, x, y)
BubbleWidget	w;
int		x, y;
{
  int	i;
  int	elew = w->bubble.elew, eleh = w->bubble.eleh;
  int	offsetx = w->bubble.offsetx, offsety = w->bubble.offsety;
  int	columns = w->bubble.columns;

#ifdef DEBUG
printf("MapXYToRowCol\n");
#endif
  for (i=0; i<w->bubble.columns; i++)
    if ((x >= (offsetx+i*elew))
	&& (x <= (offsetx+(i+1)*elew-1)))
      break;
  if (i == w->bubble.columns)
    w->bubble.column_select = -1;
  else
    w->bubble.column_select = i;

  for (i=0; i<w->bubble.rows; i++)
    if ((y >= (offsety+i*eleh))
	&& (y <= (offsety+(i+1)*eleh-1)))
      break;
  if (i == w->bubble.rows)
    w->bubble.row_select = -1;
  else
    w->bubble.row_select = i;

} /* MapXYToRowCol */

/************************************************************************/
/* BubbleSelect() processes button down event.				*/
/************************************************************************/
static void BubbleSelect(w, event)
BubbleWidget		w;
XButtonPressedEvent	*event;
{
#ifdef DEBUG
printf("BubbleSelect\n");
#endif
  MapXYToRowCol(w, event->x, event->y);

} /* BubbleSelect */

/************************************************************************/
/* BubbleNotify() processes button up event.				*/
/************************************************************************/
static void BubbleNotify(w, event)
BubbleWidget		w;
XButtonReleasedEvent	*event;
{
  XtCallCallbacks(w, XtNselect, event);

} /* BubbleNotify */

static void DrawBubble(w)
BubbleWidget	w;
{
  int	i, j;
  int	color, pixmap;
  int	elew = w->bubble.elew, eleh = w->bubble.eleh;
  int	offsetx = w->bubble.offsetx, offsety = w->bubble.offsety;
  int	rows = w->bubble.rows, columns = w->bubble.columns;
  int	itemCount;
  int	wid, ht;
  int	startx,starty;
  int	v, r, c;
  int   elementCompare();

#ifdef DEBUG
printf("DrawBubble\n");
#endif

  if (w->bubble.values == NULL)
      return;

  XFillRectangle(XtDisplay(w), XtWindow(w), w->bubble.eraseGC,
		 0, 0, w->core.width, w->core.height);

  itemCount = rows * columns;

  qsort((char *) (w->bubble.values), itemCount,
	sizeof(bubbleValue), elementCompare);

  for (i=0; i<rows; i++) {
    for (j=0; j<columns; j++) {
      v = w->bubble.values[i*columns +j].v;
      r = w->bubble.values[i*columns +j].i;
      c = w->bubble.values[i*columns +j].j;

      if (w->bubble.pixmaps != NULL) {
	pixmap = norm(w, v) * (float) w->bubble.numpixmaps;

	  wid = MINCIRCLE + SCALE * (elew * norm(w, v));
	  ht = MINCIRCLE + SCALE * (eleh * norm(w, v));
	  startx = offsetx + c * elew;
	  starty = offsety + r * eleh;

 	  XFillArc(XtDisplay(w), XtWindow(w), w->bubble.pixmapGCs[pixmap],
	      startx, starty,
	      wid, ht, 0, 360*64);

	  if (w->bubble.show_grid == TRUE)
 	     XDrawArc(XtDisplay(w), XtWindow(w), w->bubble.gridGC,
                 startx, starty,
                 wid, ht, 0, 360*64);
      }
      else {
	if (w->bubble.colors != NULL) {
	  color = norm(w, v) * (float) w->bubble.numcolors;
          if (color >= w->bubble.numcolors) {
	    fprintf(stderr, 
	      "DrawBubble: ERROR Color %d too large - using maximum allowed\n", 
	      color);
	    color = w->bubble.numcolors - 1;
	  }

	  wid = MINCIRCLE + SCALE * (elew * norm(w, v));
	  ht = MINCIRCLE + SCALE * (eleh * norm(w, v));
	  startx = offsetx + c * elew;
	  starty = offsety + r * eleh;

 	  XFillArc(XtDisplay(w), XtWindow(w), w->bubble.colorGCs[color],
	      startx, starty,
	      wid, ht, 0, 360*64);

	  if (w->bubble.show_grid == TRUE)
 	     XDrawArc(XtDisplay(w), XtWindow(w), w->bubble.gridGC,
                 startx, starty,
                 wid, ht, 0, 360*64);

	}	/* colors  */
      }		/* pixmaps */
    }		/* for j   */
  }		/* for i   */

} /* DrawBubble */

/*********************/
/* public procedures */
/*********************/

extern void BubbleSetValues(w, r, c, values)
BubbleWidget    w;
int		r;
int		c;
int		*values;
{
  register int	i, j;
  int 		rows, cols, minv, maxv, val;

#ifdef DEBUG
printf("BubbleSetValues\n");
#endif

  if ((r != w->bubble.rows) || (c != w->bubble.columns)) { 

     /* New number of rows or columns.  Free old space if appropriate,
      * check new row & column sizes to make sure within range, and
      * allocate space for new values. Flag a new size */

     if (w->bubble.values != NULL) {
	XtFree(w->bubble.values);
     }

     if ( (r > 0) && (r <= MAXROWS) ) {
        w->bubble.rows = r;
     } else {
        fprintf(stderr, 
	   "BubbleSetValues: number of rows (%d) out of range - using %d\n",
	   r, MAXROWS );
	w->bubble.rows = MAXROWS;
     }

     if ((c > 0) && (c < MAXCOLUMNS)) {
        w->bubble.columns = c;
     } else {
        fprintf(stderr,
	   "BubbleSetValues:  number of columns (%d) out of range - using %d\n",
	   c, MAXCOLUMNS );
	w->bubble.columns = MAXCOLUMNS;
     }

     rows = w->bubble.rows;
     cols = w->bubble.columns;
     w->bubble.values = 
		(bubbleValue *) XtMalloc( rows * cols * sizeof(bubbleValue));

     w->bubble.newSize = TRUE;
  } else {

     /* No change in row or column size.  But, may still need to allocate
      * space for values */

     rows = w->bubble.rows;
     cols = w->bubble.columns;

     if (w->bubble.values == NULL) {
        w->bubble.values = 
		(bubbleValue *) XtMalloc( rows * cols  * sizeof(bubbleValue));
     }
  }

  /* Okay - Ready to populate values. */

  minv = w->bubble.min_value;
  maxv = w->bubble.max_value;

  for ( i = 0; i < rows; i++ ) {
     for ( j = 0; j < cols; j++ ) {

	w->bubble.values[i * cols + j].i = i;
	w->bubble.values[i * cols + j].j = j;

	val = values[ i * c + j];	/* Use c here, not cols, in case *
					 * values had too many columns.  */
	if ( val < minv ) {
       	   fprintf(stderr,
	      "BubbleSetValues:  Value %d for (%d, %d) too small - using %d\n",
	       val, i, j, minv );
           w->bubble.values[i * cols + j].v = minv;
	} else if ( val > maxv ) {
       	   fprintf(stderr,
	      "BubbleSetValues:  Value %d for (%d, %d) too large - using %d\n",
	       val, i, j, maxv );
           w->bubble.values[i * cols + j].v = maxv;
	} else {		
           w->bubble.values[i * cols + j].v = val;
	}

      }	   /* for j */
  }	   /* for i */

  Redisplay(w);

} /* BubbleSetValues */

/*
 * HACK WARNING!!!!   If this is called and r != bubble.rows or
 * c != bubble.cols, then the values get resized & the old ones
 * are lost.  If you call SetValues() to first adjust the row and/or col,
 * then call this, the old values are saved.
 */
extern void BubbleSetValue(w, r, c, v)
BubbleWidget    w;
int		r, c;
int		v;
{

  int		i, j;			      /* loop variables             */
  int		rows, cols, minv, maxv, newv; /* local copy of widget info  */
  int		rr, cc;			      /* position of matching entry */
  Boolean	redraw = FALSE;

#ifdef DEBUG
printf("BubbleSetValue - r:%d c:%d v: %d\n", r, c, v);
#endif
  
  rows = w->bubble.rows;
  cols = w->bubble.columns;
  minv = w->bubble.min_value;
  maxv = w->bubble.max_value;

  if ( ( r > rows ) || ( c > cols ) ) {
     fprintf(stderr, 
        "BubbleSetValue: (row,col)=(%d,%d) outside display range of (%d,%d)\n",
	r, c, rows, cols );
     return;
  }

  if ( w->bubble.values == NULL ) {

      /* If space for values not yet allocated, verify that rows and cols
       * have been specified & then allocate space and populate it          */

      if ( ( rows <= 0 ) || ( cols <= 0 ) ) {
         fprintf(stderr, 
	    "BubbleSetValue: ERROR - row and column sizes must be set first\n");
	 return;
      } else {
         w->bubble.values = 
		(bubbleValue *) XtMalloc( rows * cols * sizeof(bubbleValue));

  	 for (i = 0; i < rows; i++) {
     	    for (j = 0; j < cols; j++) {
	       w->bubble.values[i * cols + j].i = i;
	       w->bubble.values[i * cols + j].j = j;
               w->bubble.values[i * cols + j].v = minv;
            } /* for j */
         }    /* for i */
	 redraw = TRUE;
	 rr = r;			/* in this case, match is at (r,c) */
	 cc = c;	
      }

   } else {

      /* Since values is sorted, we must look for location of entry that
       * has location (r,c)                                                */

      for (i = 0; i < rows; i++) {
         for (j = 0; j < cols; j++) {
            if ( ( w->bubble.values[i * cols + j].i == r ) &&
		 ( w->bubble.values[i * cols + j].j == c ) ) {
	       rr = i;		/* match is found, save location */	
	       cc = j;
	       j = cols;	/* force exit from loops         */
	       i = rows;
	     }
         } /* for j */
      }    /* for i */
   }
      

   /* Verify that value is within range & adjust if it isn't.  Only Redisplay
    * if new value differs from old.                                          */

   if ( v < minv ) {
      fprintf(stderr,
	      "BubbleSetValue:  value %d for (%d, %d) too small - using %d\n",
	       v, r, c, minv );
      newv = minv;
   } else if ( v > maxv ) {
      fprintf(stderr,
	      "BubbleSetValues:  value %d for (%d, %d) too large - using %d\n",
	       v, r, c, maxv );
      newv = maxv;
   } else {		
      newv = v;
   }

   if ( w->bubble.values[rr * cols + cc].v != newv ) {
      w->bubble.values[rr * cols + cc].v = newv;
      redraw = TRUE;
   }

   if ( redraw ) 
      Redisplay(w); 

} /* BubbleSetValue */

extern void BubbleSetColors(w, numcolors, colors)
BubbleWidget	w;
int		numcolors;
Pixel		*colors;
{
#ifdef DEBUG
printf("BubbleSetColors\n");
#endif
  DestroyColorGCs(w);
  w->bubble.numcolors = numcolors;
  w->bubble.colors = colors;
  CreateColorGCs(w);

  DrawBubble(w);

} /* BubbleSetColors */

extern void BubbleSetPixmaps(w, numpixmaps, pixmaps)
BubbleWidget	w;
int		numpixmaps;
Pixmap		*pixmaps;
{
#ifdef DEBUG
printf("BubbleSetPixmaps\n");
#endif
  DestroyPixmapGCs(w);
  w->bubble.numpixmaps = numpixmaps;
  w->bubble.pixmaps = pixmaps;
  CreatePixmapGCs(w);

  DrawBubble(w);

} /* BubbleSetPixmaps */

extern void BubbleGetRowCol(w, row, col)
BubbleWidget	w;
int		*row, *col;
{
#ifdef DEBUG
printf("BubbleGetRowCol\n");
#endif
  *row = w->bubble.row_select;
  *col = w->bubble.column_select;

} /* BubbleGetRowCol */

int elementCompare(b1, b2)
bubbleValue* b1;
bubbleValue* b2;
{
    return((b2->v) - (b1->v));
}


/*----------------------------------------------------------------------------*
 *      static void Realloc( old, new );
 *
 *      Realloc is called by Bubble::SetValues when a new array dimension
 *      is seen.  It allocates space for the new bubble, copies values
 *      from the old bubble into the new, populates any "extra" cells with
 *      min_value-1, and releases the old bubble space.
 *----------------------------------------------------------------------------*/

static void Realloc( old, new )
BubbleWidget    old, new;
{
        int     r, c;
        int     oldr = old->bubble.rows;        /* local copies of vars */
        int     oldc = old->bubble.columns;
        int     newr = new->bubble.rows;
        int     newc = new->bubble.columns;
        int     minValue = new->bubble.min_value;

	int	i, j, v;

#ifdef DEBUG
        printf( "BubbleWidget: Realloc\n" );
#endif

        if ( old->bubble.values == NULL ) {     /* no values to copy    */
                return;
        }

        new->bubble.values = 
		(bubbleValue *)XtMalloc( newr * newc * sizeof( bubbleValue ) );

        for ( r = 0; r < newr; r++ ) {
                for ( c = 0; c < newc; c++ ) {
                    new->bubble.values[r * newc + c].i = r;
                    new->bubble.values[r * newc + c].j = c;
                    new->bubble.values[r * newc + c].v = minValue;
		}
	}

	for ( i = 0; i < oldr; i++ ) {
                for ( j = 0; j < oldc; j++ ) {
		    r = old->bubble.values[ i * oldc + j].i;
		    c = old->bubble.values[ i * oldc + j].j;
		    v = old->bubble.values[ i * oldc + j].v;

                    if ( r < newr && c < newc ) {
                        new->bubble.values[r * newc + c].v = v;
#ifdef DEBUG
        printf( "New (r,c) is (%d, %d)\n", r, c );
        printf( "Old (r,c) is (%d, %d)\n", i, j );
        printf( "Value is %d\n", v );
#endif
		    }
                }
        }

        XtFree( old->bubble.values );

} /* Realloc */

