/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Brian K. Totty (totty@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/chart/RCS/Chart.c,v 1.8 1994/02/25 04:45:43 aydt Exp $
 */

/*************************************************************

	Chart.c

	This file contains the C code "guts" of the
	stripchart (chart) widget.

**************************************************************/

#include <stdio.h>
#include <X11/Xatom.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include <X11/Xaw/Form.h>
#include "ChartP.h"

	/* Private Definitions */

#define FPAD	   		3
#define FHEIGHT(f) (f->max_bounds.ascent+f->max_bounds.descent+2*FPAD)

#define MARKDISP		15		/* Space mark is above data */
#define MINDATAHEIGHT		40		/* Min height of data area */
#define MINCHARTWIDTH		50		/* Min chart width */

#define max(a, b) ((a) > (b) ? (a) : (b))
#define min(a, b) ((a) < (b) ? (a) : (b))
#define abs(a) ((a) < 0 ? -(a) : (a))
#define XtStrlen(s) ((s) ? strlen(s) : 0)

static Dimension DEFAULTVALUE = 0;

	/* Initialization of defaults */

#define offset(field) XtOffset(ChartWidget,chart.field)
#define goffset(field) XtOffset(Widget,core.field)

static XtResource resources[] =
{
	{XtNwidth,XtCWidth,XtRDimension,sizeof(Dimension),
		goffset(width),XtRString,"300"},
	{XtNheight,XtCHeight,XtRDimension,sizeof(Dimension),
		goffset(height),XtRString,"100"},
	{XtNbackground,XtCBackground,XtRPixel,sizeof(Pixel),
		goffset(background_pixel),XtRString,"white"},
	{XtNbackingStore,XtCBackingStore,XtRBackingStore,sizeof (int),
	    	offset(backing_store),XtRString,"default"},

	{XtNupdate,XtCInterval,XtRInt,sizeof(int), 
		offset(update),XtRString,"1000"},
	{XtNtext,XtCForeground,XtRPixel,sizeof(Pixel),
		offset(text_pixel),XtRString,"blue"},
	{XtNdata,XtCForeground,XtRPixel,sizeof(Pixel),
		offset(data_pixel),XtRString,"black"},
	{XtNmarker,XtCForeground,XtRPixel,sizeof(Pixel),
		offset(mark_pixel),XtRString,"red"},
	{XtNfont,XtCFont,XtRFontStruct,sizeof(XFontStruct *),
		offset(font),XtRString,"variable"},
	{XtNtitle,XtCLabel,XtRString,sizeof(String),
	        offset(chart_title),XtRString,NULL},

	{XtNinfoline,XtCBoolean,XtRBoolean,sizeof(Boolean),
		offset(infoline),XtRString,"TRUE"},
	{XtNautoscale,XtCBoolean,XtRBoolean,sizeof(Boolean),
		offset(autoscale),XtRString,"TRUE"},
	{XtNpassive,XtCBoolean,XtRBoolean,sizeof(Boolean),
		offset(passive),XtRString,"FALSE"},

	{XtNscale,XtCValue,XtRInt,sizeof(int),
		offset(scale),XtRString,"100"},
        {XtNmaxValue, XtCValue,XtRInt, sizeof(int),
                offset(max_value),XtRString,"0"},        
        {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
         	offset(select), XtRCallback, (caddr_t) NULL},
};

#undef offset
#undef goffset

/********************************/
/* private procedures		*/
/********************************/

static void ClassInitialize();
static void Initialize(), Realize();
static void Destroy(), Resize(), Redisplay(), Snap();
static void ChartSelect(), ChartNotify();
static Boolean SetValues();

/********************************/
/* public procedures		*/
/********************************/

extern void ChartAddValue(); 

/*************************/
/* chart translations    */
/*************************/
static char chart_translations[] = "	\
  <Btn1Down>:	ChartSelect()\n	\
  <Btn1Up>:	ChartNotify()	\
";

/********************/
/* chart actions    */
/********************/
static XtActionsRec	chart_actions[] = {
  {"ChartSelect",	(XtActionProc) ChartSelect	},
  {"ChartNotify",	(XtActionProc) ChartNotify	},
};

	/* Allocate a class object for chart */

ChartClassRec chartClassRec =
{
	{
		/* superclass		*/	&widgetClassRec,
		/* class_name		*/	"Chart",
		/* widget_size		*/	sizeof(ChartRec),
		/* class_initialize	*/	ClassInitialize,
		/* class_part_initialize*/	NULL,
		/* class_inited		*/	FALSE,
		/* initialize		*/	Initialize,
		/* initialize_hook	*/	NULL,
		/* realize		*/	Realize,
		/* actions		*/	chart_actions,
		/* num_actions		*/	XtNumber(chart_actions),
		/* resources		*/	resources,
		/* resource_count	*/	XtNumber(resources),
		/* xrm_class		*/	NULL,
		/* compress_motion	*/	TRUE,
		/* compress_exposure	*/	TRUE,
		/* compress_enterleave	*/	TRUE,
		/* visible_interest	*/	FALSE,
		/* destroy		*/	Destroy,
		/* resize		*/	Resize,
		/* expose		*/	Redisplay,
		/* set_values		*/	SetValues,
		/* set_values_hook	*/	NULL,
		/* set_values_almost	*/	XtInheritSetValuesAlmost,
		/* get_values_hook	*/	NULL,
		/* accept_focus		*/	NULL,
		/* version		*/	XtVersion,
		/* callback_private	*/	NULL,
		/* tm_table		*/	chart_translations,
		/* query_geometry       */	XtInheritQueryGeometry,
		/* display_accelerator  */	XtInheritDisplayAccelerator,
		/* extension            */	NULL
	}, /* Core Part */
	{
		/* dummy variable	*/	0
	}  /* Chart Part */
};

WidgetClass chartWidgetClass = (WidgetClass) &chartClassRec;

	/* Private Procedures */

/*---------------------------------------------------------------*

	ClassInitialize()

	This procedure is called by the X toolkit to initialize
	the class.  The hook to this routine is in the
	class_initialize part of the core part of the class.

 *---------------------------------------------------------------*/

static void ClassInitialize()
{
	XtAddConverter(XtRString,XtRBackingStore,
		XmuCvtStringToBackingStore,NULL,0);
} /* End ClassInitialize */



/*---------------------------------------------------------------*

	Initialize()

	This procedure is called by the X toolkit to initialize
	the widget instance.  The hook to this routine is in the
	initialize part of the core part of the class.

 *---------------------------------------------------------------*/

/* ARGSUSED */
static void Initialize(request,new)
Widget request,new;
{
	ChartWidget w;
	XtGCMask valuemask;
	XGCValues myXGCV;
	int min_height,min_width;
	Pixel fg,bg;

	w = (ChartWidget)new;
	if (w->chart.chart_title != NULL)
	        w->chart.chart_title =
			strcpy(XtMalloc(strlen(w->chart.chart_title) + 1),
			w->chart.chart_title);

	valuemask = GCForeground | GCBackground | GCLineWidth;

		/* Make a read-only GC for data lines */

	myXGCV.foreground = w->chart.data_pixel;
	myXGCV.background = w->core.background_pixel;
	myXGCV.line_width = 0;
	w->chart.dataGC = XtGetGC((Widget)w,valuemask,&myXGCV);

		/* Make a read-only GC for marker line */

	myXGCV.foreground = w->chart.mark_pixel;
	w->chart.markGC = XtGetGC((Widget)w,valuemask,&myXGCV);

		/* Make a read-only GC for erasing */

	myXGCV.foreground = w->core.background_pixel;
	w->chart.eraserGC = XtGetGC((Widget)w,valuemask,&myXGCV);

		/* Make a read-only GC for text */

	valuemask = GCForeground | GCBackground | GCFont;
	myXGCV.foreground = w->chart.text_pixel;
	if (w->chart.font != NULL)
	        myXGCV.font = w->chart.font->fid;
	    else
	        valuemask &= ~GCFont;	/* use server default font */
	w->chart.textGC = XtGetGC((Widget)w,valuemask,&myXGCV);

	if (w->chart.update < 0) w->chart.update = 0;
	w->chart.interval_id = 0;
	w->chart.array_size = 100;
	w->chart.value_array = (int *)XtMalloc(sizeof(int) *
		w->chart.array_size);
	w->chart.last_filled = -1;
	w->chart.marker = 0;
} /* End Initialize */



/*---------------------------------------------------------------*

	Realize()

	This function is called to realize a chart widget.

 *---------------------------------------------------------------*/

static void Realize(gw,valueMask,attrs)
Widget gw;
XtValueMask *valueMask;
XSetWindowAttributes *attrs;
{
	ChartWidget w;

	w = (ChartWidget)gw;
#ifdef notdef
	*valueMask |= CWBitGravity;
	attrs->bit_gravity = ForgetGravity;
#endif
	switch (w->chart.backing_store)
	{
	case Always:
	case NotUseful:
	case WhenMapped:
		*valueMask |= CWBackingStore;
		attrs->backing_store = w->chart.backing_store;
		break;
	}
	XtCreateWindow(gw,InputOutput,(Visual *)CopyFromParent,
		*valueMask,attrs);
	Resize(gw);
} /* End Realize */



/*---------------------------------------------------------------*

	Destroy()

	This function is called to destroy a chart widget.

 *---------------------------------------------------------------*/

static void Destroy(gw)
Widget gw;
{
	ChartWidget w;

	w = (ChartWidget)gw;
	if (w->chart.interval_id) XtRemoveTimeOut(w->chart.interval_id);
	XtDestroyGC(w->chart.dataGC);
	XtDestroyGC(w->chart.markGC);
	XtDestroyGC(w->chart.textGC);
	XtDestroyGC(w->chart.eraserGC);
	XtFree(w->chart.value_array);
	if (w->chart.chart_title != NULL) XtFree(w->chart.chart_title);
        XtRemoveAllCallbacks(w, XtNselect);
} /* End Destroy */



/*---------------------------------------------------------------*

	Resize()

	This function is called to resize a chart widget.

 *---------------------------------------------------------------*/

static void Resize(gw) 
Widget gw;
{
	ChartWidget w;
	int *array,size,i;

	w = (ChartWidget)gw;
	Snap(w);
	size = w->core.width;
	array = (int *)XtMalloc(sizeof(int) * size);
	for (i = 0; i < size; i++) array[i] = 0;
	for (i = 0; i < min(size,w->chart.array_size); i++)
		array[i] = w->chart.value_array[i];
	XtFree(w->chart.value_array);
	w->chart.array_size = size;
	w->chart.value_array = array;
	if (w->chart.last_filled >= size)
		w->chart.last_filled = size - 1;
} /* End Resize */



/*---------------------------------------------------------------*

	Snap(w)

	This routine takes a widget <w> and "snaps" the
	widget to an appropriate size by updating the
	size fields in the widget.

 *---------------------------------------------------------------*/

static void Snap(w)
ChartWidget w;
{
	int info_h,title_h,wid_h,wid_w;

	if (XtIsRealized(w))
	{
		wid_w = w->core.width;
		wid_h = w->core.height;
		w->chart.info_y = 0;
		if (!w->chart.infoline)
			info_h = 0;
		    else
			info_h = FHEIGHT(w->chart.font);
		w->chart.mark_y = w->chart.info_y + info_h;
		w->chart.data_y = w->chart.mark_y + MARKDISP;
		if (!w->chart.chart_title)
			title_h = 0;
		    else
			title_h = FHEIGHT(w->chart.font);
		w->chart.data_h = max(wid_h - info_h - MARKDISP - title_h,
			MINDATAHEIGHT);
		w->chart.title_y = w->chart.data_y + w->chart.data_h;
		wid_w = max(wid_w,MINCHARTWIDTH);
		wid_h = info_h + MARKDISP + w->chart.data_h + title_h;
		w->core.width = wid_w;
		w->core.height = wid_h;
	}
} /* End Snap() */



/*---------------------------------------------------------------*

	DrawCenteredText(gw,which,gc,str)

	This routine draws the string <str> into either
	the info or title area of widget <gw>.  It draws the
	text into the info area if <which> is 0 or into
	the title area if <which> is 1.  The text is attempted
	to be centered.  If the text is too big, it will
	be clipped off.

 *---------------------------------------------------------------*/

static void DrawCenteredText(gw,which,gc,str)
Widget gw;
int which;
GC gc;
char *str;
{
	ChartWidget w;
	int x,y,len,txt_width;

	w = (ChartWidget)gw;
	if (!XtIsRealized(w)) return;
	if (str == NULL) return;
	len = strlen(str);
	txt_width = XTextWidth(w->chart.font,str,len);
	x = (w->core.width - txt_width) / 2;
	if (which == 0)
		y = w->chart.info_y + FPAD + w->chart.font->max_bounds.ascent;
	    else
		y = w->chart.title_y + FPAD + w->chart.font->max_bounds.ascent;
	XDrawString(XtDisplay(w),XtWindow(w),gc,x,y,str,len);
} /* End DrawCenteredText */


/*---------------------------------------------------------------*

	Redisplay(gw,event,region)

	This routine is called to redraw the region <region>
	of the widget <gw>.  The event that caused the redraw
	is pointed to by <event>.

 *---------------------------------------------------------------*/

/* ARGSUSED */
static void Redisplay(gw,event,region)
Widget gw;
XEvent *event;
Region region;
{
	ChartWidget w;

	w = (ChartWidget)gw;
	if (!XtIsRealized(w)) return;
	ChartRedraw(w,event->xexpose.x,event->xexpose.y,
		event->xexpose.width,event->xexpose.height);
} /* End Redisplay */



/*---------------------------------------------------------------*

	ChartRedraw(chart,x,y,w,h)

	This routine redraws the rectangle <x,y,w,h> of the
	widget <chart>.

 *---------------------------------------------------------------*/

ChartRedraw(chart,x,y,w,h)
ChartWidget chart;
int x,y,w,h;
{
	Display *dpy;
	Window *win;
	int first,last,top,bot,line,size;
	float value;

	if (!XtIsRealized(chart)) return;
	XFillRectangle(XtDisplay(chart),XtWindow(chart),
		chart->chart.eraserGC,x,y,w,h);
	InfoRedraw(chart,x,y,w,h);
	MarkDataRedraw(chart,x,y,w,h);
	TitleRedraw(chart,x,y,w,h);
} /* End ChartRedraw */


/*---------------------------------------------------------------*

	InfoRedraw(chart,x,y,w,h)

	This function redraws the info line portion of
	the widget <chart> if it falls within <x,y,w,h>.

 *---------------------------------------------------------------*/

InfoRedraw(chart,x,y,w,h)
ChartWidget chart;
int x,y,w,h;
{
	char buffer[80];

	if (!XtIsRealized(chart)) return;
	if (chart->chart.data_h <= 0)
	{
		XFillRectangle(XtDisplay(chart),XtWindow(chart),
			       chart->chart.textGC,x,y,w,h);
		return;
	}
	if (chart->chart.infoline)
	{
		sprintf(buffer,"Maximum: %d  Scale: %d",
			chart->chart.max_value,chart->chart.scale);
		XFillRectangle(XtDisplay(chart),XtWindow(chart),
			       chart->chart.eraserGC,
			       0,chart->chart.info_y,w,
			       chart->chart.mark_y - chart->chart.info_y);
		DrawCenteredText(chart,0,chart->chart.textGC,buffer);
	}
} /* End InfoRedraw */


/*---------------------------------------------------------------*

	TitleRedraw(chart,x,y,w,h)

	This function redraws the title line portion of
	the widget <chart> if it falls within <x,y,w,h>.

 *---------------------------------------------------------------*/

TitleRedraw(chart,x,y,w,h)
ChartWidget chart;
int x,y,w,h;
{
	if (!XtIsRealized(chart)) return;
	if (chart->chart.data_h <= 0)
	{
		XFillRectangle(XtDisplay(chart),XtWindow(chart),
			       chart->chart.textGC,x,y,w,h);
		return;
	}
	if (chart->chart.chart_title)
	{
		DrawCenteredText(chart,1,chart->chart.textGC,
			chart->chart.chart_title);
	}
} /* End TitleRedraw */


/*---------------------------------------------------------------*

	MarkDataRedraw(wid,x,y,w,h)

	This function redraws the marker and data line
	portion of the widget <wid>.  It tries to
	optimize the draw to only draw within the
	rectangle <x,y,w,h>.

 *---------------------------------------------------------------*/
	
MarkDataRedraw(wid,x,y,w,h)
ChartWidget wid;
int x,y,w,h;
{
	Display *dpy;
	Window win;
	int top,bot,line,first,last,size;
	int value;

	if (!XtIsRealized(wid)) return;
	if (wid->chart.data_h <= 0)
	{
		XFillRectangle(XtDisplay(wid),XtWindow(wid),
			       wid->chart.textGC,x,y,w,h);
		return;
	}
	if (wid->chart.last_filled != -1)
	{
		dpy = XtDisplay(wid);
		win = XtWindow(wid);
		bot = wid->chart.title_y - 1;
		XDrawLine(dpy,win,wid->chart.markGC,wid->chart.marker,
			wid->chart.mark_y,wid->chart.marker,bot);
		first = x;
		last = min(x + w - 1,wid->chart.last_filled);
		for (line = first; line <= last; line++)
		{
			value = (int)(wid->chart.value_array[line]);
			if (wid->chart.max_value == 0)
				size = 0;
			    else
				size = (int)((value / (float)
					(wid->chart.scale))
					* (float)wid->chart.data_h);
			size = min(size,wid->chart.data_h);
			top = max(y,wid->chart.title_y - 1 - size);
			XDrawLine(dpy,win,wid->chart.dataGC,
				line,top,line,bot);
		}
		XDrawLine(dpy,win,wid->chart.markGC,wid->chart.marker,
			wid->chart.mark_y,wid->chart.marker,bot);
	}
} /* End MarkDataRedraw */



/*---------------------------------------------------------------*

	MarkDataLineRedraw(wid,lineno)

	This routine redraws the chart line with
	index <lineno> in the chart widget <wid>.
	The first line is 0.

 *---------------------------------------------------------------*/

MarkDataLineRedraw(wid,lineno)
ChartWidget wid;
int lineno;
{
	Display *dpy;
	Window win;
	int top,bot,line,first,last,size;
	float value;

	if (!XtIsRealized(wid)) return;
	if (wid->chart.last_filled != -1)
	{
		dpy = XtDisplay(wid);
		win = XtWindow(wid);
		bot = wid->chart.title_y - 1;
		if (lineno == wid->chart.marker)
		{
			XDrawLine(dpy,win,wid->chart.markGC,
				wid->chart.marker,wid->chart.mark_y,
				wid->chart.marker,wid->chart.title_y - 1);
		}
		    else
		{
			if (lineno <= wid->chart.last_filled)
				value = (int)
					(wid->chart.value_array[lineno]);
			    else
				value = 0;
			if (wid->chart.max_value == 0)
				size = 0;
			    else
				size = (int)((value / (float)
					(wid->chart.scale))
					* (float)wid->chart.data_h);
			size = min(size,wid->chart.data_h);
			top = wid->chart.title_y - 1 - size;
			XDrawLine(dpy,win,wid->chart.eraserGC,
				lineno,wid->chart.mark_y,
				lineno,top);
			XDrawLine(dpy,win,wid->chart.dataGC,
				lineno,top,lineno,wid->chart.title_y - 1);
		}
	}
} /* End MarkDataLineRedraw */



/*---------------------------------------------------------------*

	ChartAddValue(wid,value)

	This routine takes an integer value <value>
	and adds it to the chart <wid>.  Each time
	a new value is added, the chart is scaled
	to insure the new value fits within it.  The
	chart is redrawn appropriately.  The value
	is interpreted as an unsigned integer.

 *---------------------------------------------------------------*/

extern void ChartAddValue(wid,value)
ChartWidget wid;
unsigned int value;
{
	int i,maximum,old_marker;

	if (!XtIsRealized(wid)) return;
	old_marker = wid->chart.marker;
	wid->chart.value_array[wid->chart.marker] = value;	
	wid->chart.last_filled = max(wid->chart.last_filled,wid->chart.marker);
	++ wid->chart.marker;
	if (wid->chart.marker >= wid->chart.array_size) wid->chart.marker = 0;
	maximum = 0;
	for (i = 0; i <= wid->chart.last_filled; i++)
		maximum = max(maximum,wid->chart.value_array[i]);
	if (maximum != wid->chart.max_value)
	{
		InfoRedraw(wid,wid->core.x,wid->core.y,
			wid->core.width,wid->core.height);
		wid->chart.max_value = maximum;
	}
	if (maximum != 0 && (maximum < wid->chart.scale / 2 ||
		maximum >= wid->chart.scale))
	{
		wid->chart.scale = (4 * maximum) / 3;
		ChartRedraw(wid,0,0,wid->core.width,wid->core.height);
	}
	    else
	{
		MarkDataLineRedraw(wid,old_marker);
		MarkDataLineRedraw(wid,wid->chart.marker);
	}
} /* End ChartAddValue */



/* ARGSUSED */
static Boolean SetValues(gcurrent,grequest,gnew)
Widget gcurrent,grequest,gnew;
{
	ChartWidget current,new;
	XtGCMask valuemask;
	Boolean redisplay;
	XGCValues myXGCV;
	int info_h,title_h,wid_h,wid_w;

	current = (ChartWidget)gcurrent;
	new = (ChartWidget)gnew;
	if (current->chart.chart_title != new->chart.chart_title)
	{
		new->chart.chart_title = strcpy(XtMalloc((unsigned)
			XtStrlen(new->chart.chart_title) + 1),
	                new->chart.chart_title);
		if (current->chart.chart_title != NULL)	
	                XtFree((char *)current->chart.chart_title);
		Resize(new);
		redisplay = TRUE;
	}
	if (new->chart.font != current->chart.font)
	{
		Resize(new);
		redisplay = TRUE;
	}
	if ((new->chart.data_pixel != current->chart.data_pixel) ||
		(new->core.background_pixel != current->core.background_pixel))
	{
		valuemask = GCForeground | GCBackground | GCLineWidth;
		myXGCV.foreground = new->chart.data_pixel;
		myXGCV.background = new->core.background_pixel;
		myXGCV.line_width = 0;
		XtDestroyGC(current->chart.dataGC);
		new->chart.dataGC = XtGetGC(gcurrent,valuemask,&myXGCV);
		redisplay = TRUE;
	}
	if ((new->chart.text_pixel != current->chart.text_pixel) ||
		(new->core.background_pixel != current->core.background_pixel))
	{
		valuemask = GCForeground | GCBackground | GCFont;
		myXGCV.foreground = new->chart.text_pixel;
		myXGCV.background = new->core.background_pixel;
		myXGCV.font = new->chart.font->fid;
		XtDestroyGC(current->chart.textGC);
		new->chart.textGC = XtGetGC(gcurrent,valuemask,&myXGCV);
		redisplay = TRUE;
	}
	if ((new->chart.mark_pixel != current->chart.mark_pixel) ||
		(new->core.background_pixel != current->core.background_pixel))
	{
		valuemask = GCForeground | GCBackground | GCLineWidth;
		myXGCV.foreground = new->chart.mark_pixel;
		myXGCV.background = new->core.background_pixel;
		myXGCV.line_width = 0;
		XtDestroyGC(current->chart.markGC);
		new->chart.markGC = XtGetGC(gcurrent,valuemask,&myXGCV);
		redisplay = TRUE;
	}
	if (new->core.background_pixel != current->core.background_pixel)
	{
		valuemask = GCBackground | GCLineWidth;
		myXGCV.foreground = new->core.background_pixel;
		myXGCV.background = new->core.background_pixel;
		myXGCV.line_width = 0;
		XtDestroyGC(current->chart.eraserGC);
		new->chart.eraserGC = XtGetGC(gcurrent,valuemask,&myXGCV);
		redisplay = TRUE;
	}
			wid_w = new->core.width;
			wid_w = max(wid_w,MINCHARTWIDTH);
	if ((new->core.height != current->core.height) ||
		(new->core.width != current->core.width))
	{
		Snap(new);
		redisplay = TRUE;
	}
	return (redisplay);
} /* End SetValues */

/************************************************************************/
/* ChartSelect() processes button down event.				*/
/************************************************************************/
static void ChartSelect(w, event)
ChartWidget		w;
XButtonPressedEvent	*event;
{
#ifdef DEBUG
printf("ChartSelect\n");
#endif
  w->chart.value_select = w->chart.value_array[w->chart.marker-1];
#ifdef DEBUG
printf ("ChartSelect marker %d value: %d\n", w->chart.marker-1,
				w->chart.value_select);
#endif

} /* ChartSelect */

/************************************************************************/
/* ChartNotify() processes button up event.				*/
/************************************************************************/
static void ChartNotify(w, event)
ChartWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("ChartNotify\n");
#endif
  XtCallCallbacks(w, XtNselect, (caddr_t) w->chart.value_select);

} /* ChartNotify */
