/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Authors: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *                       Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/contour/RCS/Contour.c,v 1.9 1994/02/25 04:45:48 aydt Exp $
 */

/* #define DEBUG  */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/keysymdef.h>
#include "ContourP.h"

/*============================================================================*
 * 	Macros used by the Contour widget
 *===========================================================================*/

#define FontHt(f)  ( f->max_bounds.ascent + f->max_bounds.descent + FONTPAD )

#define PlaceX(x) ( (x)*((float)w->contour.sizex) + (float)w->contour.offsetx )

#define PlaceY(y) ( (y)*((float)w->contour.sizey) + (float)w->contour.offsety )

#define abs(A)	   ( (A)>=0.0 ? (A) : -(A) )

/*============================================================================*
 * 	Resource list
 *============================================================================*/

static XtResource resources[] = {
  	/* core resources */
  	{ XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     	   XtOffset(ContourWidget, core.width), XtRString, "200" },
  	{ XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     	   XtOffset(ContourWidget, core.height), XtRString, "200" },
  	{ XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
     	   XtOffset(ContourWidget, core.background_pixel), XtRString, "white"},
  	/* contour resources */
  	{ XtNcolumns, XtCColumns, XtRInt, sizeof(int), 
     	   XtOffset(ContourWidget, contour.columns), XtRString, "10" },
  	{ XtNcontourCnt, XtCContourCnt, XtRInt, sizeof(int),
     	   XtOffset(ContourWidget, contour.contour_cnt), XtRString, "10" },
  	{ XtNfont, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
     	   XtOffset(ContourWidget, contour.font), XtRString, "fixed" },
  	{ XtNgridColor, XtCGridColor, XtRPixel, sizeof(Pixel),
     	   XtOffset(ContourWidget, contour.grid_color), XtRString, "blue" },
  	{ XtNrows, XtCRows, XtRInt, sizeof(int), 
     	   XtOffset(ContourWidget, contour.rows), XtRString, "10" },
  	{ XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     	   XtOffset(ContourWidget, contour.select), 
	   XtRCallback, (caddr_t) NULL },
  	{ XtNshowContours, XtCShowContours, XtRBoolean, sizeof (Boolean),
     	   XtOffset(ContourWidget, contour.show_contours), XtRString, "FALSE"},
  	{ XtNshowGrid, XtCShowGrid, XtRBoolean, sizeof (Boolean),
     	   XtOffset(ContourWidget, contour.show_grid), XtRString, "TRUE" },
  	{ XtNshowRange, XtCShowRange, XtRBoolean, sizeof (Boolean),
     	   XtOffset(ContourWidget, contour.show_range), XtRString, "TRUE" },
        { XtNshowRangeAbove, XtCShowRangeAbove, XtRBoolean, sizeof(Boolean), 
           XtOffset(ContourWidget, contour.show_range_above),XtRString,"TRUE"},
  	{ XtNtextColor, XtCTextColor, XtRPixel, sizeof(Pixel),
     	   XtOffset(ContourWidget, contour.text_color), XtRString, "black" },
};

/*============================================================================*
 * 	Action function declarations. 
 * 	Default translation table.
 * 	Actions table.
 *============================================================================*/

static void	ContourSelect(), ContourNotify();

static char defaultTranslations[] = 
  	"<Btn1Down>:	ContourSelect()		\n\
  	 <Btn1Up>:	ContourNotify()";

static XtActionsRec	actions[] = {
  	{ "ContourSelect", (XtActionProc) ContourSelect },
  	{ "ContourNotify", (XtActionProc) ContourNotify },
};

/*============================================================================*
 * 	Function declarations 
 *============================================================================*/
	
	/* Core methods 	    */
static void	ClassInitialize();
static void	Destroy();
static void	Initialize();
static void	Redisplay();
static void 	Resize();
static Boolean	SetValues();

	/* Contour private functions */
static void	DrawContour();
static void	DrawContourPlot();
static void	DrawGrid();
static void	DrawLine();
static void	DrawRange();
static void	FreeColorGCs();
static void	Realloc();
static void	SelectContours();

	/* Contour private function for debuggging */
static void	DrawContourValues();
	
/*============================================================================*
 * 	Static allocation and initialization of Contour class record
 *      Definition and initialization of class record pointer
 *============================================================================*/

ContourClassRec contourClassRec = {

	/* Core part */
	{
  	   &widgetClassRec,		/* superclass			*/
  	   "Contour",			/* class_name			*/
  	   sizeof(ContourRec),		/* widget_size			*/
  	   ClassInitialize,		/* class_initialize		*/
  	   NULL,			/* class_part_initialize 	*/
  	   FALSE,			/* class_inited			*/
  	   Initialize,			/* initialize			*/
  	   NULL,			/* initialize_hook		*/
  	   XtInheritRealize,		/* realize			*/
  	   actions,			/* actions			*/
  	   XtNumber(actions),		/* num_actions			*/
  	   resources,			/* resources			*/
  	   XtNumber(resources),		/* resource_count		*/
  	   NULL,			/* xrm_class			*/
  	   TRUE,			/* compress_motion		*/
  	   TRUE,			/* compress_exposure		*/
  	   TRUE,			/* compress_enterleave		*/
  	   FALSE,			/* visible_interest		*/
  	   Destroy,			/* destroy			*/
  	   Resize,			/* resize			*/
  	   Redisplay,			/* expose			*/
  	   SetValues,			/* set_values			*/
  	   NULL,			/* set_values_hook		*/
  	   XtInheritSetValuesAlmost,	/* set_values_almost		*/
  	   NULL,			/* get_values_hook		*/
  	   NULL,			/* accept_focus			*/
  	   XtVersion,			/* version			*/
  	   NULL,			/* callback_private		*/
  	   defaultTranslations,		/* tm_table			*/
  	   XtInheritQueryGeometry,	/* query_geometry          	*/
  	   XtInheritDisplayAccelerator,	/* display_accelerator    	*/
  	   NULL,			/* extension               	*/
	},
	/* Contour part */
	{
	   0,				/* dummy 			*/
	}
};
	
WidgetClass contourWidgetClass = (WidgetClass) &contourClassRec;


/*============================================================================*
 * 	Function definitions for Core methods
 *============================================================================*/

/*----------------------------------------------------------------------------*
 *	static void ClassInitialize()
 *
 *	This method is called once by the X toolkit, the first time an
 *	instance of the Contour class is created by the application.
 *	The class is only initialized once, and is shared among all
 *	instances of the class.  
 *	
 *	We use this to register non-standard type converters for the 
 *	Contour class.
 *----------------------------------------------------------------------------*/

static void ClassInitialize()
{
  	XtAddConverter( XtRString, XtRBackingStore, XmuCvtStringToBackingStore,
		 NULL, 0);

} /* ClassInitialize() */


/*----------------------------------------------------------------------------*
 *	static void Initialize( request, new )
 *
 *	This method is called by the X toolkit to initialize the widget's
 *	instance variables.  It checks to be sure that public variables
 * 	have reasonable values and sets initial values for private
 *	instance variables.  The widget will not be realized yet (no window 
 *	will have been created) at the time this method is called.
 *
 *	The parameters <request> and <new> are widgets where <request>
 *	represents the widget before the superclass modified any of the
 *	variables, and <new> is the current state of the widget.  <request> 
 *	can be used to determine the original requested data.  The widget
 *	<new> will become the new widget.
 *----------------------------------------------------------------------------*/

static void Initialize( request, new )
ContourWidget request, new;
{
	int		i;
	XtGCMask	GCmask;
	XGCValues	GCvalues;
	char		msgbuf[256];

#	ifdef DEBUG
	fprintf( stderr,"ContourWidget: Initialize\n");
#	endif

	/* 
	 * Verify that the window size is not zero.  
	 * The Core Initialize() method doesn't do this.		
	 */
	if ( new->core.width == 0 )
	  new->core.width = MINSIZE;
	if ( new->core.height == 0 )
	  new->core.height = MINSIZE;

	/*
	 * Verify that we have reasonable values for all the public
	 * instance variables (resources).
	 */

	if ( new->contour.rows < 2 ) {
		XtWarning( "Contour: There must be at least 2 rows." );
		new->contour.rows = 2;
	}

	if ( new->contour.rows > MAXROWS ) {
		sprintf( msgbuf, 
			 "Contour: Number of rows cannot exceed %d", MAXROWS );
		XtWarning( msgbuf );
		new->contour.rows = MAXROWS;
	}

	if ( new->contour.columns < 2 ) {
		XtWarning( "Contour: There must be at least 2 columns" );
		new->contour.columns = 2;
	}
		
	if ( new->contour.columns > MAXCOLUMNS ) {
		sprintf( msgbuf, "Contour: Number of columns cannot exceed %d", 
				 MAXCOLUMNS );
		XtWarning( msgbuf );
		new->contour.columns = MAXCOLUMNS;
	}

	/*
	 * Initialize the private instance (state) variables, including
	 * the graphics contexts. So far the application hasn't has a
	 * chance to set the contour colors. For now we put a single 
	 * color in the contourGCs and set it to be the same as the 
	 * grid color.
	 */
	new->contour.values = NULL;
  
	new->contour.numcolors = 1;
	new->contour.contourGCs = 
		      (GC*) XtMalloc( new->contour.numcolors * sizeof(GC) );
	GCmask = GCForeground | GCBackground;
	GCvalues.foreground = new->contour.grid_color;
	GCvalues.background = new->core.background_pixel;
	new->contour.contourGCs[0] = XtGetGC( (Widget)new, GCmask, &GCvalues );

	GCmask = GCForeground | GCBackground;
	GCvalues.foreground = new->contour.grid_color;
	GCvalues.background = new->core.background_pixel;
	new->contour.gridGC = XtGetGC( (Widget)new, GCmask, &GCvalues );

  	GCmask = GCForeground | GCBackground | GCFont | GCLineWidth;
  	GCvalues.foreground = new->contour.text_color;
  	GCvalues.background = new->core.background_pixel;
        if ( new->contour.font != NULL ) {
    		GCvalues.font = new->contour.font->fid;
  	} else {
    		GCmask &= ~GCFont;		/* use server default font */
	}
  	GCvalues.line_width = 0;
  	new->contour.textGC = XtGetGC( (Widget)new, GCmask, &GCvalues );

	/* 
	 * We call Resize here to set up state variables that relate to
	 * the display.  If Realize() were a local method, it would be
	 * better to call Resize() from there.  But, since we inherit the
	 * Realize() method, we do it here so that things are set properly
	 * when it does come time to do the drawing. 
	 */

	Resize( new );

} /* Initialize */


/*----------------------------------------------------------------------------*
	static void Destroy()

	This method is called by the X toolkit before the Contour widget 
	is destroyed.  
	
	We free space allocated to hold values, release graphics contexts,
	and remove our callbacks.
 *----------------------------------------------------------------------------*/

static void Destroy( w )
ContourWidget w;
{
#	ifdef DEBUG
	fprintf( stderr, "ContourWidget: Destroy\n" );
#	endif
	
	if ( w->contour.values != NULL ) {
		XtFree( w->contour.values );
	}

  	FreeColorGCs( w );
  	XtReleaseGC( w, w->contour.gridGC );
  	XtReleaseGC( w, w->contour.textGC );

  	XtRemoveAllCallbacks( w, XtNselect );

} /* Destroy */


/*----------------------------------------------------------------------------*
 *	static void Resize( w )
 *
 *	This method is called when the parent widget resizes the widget.
 *	It is also called by Contour::Initialize and Contour::SetValues() to 
 *	update sizing-related when other instance variables they depend 
 *	on have changed.
 *
 *	Resize() recalculates the sizing-related state variables based on the 
 *	position and size of the window and characteristics of the contour.
 *
 *	Note that we have (columns) x-axes and (rows) y-axes.  That means we
 *	are (columns - 1) elements wide and (rows - 1) elements high, where
 *	an element is the space between the axes.
 *----------------------------------------------------------------------------*/

static void Resize( w ) 
ContourWidget	w;

{
	int	eleh, elew;
	int	rangeh;

#	ifdef DEBUG
	fprintf( stderr, "ContourWidget: Resize\n" );
#	endif

	/*
	 * We pad all sides of the contour with PADDING.  If we will show
	 * the range of values, also allow additional FontHt + PADDING
	 * on the bottom (or top). Adjust the grid to be centered within those 
	 * boundaries. 
	 */
        if ( (w->contour.show_range)&&(w->contour.show_range_above) ) {
           rangeh = 2 * (w->contour.font->max_bounds.ascent + PADDING);
        } else if ( w->contour.show_range ) {
 	   rangeh = FontHt( w->contour.font ) + PADDING;
	} else {
  	   rangeh = 0;
	}

	elew = ( w->core.width - 2*PADDING ) / ( w->contour.columns - 1 );
	eleh = (w->core.height - 2*PADDING - rangeh ) / ( w->contour.rows - 1);

	w->contour.sizex = elew * (w->contour.columns - 1 );
        w->contour.sizey = eleh * (w->contour.rows - 1);

	w->contour.offsetx = ( w->core.width - w->contour.sizex ) / 2;
	if ( (w->contour.show_range)&&(w->contour.show_range_above) ) {
           w->contour.offsety = (w->core.height - w->contour.sizey - PADDING);
        } else { 
           w->contour.offsety = (w->core.height - w->contour.sizey - rangeh)/2;
        }

} /* Resize */


/*----------------------------------------------------------------------------*
 *	static void Redisplay( w, event, region )
 *	
 *	This method is responsible for redrawing the widget whenever an Expose
 *	event arrives from the server. It may be called before the
 *	widget is realized, so we check that.
 *----------------------------------------------------------------------------*/

static void Redisplay( w, event, region )
ContourWidget	w;
XEvent		*event;
Region		region;
{
#	ifdef DEBUG
	fprintf( stderr, "ContourWidget: Redisplay\n" );
#	endif

	if (  XtIsRealized( w ) && w->core.visible  ) {
		DrawContour( w );
	}

} /* Redisplay */


/*----------------------------------------------------------------------------*
 *	static Boolean SetValues( current, request, new )
 *	
 *	This method is called when the resource manager initializes the 
 *	widget's resources, or whenever the application calls XtSetValues to
 *	set the resources of the widget.  The method recalculates private
 *	state variables based on the new public instance variable values.
 *
 *	<current> contains the previous, unaltered state of the widget.
 *	<request> contains the values requested for the widget by a combination
 *	of the user's resource files, the widget default resources, and the 
 *	application.  <new> contains the state of the widget after all 
 *	superclass's SetValues() methods have been called.
 *----------------------------------------------------------------------------*/

static Boolean SetValues( current, request, new )
ContourWidget	current, request, new;
{
	int		i;
	int		currentLimit, newLimit;
	XtGCMask	GCmask;
	XGCValues	GCvalues;
	Boolean		reSize = FALSE;
	Boolean		reDraw = FALSE;
	Boolean		reAlloc = FALSE;
	char		msgbuf[256];

#ifdef DEBUG
	fprintf( stderr, "ContourWidget: SetValues\n" );
#endif
	/* change of width or height */
	if ( (new->core.width != current->core.width)  
	    || (new->core.height != current->core.height) ) {
	  	reSize = TRUE;
		reDraw = TRUE;
	}

	/* change show_contours */
	if ( new->contour.show_contours != current->contour.show_contours ) {
		reDraw = TRUE;
	}

	/* change show_grid */
	if ( new->contour.show_grid != current->contour.show_grid ) {
		reDraw = TRUE;
	}

	/* change show_range */
	if ( new->contour.show_range != current->contour.show_range ) {
		reSize = TRUE;
		reDraw = TRUE;
	}

	/* change grid color */
	if ( (new->contour.grid_color != current->contour.grid_color)
	    || (new->core.background_pixel != current->core.background_pixel) ){
	  	GCmask = GCForeground | GCBackground;
	  	GCvalues.foreground = new->contour.grid_color;
	  	GCvalues.background = new->core.background_pixel;
	  	XtReleaseGC( current, current->contour.gridGC );
	  	new->contour.gridGC = XtGetGC( (Widget)new, GCmask, &GCvalues);
	  	reDraw = TRUE;
	}
	
	/* change text color or font */
	if ( (new->contour.text_color != current->contour.text_color)
	    || (new->contour.font != current->contour.font)
	    || (new->core.background_pixel != current->core.background_pixel) ){
  		GCmask = GCForeground | GCBackground | GCFont | GCLineWidth;
  		GCvalues.foreground = new->contour.text_color;
  		GCvalues.background = new->core.background_pixel;
        	if ( new->contour.font != NULL ) {
    		    GCvalues.font = new->contour.font->fid;
  		} else {
    		    GCmask &= ~GCFont;		/* use server default font */
		}
  		GCvalues.line_width = 0;
	  	XtReleaseGC( current, current->contour.textGC );
	  	new->contour.textGC = XtGetGC( (Widget)new, GCmask, &GCvalues);

	        if ( new->contour.font != current->contour.font ) {
		    reSize = TRUE;		/* perhaps new font size   */
		}
	  	reDraw = TRUE;
	}
	
	/* change number of rows */
	if ( new->contour.rows != current->contour.rows ) {
		if ( new->contour.rows < 2 ) {
		    XtWarning( "Contour: There must be at least 2 rows." );
		    new->contour.rows = 2;
		}
		if ( new->contour.rows > MAXROWS ) {
		    sprintf( msgbuf, 
			     "Contour: Number of rows cannot exceed %d", 
			     MAXROWS );
		    XtWarning( msgbuf );
		    new->contour.rows = MAXROWS;
		}
		reAlloc = TRUE;
		reSize = TRUE;
		reDraw = TRUE;
	}

	/* change number of columns */
	if ( new->contour.columns != current->contour.columns ) {
		if ( new->contour.columns < 2 ) {
		    XtWarning( "Contour: There must be at least 2 columns." );
		    new->contour.columns = 2;
		}
		if ( new->contour.columns > MAXCOLUMNS ) {
		    sprintf( msgbuf, 
			     "Contour: Number of columns cannot exceed %d", 
			     MAXCOLUMNS );
		    XtWarning( msgbuf );
		    new->contour.columns = MAXCOLUMNS;
		}
		reAlloc = TRUE;
		reSize = TRUE;
		reDraw = TRUE;
	}

	/* change number of contours */
	if ( new->contour.contour_cnt != current->contour.contour_cnt ) {
		reDraw = TRUE;
	}

	/* reallocate array space, resize, and redraw as necessary */
	if ( reAlloc )
		Realloc( current, new );

	if ( reSize ) 
		Resize( new );

	return ( reDraw ); 
	
} /* SetValues */


/*============================================================================*
 * 	Function definitions for private methods
 *============================================================================*/

/*----------------------------------------------------------------------------*
 *	static void DrawContour( w );
 *
 *	DrawContour is called by Redisplay(), ContourSetValues(), and 
 *	ContourSetValue() to redraw the entire contour display.
 *
 *	It clears the window, and if there are values set, calls
 *	SelectContours() and DrawContourPlot() to display the contour 
 *	lines. DrawGrid and DrawRange may also be called depending on 
 *	their respective 'show' instance variables.
 *----------------------------------------------------------------------------*/

static void DrawContour( w )
ContourWidget	w;
{
	int	i; 
	float	min, max;

#ifdef DEBUG
	fprintf( stderr, "ContourWidget: DrawContour\n" );
#endif

	XClearWindow( XtDisplay(w), XtWindow(w) );

	if ( w->contour.show_grid ) {
		DrawGrid( w );
	}

	if ( w->contour.values != NULL ) {
		SelectContours( w, &min, &max );
		for ( i = 0; i < w->contour.contour_cnt; i++ ) {
		    DrawContourPlot( w, i );
		}
		if ( w->contour.show_range ) {
		    DrawRange( w, min, max );
		}
		/*** Remove this comment to display list of contour values 
		 *** for debugging 
		 DrawContourValues( w );
		 ******/
	}

} /* DrawContour */

/*----------------------------------------------------------------------------*
 *	static void DrawContourPlot( w, cnumber );
 *
 * This is the main subroutine which actually traces the given contour
 * and does the plotting.  
 *
 * < w > is the contour widget instance
 * < cnumber > is the number of the contour in the contours array we are 
 *	       plotting.
 *
 * I use the following method for finding the contours:
 *
 *	* Consider a single cell which has four data values at its
 *	  corners.
 *
 *	* Add a center vertex to the cell whose value is the average
 *	  of the corners.
 *
 *	* Connect all vertices to form 8 edges.
 *
 *	* Linearly interpolate along each edge and find a contour crossing
 *	  point for the edge, if such a point exists.
 *
 *	* Connect all such contour crossing points for all edges of the
 *	  cell.
 *
 * Some notes follow on the implementation details of the algorithm.
 *
 * We trace contours in a counter-clockwise fashion.
 * We scan a cell which has the following vertices:
 *
 *	r,c    r,c+1
 *	*------*
 *	|      |
 *      |      |
 *      *------*
 *	r+1,c  r+1,c+1
 *
 * We also compute the average value for the center of the
 * cell and number the resulting eight edges as follows:
 *
 *	*---6--*
 *	|\    /|
 *      | 7  5 |
 *	|  \/  |
 *	0  /\  4
 *	| 1  3 |
 *	|/    \|
 *      *---2--*
 *
 * vlist is a list of vertex offsets from r,c corresponding to a given
 * edge.  An offset of 0.5, 0.5 refers to the center.  For example,
 * edge n for cell (r,c) is an edge from (c+vlist[n].x1, r+vlist[n].y1)
 * to (c+vlist[n].x2, r+vlist[n].y2)
 *
 * In looking at the following code, it is important to remember that
 * in X-windows the origin (0,0) is in the upper leftmost corner of the
 * display.  X increases left to right.  Y increases top to bottom.
 * And - as X increases we span COLUMNS in our contour array.  As Y
 * increases, we span ROWS.  This is not intuitive!
 *----------------------------------------------------------------------------*/

struct vlist {
	float x1, y1,
	      x2, y2;
} vlist[] = {
	{0.0, 0.0, 0.0, 1.0},
	{0.0, 1.0, 0.5, 0.5},
	{0.0, 1.0, 1.0, 1.0},
	{1.0, 1.0, 0.5, 0.5},
	{1.0, 1.0, 1.0, 0.0},
	{1.0, 0.0, 0.5, 0.5},
	{1.0, 0.0, 0.0, 0.0},
	{0.0, 0.0, 0.5, 0.5}};

static void DrawContourPlot( w, cnumber )
ContourWidget	w;
int		cnumber;
{
	Boolean	ShowContour;	/* Show contour for this line?	            */
	int	r, c;		/* array indices into vlist		    */
	int	e, scan_e;	/* edge indices                             */
	int	tw;		/* text width for label 		    */
	int	lx, ly;		/* x and y coordinates for label            */
	float	avg;		/* cell center average 			    */
	float	dx, dy;		/* fraction this cell is of total display   */
	float	x, y; 		/* coordinates we are unsure we'll need     */
	float	t;		/* distance along some edge where a contour 
				 * intersection occurs                      */
	float	p1, p2;		/* difference between 2 vertex values at the
				 * ends of an edge and the contour value    */
	float	x1, y1, x2, y2;	/* cell edges indices (copied out of vlist) */
	char	label[128];	/* contour label string */

	/* Local copies of variables we use frequently */

	float	*values	= w->contour.values;
	float	cvalue = w->contour.contours[cnumber];
	int	columns = w->contour.columns;


#	ifdef DEBUG
	printf ("Contour Widget: DrawContourPlot\n");
#	endif

	ShowContour = w->contour.show_contours;

	dx = 1.0 / ( (float)columns - 1 );
	dy = 1.0 / ( (float)w->contour.rows - 1 );

	for ( r = 0; r < w->contour.rows-1; r++ ) {
#	   ifdef DEBUG
	   fprintf( stderr, "\nDrawContourPlot: Row is %d\n", r );
#	   endif

	   for ( c = 0; c < columns-1; c++ ) {
#	      ifdef DEBUG
	      fprintf( stderr, "DrawContourPlot: Column is %d\n", c );
#	      endif

	      /* compute the cell center avg */
	      avg = ( values[r*columns + c] + values[(r+1)*columns + c] 
		      + values[(r+1)*columns + (c+1)] 
		      + values[r*columns + (c+1)] )
		      / 4.0;

	      /* look at all the cell faces for a starting point for a 
	       * contour intersection 					*/
	      for ( e = 0; e < 8; e += 2 ) {
#	         ifdef DEBUG
	         fprintf( stderr, "DrawContourPlot: Edge is %d\n", e );
#	         endif

		 /* get 'coordinates' of the cell edge */
		 x1 = c + vlist[e].x1;
		 y1 = r + vlist[e].y1;
		 x2 = c + vlist[e].x2;
		 y2 = r + vlist[e].y2;

		 /* p1 and p2 are the differences between the field values 
		  * at either end of the edge and the contour value.  We 
		  * actually look for zero crossings. Because of the way 
		  * vlist is arranged, (r1,y1) where p1 is the value will
		  * always be a corner.  (x2,y2) where p2 is the value may
		  * be a corner or the center depending on the edge.	   */

	  	 p1 = values[(int)y1*columns + (int)x1] - cvalue;
		 if ( y2 == (float)((int)y2) ) {	     /* a corner   */
		    p2 = values[(int)y2*columns + (int)x2] - cvalue;
		 } else {				     /* the center */
		    p2 = avg - cvalue;
		 }

		 /* if no contour intersection, continue with next edge    */
		 if ( (p1 * p2) > 0.0 ) {
		    continue;
		 }

		 /* Special case: contour runs along the edge.           
		  * Plot the whole edge. (This results in funny looking 
		  * contour plots if we have a region of constant value 
		  * equal to a contour we want to plot)                   */

		 if ( abs(p2 - p1) < ( 1e-6 ) ) {
		    w->contour.xmove = PlaceX( dx * x1 );
		    w->contour.ymove = PlaceY( dy * y1 );

		    w->contour.xloc = PlaceX( dx * x2 );
		    w->contour.yloc = PlaceY( dy * y2 );
		    DrawLine( w, cnumber );

		    continue;
		 }

		 /* Normal case: compute the contour intersection point  */
		 t = p2 / (p2 - p1);
		 x = PlaceX( dx * ( x1*t + x2*(1.0-t) ) );
		 y = PlaceY( dy * ( y1*t + y2*(1.0-t) ) );

		 /* Check to see if the next cell edge in the search path 
		  * has a contour intersection.  If it doesn't, we don't 
		  * follow this contour. It is the ending point of a contour 
		  * we will come to later. 				 */

		 x1 = c + vlist[(e+1)%8].x1;
		 y1 = r + vlist[(e+1)%8].y1;
		 x2 = c + vlist[(e+1)%8].x2;
		 y2 = r + vlist[(e+1)%8].y2;

	  	 p1 = values[(int)y1*columns + (int)x1] - cvalue;
		 if ( y2 == (float)((int)y2) ) {	     /* a corner   */
		    p2 = values[(int)y2*columns + (int)x2] - cvalue;
		 } else {				     /* the center */
		    p2 = avg - cvalue;
		 }
		 if ( (p1 * p2) > 0.0) {
		    continue;
		 }

		 /* If we get this far, this isn't an ending point and
		  * we start following the contour.                       */
		 w->contour.xmove = x;
		 w->contour.ymove = y;

		 /* Should we label the contour here? */
		 if ( ShowContour ) {
	   	    sprintf( label, "%.3f", cvalue );
   		    tw = XTextWidth( w->contour.font, label, strlen(label) );

	            /* We attempt to position label so that it is in the
		     * display area.  If there is space, we put it to the
		     * right and even with the point. We know that the
		     * label is numeric so we don't have to worry about 
		     * descent space.                                     */

		    if ( (x + tw + 1) < 
				(w->contour.offsetx + w->contour.sizex) ) {
		       lx = x + 0.5;	         /* label starts at pt    */
		       if ( (int)lx <= w->contour.offsetx ) {
			  lx = w->contour.offsetx + 1;
		       }
		    } else {
		       lx = x - tw;		 /* label ends at pt      */
		    }

		    ly = y + w->contour.font->max_bounds.ascent + 1;
		    if ( y >= (w->contour.offsety + w->contour.sizey) ) {
		       ly = w->contour.offsety + w->contour.sizey - 1;
		    }

       	            XDrawString( XtDisplay(w), XtWindow(w), w->contour.textGC, 
				 lx, ly, label, strlen(label) );
		    ShowContour = FALSE;
		 }

		 /* Follow the contour until we come to a contour 
		  * intersection on a cell face. We are on a cell face 
		  * if the edge number is even                           */

		 scan_e = e;

		 do {
		    scan_e = ( scan_e + 1 )%8;

		    x1 = c + vlist[scan_e].x1;
		    y1 = r + vlist[scan_e].y1;
		    x2 = c + vlist[scan_e].x2;
		    y2 = r + vlist[scan_e].y2;

	  	    p1 = values[(int)y1*columns + (int)x1] - cvalue;
		    if ( y2 == (float)((int)y2) ) {	     /* a corner   */
		       p2 = values[(int)y2*columns + (int)x2] - cvalue;
		    } else {				     /* the center */
		       p2 = avg - cvalue;
		    }

#	            ifdef DEBUG
	               fprintf( stderr, "DrawContourPlot: ScanEdge = %d\n", e );
		       fprintf( stderr, " Pt1 (%f,%f)=%f  Pt2 (%f,%f)=%f \n",
				  x1, y1, p1, x2, y2, p2 );
#	            endif

		    if ( abs(p2 - p1) < 1e-6 ) {
		       if ( (abs(p2) < 1e-6) && (abs( p1 ) < 1e-6) ) {
			  t = 1.0;
		       } else {
			  t = -1.0;
		       }
		    } else {
		       t = p2 / (p2 - p1);
		    }

		    if ( t >= 0.0 && t <= 1.0 ) {
		       w->contour.xloc = PlaceX( dx * ( x1*t + x2*(1.0-t) ) );
		       w->contour.yloc = PlaceY( dy * ( y1*t + y2*(1.0-t) ) );
		       DrawLine( w, cnumber );
	 	    }

	         } while ( ! ( (scan_e%2 == 0) && (t >= 0.0) && (t <= 1.0) ) );

	      }  /* for e */
	   }	 /* for c */
	}	 /* for r */

} /* DrawContourPlot */


/*----------------------------------------------------------------------------*
 *	static void DrawGrid( w );
 *
 *	DrawGrid is called by DrawContour() if show_grid is True 
 *	to draw the grid lines.  
 *----------------------------------------------------------------------------*/

static void DrawGrid( w )
ContourWidget	w;
{
	int 	i;
	int	eleh, elew;

	/* Local copies of variables */
	int 	rows = w->contour.rows;	
	int 	columns = w->contour.columns;
	int 	offsetx = w->contour.offsetx;
        int     offsety = w->contour.offsety;
	int	sizex = w->contour.sizex;
	int	sizey = w->contour.sizey;

#ifdef DEBUG
	fprintf( stderr, "ContourWidget: DrawGrid\n" );
#endif
	elew = sizex / ( columns - 1 );
        eleh = sizey / ( rows - 1);

	for ( i = 0; i < rows; i++ ) {
	    	XDrawLine( XtDisplay(w), XtWindow(w), w->contour.gridGC,
		           offsetx, offsety + i*eleh,
		           offsetx + sizex, offsety + i*eleh);
	}
	for( i = 0;  i < columns; i++ ) {
 	    	XDrawLine( XtDisplay(w), XtWindow(w), w->contour.gridGC,
		           offsetx + i*elew, offsety,
		           offsetx + i*elew, offsety + sizey );
	}

} /* Draw Grid */


/*----------------------------------------------------------------------------*
 *	static void DrawLine( w, cnumber );
 *
 *	DrawLine is called by DrawContourPlot() to draw the actual line
 *	that make up the plot.
 *
 * 	<cnumber> is the current contour number we are drawing.  It is used
 *	as an index into the contourGCs to select the appropriate color.
 *----------------------------------------------------------------------------*/

static void DrawLine( w, cnumber )
ContourWidget	w;
int		cnumber;
{
	int cidx;
      	int a1 = w->contour.xmove + 0.5;
      	int b1 = w->contour.ymove + 0.5;
      	int a2 = w->contour.xloc + 0.5;
      	int b2 = w->contour.yloc + 0.5;

#ifdef DEBUG
	fprintf( stderr, "ContourWidget: DrawLine (%d, %d) to (%d, %d) \n", 
		a1, b1, a2, b2 );
#endif
	if ( cnumber < w->contour.numcolors ) {
		cidx = cnumber;
	} else {
		cidx = w->contour.numcolors - 1;
	}

      	XDrawLine( XtDisplay(w), XtWindow(w), w->contour.contourGCs[cidx],
		   a1, b1, a2, b2);

        w->contour.xmove = w->contour.xloc;
       	w->contour.ymove = w->contour.yloc;

} /* DrawLine */


/*----------------------------------------------------------------------------*
 *	static void DrawRange(w, min, max );
 *
 *	DrawRange is called by DrawContour() if show_range is True 
 *	to display the Range of values.
 *----------------------------------------------------------------------------*/

static void DrawRange( w, min, max )
ContourWidget w;
float	      min, max;
{
	int	x, y;
	char	label[128], labelMin[128], labelMax[128];

#ifdef DEBUG
	fprintf ( stderr, "ContourWidget: DrawRange\n");
#endif
        if ( w->contour.show_range_above == TRUE ) {
           sprintf( labelMin, "Min: %.3f", min );
           x = (w->core.width - 
                XTextWidth(w->contour.font,labelMin,strlen(labelMin)) )/2;
           y = PADDING + w->contour.font->max_bounds.ascent;
           XDrawString( XtDisplay(w), XtWindow(w), w->contour.textGC, 
		        x, y, labelMin, strlen(labelMin) );
           sprintf( labelMax, "Max: %.3f", max );
           x = (w->core.width -
                XTextWidth(w->contour.font,labelMax,strlen(labelMax)) )/2;
           y = 2 * (PADDING + w->contour.font->max_bounds.ascent);
   	   XDrawString( XtDisplay(w), XtWindow(w), w->contour.textGC, 
		        x, y, labelMax, strlen(labelMax) );
        } else {
           sprintf( label, "Min: %.3f  Max: %.3f", min, max );
           y = w->contour.offsety + w->contour.sizey + PADDING 
					+ w->contour.font->max_bounds.ascent;
   	   XDrawString( XtDisplay(w), XtWindow(w), w->contour.textGC, 
		        w->contour.offsetx, y, label, strlen(label) );
        }
} /* DrawRange */

/*----------------------------------------------------------------------------*
 *      static void FreeColorGCs( w );
 *
 *      FreeColorGCs is called by other routines of the Contour widget
 *      to free the Color graphics contexts when appropriate.  It also
 *      sets numcolors to zero.
 *----------------------------------------------------------------------------*/

static void FreeColorGCs( w )
ContourWidget    w;
{
        int     i;

#       ifdef DEBUG
        printf( "ContourWidget: FreeColorGCs\n" );
#       endif

        if ( w->contour.numcolors == 0 ) {
                return;
        }

        for ( i = 0; i < w->contour.numcolors; i++ ) {
                XtReleaseGC( w, w->contour.contourGCs[i] );
        }
        XtFree( w->contour.contourGCs );
        w->contour.numcolors = 0;

} /* FreeColorGCs */


/*----------------------------------------------------------------------------*
 *	static void Realloc( old, new );
 *
 *	Realloc is called by Contour::SetValues when a new array dimension
 *	is seen.  It allocates space for the new contour, copies values
 *	from the old contour into the new, populates any "extra" cells with
 *	zeros, and releases the old contour space.
 *----------------------------------------------------------------------------*/

static void Realloc( old, new )
ContourWidget	old, new;
{
	int	r, c;
	int	oldr = old->contour.rows;	/* local copies of vars */
	int	oldc = old->contour.columns;
	int	newr = new->contour.rows;
	int	newc = new->contour.columns;
	
#ifdef DEBUG
	fprintf( stderr, "ContourWidget: Realloc\n" );
#endif

	if ( old->contour.values == NULL ) {	/* no values to copy    */
		return;
	}

  	new->contour.values = (float *) XtMalloc( newr * newc * sizeof(float) );

	for ( r = 0; r < newr; r++ ) {
		for ( c = 0; c < newc; c++ ) {
		    if ( r < oldr && c < oldc ) {
			new->contour.values[r * newc + c] = 
					old->contour.values[r * oldc + c];
		    } else {
			new->contour.values[r * newc + c] = 0.0;
		    }
		}
	}

	XtFree( old->contour.values );

} /* Realloc */

/*----------------------------------------------------------------------------*
 *	static void SelectContours( w, *min, *max )
 *
 *	SelectContours() is called by DrawContours to determine the
 *	min and max values, and build a contour list of contour_cnt
 *	entries which lie between the min and max.
 *----------------------------------------------------------------------------*/

static void SelectContours( w, minp, maxp )
ContourWidget	w;
float		*minp, *maxp;
{
	float	min, max;
	int	i, j;			/* array indices */
	float	t;			/* a parameter */

	float	*values = w->contour.values;
	int	cnt = w->contour.contour_cnt;

#ifdef DEBUG
	fprintf( stderr, "ContourWidget: SelectContours\n" );
#endif

	max = min = values[0];

	for ( i = 0; i < (w->contour.rows * w->contour.columns); i++ ) {
		if ( values[ i ] > max ) {
		    max = values[ i ];
		} else if ( values[ i ] < min ) {
		    min = values[ i ];
		}
	}

	/* 
	 * Fill in the contour list with cnt equally spaced values.
	 * Notice that we save the largest contour value first in the
	 * array.  This is so when we plot the lines, the lines drawn
	 * first will be those representing the highest values and
	 * will use the first entries in the contourGCs array.  If 
	 * numcolors < contour_cnt, we'll end up reusing the last
	 * contourGC's color for the final contour plot lines.
	 */

	for ( i = 0; i < cnt; i++ ) {
		t = ( i + 1 )/(float)( cnt + 1 );
		w->contour.contours[cnt-i-1] = 
					( (1.0 - t) * min ) + ( t * max );
	}

	*minp = min;
	*maxp = max;
}

/*============================================================================*
 * 	Function definitions for private methods useful for debugging
 *============================================================================*/

/*----------------------------------------------------------------------------*
 *	static void DrawContourValues( w );
 *
 *	DrawContourValues is provided as a debugging aid.
 *
 *	Contour values will only be displayed if there is space to show them
 *	WITHIN the drawing region of the plot.  They should not run over
 *	into the Range information area (if one is allocated).
 *----------------------------------------------------------------------------*/

static void DrawContourValues( w )
ContourWidget 	w;
{
	int		bottomY; 
	int		fontHt;
	int		i, x, y, tw;
	char		label[64];
	static char	Lab[] = "Contour Levels:";

	int		offsetx = w->contour.offsetx;

#ifdef DEBUG
	fprintf( stderr, "ContourWidget: DrawContourValues\n");
#endif
	bottomY = w->contour.offsety + w->contour.sizey;  
	fontHt = FontHt( w->contour.font );
	y = bottomY - fontHt - ( w->contour.contour_cnt * fontHt )
					+ w->contour.font->max_bounds.ascent;

	if ( y >= w->contour.offsety ) {	
   		tw = XTextWidth( w->contour.font, Lab, strlen(Lab) );
   		if ( ( x = ( w->core.width - tw )/2 ) < offsetx ) {
		    x = offsetx;
		}
   		XDrawString( XtDisplay(w), XtWindow(w), w->contour.textGC,
		             x, y, Lab, strlen(Lab) );

   		for ( i = 0; i < w->contour.contour_cnt; i++) {
		    y +=  fontHt;
        	    sprintf( label, "  %.3f", w->contour.contours[i] );
   		    tw = XTextWidth( w->contour.font, label, strlen(label) );
   	 	    if ( ( x = ( w->core.width - tw )/2 ) < offsetx ) {
		        x = offsetx;
		    }
       		    XDrawString( XtDisplay(w), XtWindow(w), w->contour.textGC,
			         x, y, label, strlen(label) );
		}
   	}

} /* DrawContourValues */



/* -----------------------------------------------------------------------
/*      static void MapXYToRowCol( w, x, y )
 *
 *	MapXYToRowCol accepts an (x,y) location and determines the 
 *      corresponding (r,c) location, saving that location for future
 *      reference.
 * ----------------------------------------------------------------------- */
static void MapXYToRowCol(w, x, y)
ContourWidget	w;
int		x, y;
{
  int	i;
  int	elew = w->contour.sizex / ( w->contour.columns - 1 );
  int   eleh = w->contour.sizey / ( w->contour.rows - 1);
  int	offsetx = w->contour.offsetx, offsety = w->contour.offsety;
  int	columns = w->contour.columns;


  for (i=0; i<w->contour.columns; i++)
    if ((x >= (offsetx+i*elew))
	&& (x <= (offsetx+(i+1)*elew-1)))
      break;
  if (i == w->contour.columns)
    w->contour.column_select = -1;
  else
    w->contour.column_select = i;

  for (i=0; i<w->contour.rows; i++)
    if ((y >= (offsety+i*eleh))
	&& (y <= (offsety+(i+1)*eleh-1)))
      break;
  if (i == w->contour.rows)
    w->contour.row_select = -1;
  else
    w->contour.row_select = i;

} /* MapXYToRowCol */




/*============================================================================*
 * 	Function definitions for public methods
 *============================================================================*

/*----------------------------------------------------------------------------*
 *      extern void ContourSetColors( w, numcolors, colors )
 *
 *      ContourSetColors is called by the application code to set up a color
 *      table for the Contour widget.
 *
 *      <w> is the Contour Widget.
 *      <numcolors> is the number of colors we will be setting
 *      <colors> is a pointer to a Pixel array with <numcolors> elements
 *----------------------------------------------------------------------------*/

extern void ContourSetColors( w, numcolors, colors )
ContourWidget    w;
int             numcolors;
Pixel           *colors;
{
        int             i;
        XGCValues       gcValues;
        XtGCMask        gcMask;

#ifdef DEBUG
        printf("ContourWidget: ContourSetColors\n");
#endif

        FreeColorGCs( w );

        w->contour.numcolors = numcolors;
        w->contour.contourGCs =
                        (GC *) XtMalloc( w->contour.numcolors * sizeof(GC) );

        gcMask = GCForeground | GCBackground;
	gcValues.background = w->core.background_pixel;
        for ( i=0; i < numcolors; i++) {
                gcValues.foreground = colors[i];
                w->contour.contourGCs[i] = XtGetGC( w, gcMask, &gcValues );
        }

        if (  XtIsRealized( w ) && w->core.visible  ) {
                DrawContour(w);
        }

} /* ContourSetColors */


/*----------------------------------------------------------------------------*
 *	extern void ContourSetValues( w, values )
 *
 *	ContourSetValues is called by the application code to set the values
 *	for all cells in the Contour.  There is no check that the number of
 *	integers pointed to by values equals (rows * columns) so be careful!
 *
 *	If appropriate, DrawContour() is called to update the display.
 *----------------------------------------------------------------------------*/

extern void ContourSetValues( w, values )
ContourWidget    w;
float		*values;
{
	int	i;

#ifdef DEBUG
	fprintf( stderr, "ContourWidget: ContourSetValues\n" );
#endif

	if ( w->contour.values == NULL ) {
		w->contour.values = (float *) XtMalloc( 
			w->contour.rows * w->contour.columns * sizeof(float) );
	}

	for ( i = 0; i < ( w->contour.rows * w->contour.columns ); i++ ) {
		    w->contour.values[ i ] = values[ i ];
	}

	if ( XtIsRealized( w ) && w->core.visible ) {
		DrawContour( w );
	}

} /* ContourSetValues */

/*----------------------------------------------------------------------------*
 *	extern void ContourSetValue( w, r, c, v )
 *
 *	ContourSetValue is called by the application code to set the value
 *	of a single cell Contour[r,c] to v.  Note that if this is called
 * 	prior to any calls to ContourSetValues(), the contour.values array
 *	will be allocated here, and the elements will all be initialized
 *	to zero.  
 *
 *	If appropriate, DrawContour() is called to update the display.
 *----------------------------------------------------------------------------*/

extern void ContourSetValue( w, r, c, v )
ContourWidget   w;
int		r, c; 
float		v;
{
  	int 	rows = w->contour.rows;
  	int 	cols = w->contour.columns;
	char	msgbuf[256];

#ifdef DEBUG
	fprintf( stderr, "ContourWidget: ContourSetValue - r:%d c:%d v: %.3f\n", 
								r, c, v);
#endif

  	if ( ( r > rows ) || ( c > cols ) ) {
     		sprintf( msgbuf, 
			 "Contour: (%d,%d) outside display range of (%d,%d)",
        	         r, c, rows, cols );
		XtWarning( msgbuf );
     		return;
  	}

  	if ( w->contour.values == NULL ) {
         	w->contour.values = 
			      (float *) XtCalloc( rows * cols, sizeof(float) );
      	}

	w->contour.values[r * cols + c] = v;

	if ( XtIsRealized( w ) && w->core.visible ) {
		DrawContour( w );
	}

} /* ContourSetValue */


/*----------------------------------------------------------------------------*
 *	extern void ContourGetValue( w, r, c, &v )
 *
 *	ContourGetValue is called by the application code to retrieve a
 *      data value from the cell Contour[r,c].  Note that if this is called
 * 	prior to any calls to ContourSetValues(), the contour.values array
 *	will be allocated here, and the elements will all be initialized
 *	to zero.  
 *
 *---------------------------------------------------------------------------*/

extern void ContourGetValue( w, r, c, v )
ContourWidget   w;
int		r, c; 
float          *v;
{
  	int 	rows = w->contour.rows;
  	int 	cols = w->contour.columns;
	char	msgbuf[256];

  	if ( ( r > rows ) || ( c > cols ) ) {
    	    sprintf( msgbuf, 
		  "ContourGetValue: (%d,%d) outside display range of (%d,%d)",
        	         r, c, rows, cols );
            XtWarning( msgbuf );
     	    *v = 0.0;
  	} else if ( w->contour.values == NULL ) {
            *v = 0.0;
        } else {
            *v =  w->contour.values[r * cols + c];
       }
} /* ContourGetValue */




/*----------------------------------------------------------------------------*
 *	extern void ContourGetRowCol( w, r, c )
 *
 *	ContourGetRowCol is called by the application code to retrieve the
 *      (r,c) location of the latest select event.
 *
 *---------------------------------------------------------------------------*/

extern void ContourGetRowCol( w, r, c )
ContourWidget   w;
int		*r, *c;
{
     *r = w->contour.row_select;
     *c = w->contour.column_select;
}



/*============================================================================*
 * 	Function definitions for action methods
 *============================================================================*/

/*----------------------------------------------------------------------------*
 *	static void ContourSelect( w, event)
 *
 *	ContourSelect processes a button down event, 
 *
 *----------------------------------------------------------------------------*/

static void ContourSelect( w, event )
ContourWidget		w;
XButtonPressedEvent	*event;
{
	int	i;

#ifdef DEBUG
	fprintf( stderr, "ContourWidget: ContourSelect\n" );
#endif

  	w->contour.cb_info.cnt = w->contour.contour_cnt;

	for (i = 0; i < w->contour.cb_info.cnt; i++) {
     		w->contour.cb_info.contours[i] = w->contour.contours[i];
	}
        MapXYToRowCol(w, event->x, event->y);

} /* ContourSelect */


/*----------------------------------------------------------------------------*
 *	static void ContourNotify( w, event)
 *
 *	ContourNotify processes a button up event, calling the Callback routine
p *	if one has been registered.
 *----------------------------------------------------------------------------*/

static void ContourNotify( w, event )
ContourWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
	fprintf( stderr, "ContourWidget: ContourSelect\n" );
#endif

	XtCallCallbacks( w, XtNselect, &(w->contour.cb_info) );

} /* ContourNotify */


