/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */

#include <stdio.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include "graphfmt.h"

double	sqrt();
#define round(x)	(x >= 0.0 ? (int)(x + .5) : (int)(x - .5))

void		ReadInputGraph();
void		AddNodeToGraph();
LinkPtr		ConstructLink();
void		Constructlinks();
void		FindLinkAttachments();
void		DetermineLinkSegments();
void		ScaleNodes();
void		FreeLinks();
NodePtr		FindNode();
int		FindToNode();

LinkPtr	free_links = NULL;

void read_to_eoln()
{
#ifdef DEBUG
printf("read_to_eoln\n");
#endif
  while (getchar() != '\n') ;

} /* read_to_eoln */

int read_string(s)
char	*s;
{
  char	c;
  int	i;

#ifdef DEBUG
printf("read_string\n");
#endif
  s[0] = '\0';
  while ((c=getchar()) != '"') {
#ifdef DEBUG
printf("%c", c);
#endif
    if (c=='%') {
      read_to_eoln();
      return(1);
    }
    if (c=='\n')
      return(1);
  }
  i = 0;
  while ((c=getchar()) != '"') {
    if (c=='\n')
      return(1);
    s[i] = c;
    i++;
  }
  s[i] = '\0';
  return(0);

} /* read_string */

int read_direction()
{
  char	c;
  int	d;

#ifdef DEBUG
printf("read_direction\n");
#endif
  while ((c=getchar()) != '-') {
    if (c=='%') {
      read_to_eoln();
      return(-1);
    }
    if (c=='\n') {
      return(-1);
    }
  }
  scanf("%d", &d);
  return(d);

} /* read_direction */

I_NodePtr find_node(igraph, id)
I_GraphPtr	igraph;
I_NodeID	id;
{
  I_NodePtr	node = igraph->nodes;

#ifdef DEBUG
printf("find_node\n");
#endif
  while (node != NULL)
    if (node->id == id)
      return(node);

} /* find_node */

void ReadGraphFromFile(file, igraph)
FILE		*file;
I_GraphPtr	igraph;
{
  int		c;
  char		s[100];
  int		x, y, d;
  I_NodeID	from, to;
  I_NodePtr	node;

#ifdef DEBUG
printf("ReadGraphFromFile\n");
#endif
  igraph->title = NULL;
  igraph->numnodes = 0;
  igraph->nodes = NULL;
  while ((c = getchar()) != EOF) {
    switch (c) {
    case 'g':
#ifdef DEBUG
printf("g command\n");
#endif
      scanf("%d", &igraph->numnodes);
      if (read_string(s) == 0)
	read_to_eoln();
      if (s[0] != '\0') {
	igraph->title = (char *) malloc(strlen(s));
	strcpy(igraph->title, s);
      }
      break;
    case 'n':
      scanf("%d %d %d", &from, &x, &y);
#ifdef DEBUG
printf("n command: %d %d %d\n", from, x, y);
#endif
      node = (I_NodePtr) malloc(sizeof(I_Node));
      node->name = NULL;
      node->id = from;
      node->x = x;
      node->y = y;
      node->todegree = 0;
      node->next = igraph->nodes;
      igraph->nodes = node;
      if (read_string(s) == 0)
	read_to_eoln();
      if (s[0] != '\0') {
#ifdef DEBUG
printf("%s\n", s);
#endif
	node->name = (char *) malloc(strlen(s));
	strcpy(node->name, s);
      }
      break;
    case 'l':
#ifdef DEBUG
printf("l command\n");
#endif
      scanf("%d %d", &from, &to);;
      if ((node = find_node(igraph, from)) == NULL) {
	printf("cannot find node %d\n", from);
	exit(1);
      }
      d = read_direction();
      if (d != -1) {
	node->directed[node->todegree] = d;
	read_to_eoln();
      }
      else
	node->directed[node->todegree] = 0;
      node->to[node->todegree] = to;
      node->todegree++;
      break;
    case '%':
      read_to_eoln();
      break;
    default:
      printf("illegal graph command '%c'\n", c);
      exit(1);
      break;
    }
  }

} /* ReadGraphFromFile */

void ReadInputGraph(igraph, graph)
I_GraphPtr	igraph;
GraphPtr	graph;
{
  int		i, numnodes;
  I_NodePtr	inodes;
  LinkPtr	link;
  NodePtr	node;

#ifdef DEBUG
printf("ReadInputGraph\n");
#endif
  if (igraph->title == NULL)
    graph->title = NULL;
  else {
    graph->title = (char *) malloc(strlen(igraph->title));
    strcpy(graph->title, igraph->title);
  }
  numnodes = graph->numnodes = igraph->numnodes;
  graph->nodes = NULL;
  inodes = igraph->nodes;
  for (i=0; i<numnodes; i++) {
    if (inodes == NULL) {
      printf("not enough nodes specified\n");
      exit(1);
    }
    AddNodeToGraph(graph, inodes);
    inodes = inodes->next;
  }

} /* ReadInputGraph */

void AddNodeToGraph(graph, inode)
GraphPtr	graph;
I_NodePtr	inode;
{
  int		i, j;
  LinkPtr	temp_link;
  NodePtr	node;

#ifdef DEBUG
printf("AddNodeToGraph\n");
#endif
  node = (NodePtr) malloc(sizeof(Node));
  node->next = graph->nodes;
  graph->nodes = node;
  node->displayed = inode->displayed;
  node->id = inode->id;
  if (inode->name == NULL)
    node->name = NULL;
  else {
    node->name = (char *) malloc(strlen(inode->name));
    strcpy(node->name, inode->name);
  }
  node->point.x = inode->x;
  node->point.y = inode->y;
  node->todegree = inode->todegree;
  for (j=0; j<node->todegree; j++) {
    node->to[j] = inode->to[j];
    if (free_links != NULL) {
      temp_link = free_links->next;
      node->link[j] = free_links;
      free_links = temp_link->next;
    }
    else
      node->link[j] = (LinkPtr) malloc(sizeof(Link));
    node->link[j]->value = 0.0;
    node->link[j]->next = NULL;
  }

} /* AddNodeToGraph */

LinkPtr ConstructLink(from, to, nodes, link)
NodeID	from, to;
NodePtr	nodes;
LinkPtr	link;
{
  int		x1, y1, x2, y2;
  LinkPtr	new_link;

#ifdef DEBUG
printf("ConstructLink\n");
#endif

} /* ConstructLink */

void ConstructLinks(graph)
GraphPtr	graph;
{
  int		i;
  LinkPtr	link;
  NodePtr	node = graph->nodes;

#ifdef DEBUG
printf("ConstructLinks\n");
#endif
  while (node != NULL) {
    for (i=0; i<node->todegree; i++) {
      link = node->link[i]->next;
      ConstructLink(node->id, node->to[i], graph->nodes, link);
    }
    node = node->next;
  }

} /* ConstructLinks */

void ComputeLinkPositions(graph)
GraphPtr	graph;
{
  int		i;
  NodeID	to;
  NodePtr	from_node = graph->nodes;
  NodePtr	to_node;
  XPoint	point;
  XSegment	*segment;

#ifdef DEBUG
printf("ComputeLinkPositions\n");
#endif
  while(from_node != NULL) {
    for (i=0; i<from_node->todegree; i++) {
      to = from_node->to[i];
      to_node = graph->nodes;
      point = from_node->point;
      segment = &from_node->link[i]->segment;
      segment->x1 = point.x;
      segment->y1 = point.y;
      while (to_node != NULL) {
	if (to == to_node->id) {
	  segment->x2 = to_node->point.x;
	  segment->y2 = to_node->point.y;
	  break;
	}
	to_node = to_node->next;
      }
    }
    from_node = from_node->next;
  }

} /* ComputeLinkPositions */

/************************************************************************/
/* FindLinkAttachments() works with the link before it has been	multi-	*/
/* segmented to find the attachment point on the endpoint nodes'	*/
/* boundary.  We use these attachment points to avoid having to fill	*/
/* the node every time we draw the link.				*/
/************************************************************************/
void FindLinkAttachments(graph, node_shape, radius)
GraphPtr	graph;
int		node_shape;		/* 0 - circle, 1 - square */
int		radius;
{
  int		i, j;
  double	l, r = (double) radius;
  double	x1, y1, x2, y2;
  double	mx, my, nx, ny;
  double	s, t;
  NodePtr	from_node = graph->nodes;
  XSegment	*segment;

#ifdef DEBUG
printf("FindLinkAttachments\n");
#endif
  while(from_node != NULL) {
    for (i=0; i<from_node->todegree; i++) {
      segment = &from_node->link[i]->segment;
      x1 = (double) segment->x1; y1 = (double) segment->y1;
      x2 = (double) segment->x2; y2 = (double) segment->y2;
      if (node_shape == 0) {	/* circle */
	l = sqrt((x2-x1)*(x2-x1) + (y2-y1)*(y2-y1));
	segment->x1 = round((x1 + r * (x2-x1)/l));
	segment->y1 = round((y1 + r * (y2-y1)/l));
	segment->x2 = round((x2 + r * (x1-x2)/l));
	segment->y2 = round((y2 + r * (y1-y2)/l));
      }
      else {			/* square */
	for (j=0; j<4; j++) {
	  switch (j) {
	  case 0:
	    mx = x1 - radius; my = y1 - radius;
	    nx = x1 + radius; ny = y1 - radius;
	    break;
	  case 1:
	    mx = x1 + radius; my = y1 - radius;
	    nx = x1 + radius; ny = y1 + radius;
	    break;
	  case 2:
	    mx = x1 + radius; my = y1 + radius;
	    nx = x1 - radius; ny = y1 + radius;
	    break;
	  case 3:
	    mx = x1 - radius; my = y1 + radius;
	    nx = x1 - radius; ny = y1 + radius;
	    break;
	  }
	  s = ((nx-mx)*(my-y1) - (ny-my)*(mx-x1))
	    / ((nx-mx)*(y2-y1) - (ny-my)*(x2-x1));
	  t = ((x2-x1)*(my-y1) - (y2-y1)*(mx-x1))
	    / ((nx-mx)*(y2-y1) - (ny-my)*(x2-x1));
	  if ((s>=0) && (s<=1) && (t>=0) && (t<=1))
	    break;
	}
	if (j == 4) {
	  printf("FindLinkAttachments: no intersection\n");
	  exit(1);
	}
	else {
	  segment->x1 = round(x1 + (x2-x1)*s);
	  segment->y1 = round(y1 + (y2-y1)*t);
	  segment->x2 = round(x2 - (x2-x1)*s);
	  segment->y2 = round(y2 - (y2-y1)*t);
	}
      }
    }
    from_node = from_node->next;
  }

} /* FindLinkAttachments */

void DetermineLinkSegments()
{

} /* DeterminLinkSegments */

void ScaleGraph(graph, width, height)
GraphPtr	graph;
int		width, height;
{
#ifdef DEBUG
printf("ScaleGraph\n");
#endif
  if (graph != NULL) {
    ScaleNodes(graph, width, height);
    FreeLinks(graph);
    ComputeLinkPositions(graph);
/*  ConstructLinks(graph);*/
  }

} /* ScaleGraph */

void ScaleNodes(graph, width, height)
GraphPtr	graph;
int		width, height;
{
  int		minx = 100000, miny = 100000, maxx = -1, maxy = -1;
  float		xscale, yscale;
  NodePtr	node = graph->nodes;

#ifdef DEBUG
printf("ScaleNodes\n");
#endif
  while (node != NULL) {
    if (minx > node->point.x)
      minx = node->point.x;
    if (miny > node->point.y)
      miny = node->point.y;
    if (maxx < node->point.x)
      maxx = node->point.x;
    if (maxy < node->point.y)
      maxy = node->point.y;
    node = node->next;
  }
  xscale = (float) width / (float) (maxx - minx);
  yscale = (float) height / (float) (maxy - miny);
  node = graph->nodes;
  while (node != NULL) {
    node->point.x = xscale * (float) (node->point.x-minx);
    node->point.y = yscale * (float) (node->point.y-miny);
    node = node->next;
  }

} /* ScaleNodes */

void FreeLinks(graph)
GraphPtr	graph;
{
  int		i;
  NodePtr	node = graph->nodes;
  LinkPtr	link, next;

#ifdef DEBUG
printf("FreeLinks\n");
#endif
  while (node != NULL) {
    for (i=0; i<node->todegree; i++) {
      link = node->link[i]->next;
      while (link != NULL) {
	next = link->next;
	link->next = free_links;
	free_links = link;
	link = next;
      }
    }
    node = node->next;
  }

} /* FreeLinks */

NodePtr FindNode(graph, id)
GraphPtr	graph;
NodeID		id;
{
  NodePtr	node = graph->nodes;

#ifdef DEBUG
printf("FindNode\n");
#endif
  while (node != NULL) {
    if (id == node->id)
      return(node);
    else
      node = node->next;
  }

} /* FindNode */

int FindToNode(node, to)
NodePtr	node;
NodeID	to;
{
  int	i;

#ifdef DEBUG
printf("FindToNode\n");
#endif
  for (i=0; i<node->todegree; i++) {
    if (to == node->to[i])
      return(i);
  }

} /* FindToNode */

void PrintGraph(graph)
GraphPtr	graph;
{
  int		i, numnodes = graph->numnodes;
  NodePtr	nodes = graph->nodes;

#ifdef DEBUG
printf("PrintGraph\n");
#endif
  while (nodes != NULL) {
    printf("NodeID = %d\n", nodes->id);
    printf("\tx = %d, y = %d\n", nodes->point.x, nodes->point.y);
    printf("\ttodegree = %d\n", nodes->todegree);
    for (i=0; i<nodes->todegree; i++) {
      printf("\t\tto[%d]: %d\n", i, nodes->to[i]);
      printf("\t\tlink[%d]: x1=%d, y1=%d, x2=%d, y2=%d\n", i,
	     nodes->link[i]->segment.x1, nodes->link[i]->segment.y1,
	     nodes->link[i]->segment.x2, nodes->link[i]->segment.y2);
    }
    nodes = nodes->next;
  }

} /* PrintGraph */
