/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Allen D. Malony (malony@uicsrd.csrd.uiuc.edu)
 * Contributing Author: Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 * Contributing Author: Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/kiviat/RCS/Kiviat.c,v 1.14 1994/02/25 04:46:39 aydt Exp $
 */

/* #define DEBUG */

#include <math.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include "KiviatP.h"

#define Min(a, b)	((a) < (b) ? (a) : (b))
#define Max(a, b)	((a) > (b) ? (a) : (b))
#define XtStrlen(s)	((s) ? strlen(s) : 0)
#define Offset(field)	XtOffset(KiviatWidget, field)
#define Norm(w, v)	\
	( ( (float)v - (float)w->kiviat.min_value ) /	\
	  ( (float)w->kiviat.max_value - (float)w->kiviat.min_value + 1.0 ) )


/***********************/
/* priviate procedures */
/***********************/
static void	ClassInitialize(), Initialize();
static void	Destroy(), Resize(), Redisplay();
static Boolean	SetValues();

static void	DrawKiviatFace(), DrawKiviatAxes();
static void	DrawKiviatValues(), EraseKiviatValues();
static void	KiviatSelect(), KiviatNotify();

/*********************/
/* public procedures */
/*********************/
extern void	KiviatSetValues();
extern void	KiviatGetSelectedAxis();

/*************************/
/* kiviat translations   */
/*************************/
static char kiviat_translations[] = "	\
  <Btn1Down>:	KiviatSelect()\n	\
  <Btn1Up>:	KiviatNotify()	\
";

/********************/
/* kiviat actions   */
/********************/
static XtActionsRec	kiviat_actions[] = {
  {"KiviatSelect",	(XtActionProc) KiviatSelect	},
  {"KiviatNotify",	(XtActionProc) KiviatNotify	},
};

/********************/
/* kiviat resources */
/********************/
static XtResource resources[] = {
  /* core resources */
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     Offset(core.width), XtRString, "180"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     Offset(core.height), XtRString, "180"},
  {XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(core.background_pixel), XtRString, "white"},
  /* kiviat resources */
  {XtNfill, XtCFill, XtRBoolean, sizeof(Boolean),
     Offset(kiviat.fill), XtRString, "TRUE"},
  {XtNhistory, XtCHistory, XtRBoolean, sizeof(Boolean),
     Offset(kiviat.history), XtRString, "TRUE"},
  {XtNmaxHistory, XtCMaxHistory, XtRBoolean, sizeof(Boolean),
     Offset(kiviat.maxhistory), XtRString, "FALSE"},
  {XtNfaceBorderColor, XtCFaceBorderColor, XtRPixel, sizeof(Pixel),
     Offset(kiviat.face_border_color), XtRString, "black"},
  {XtNfillColor, XtCFillColor, XtRPixel, sizeof(Pixel),
     Offset(kiviat.fill_color), XtRString, "blue"},
  {XtNhistoryColor, XtCHistoryColor, XtRPixel, sizeof(Pixel),
     Offset(kiviat.history_color), XtRString, "sky blue"},
  {XtNaxisColor, XtCAxisColor, XtRPixel, sizeof(Pixel),
     Offset(kiviat.axis_color), XtRString, "red"},
  {XtNbackingStore, XtCBackingStore, XtRBackingStore, sizeof(int),
     Offset(kiviat.backing_store), XtRString, "default"},
  {XtNnumaxes, XtCNumaxes, XtRInt, sizeof(int), 
     Offset(kiviat.numaxes), XtRString, "8" },
  {XtNminValue, XtCMinValue, XtRInt, sizeof(int),
     Offset(kiviat.min_value), XtRString, "0"},
  {XtNmaxValue, XtCMaxValue, XtRInt, sizeof(int),
     Offset(kiviat.max_value), XtRString, "100"},
  {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     Offset(kiviat.select), XtRCallback, (caddr_t) NULL},
};
#undef offset

/***********************/
/* kiviat class record */
/***********************/
KiviatClassRec kiviatClassRec = {
  { /* core fields */
    /* superclass		*/	&widgetClassRec,
    /* class_name		*/	"Kiviat",
    /* widget_size		*/	sizeof(KiviatRec),
    /* class_initialize		*/	ClassInitialize,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	kiviat_actions,
    /* num_actions		*/	XtNumber(kiviat_actions),
    /* resources		*/	resources,
    /* resource_count		*/	XtNumber(resources),
    /* xrm_class		*/	NULL,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	kiviat_translations,
    /* query_geometry           */	XtInheritQueryGeometry,
    /* display_accelerator      */	XtInheritDisplayAccelerator,
    /* extension                */	NULL
    }
};

WidgetClass kiviatWidgetClass =(WidgetClass) &kiviatClassRec;

/************************************************************************/
/*                          PRIVATE PROCEDURES                          */
/************************************************************************/

/************************************************************************/
/* ClassInitialize()							*/
/************************************************************************/
static void ClassInitialize()
{
  XtAddConverter(XtRString, XtRBackingStore, XmuCvtStringToBackingStore,
		 NULL, 0);
}

/************************************************************************/
/* Initialize()								*/
/************************************************************************/
static void Initialize(request, new)
KiviatWidget request, new;
{
  int		i;
  XtGCMask	GCmask;
  XGCValues	GCvalues;

#ifdef DEBUG
printf("Initialize: request 0x%x; new 0x%x\n", request, new);
#endif
  for (i=0; i<MAX_AXES; i++) {
    new->kiviat.values[i] = new->kiviat.min_value;
    new->kiviat.history_values[i] = new->kiviat.min_value;
  }

  new->kiviat.count = 0;

  if(new->core.width == 0)
    new->core.width = MIN_SIZE;
  if(new->core.height == 0)
    new->core.height = MIN_SIZE;

  GCmask = GCForeground | GCBackground;
  GCvalues.background = new->core.background_pixel;
  GCvalues.foreground = new->kiviat.face_border_color;
  new->kiviat.face_borderGC = XtGetGC((Widget)new, GCmask, &GCvalues);
  
  GCmask = GCForeground;
  GCvalues.foreground = new->kiviat.fill_color;
  new->kiviat.fillGC = XtGetGC((Widget)new, GCmask, &GCvalues);

  GCmask = GCForeground;
  GCvalues.foreground = new->kiviat.history_color;
  new->kiviat.historyGC = XtGetGC((Widget)new, GCmask, &GCvalues);

  GCmask = GCForeground;
  GCvalues.foreground = new->kiviat.axis_color;
  new->kiviat.axisGC = XtGetGC((Widget)new, GCmask, &GCvalues);
  
  GCmask = GCForeground;
  GCvalues.foreground = new->core.background_pixel;
  new->kiviat.eraseGC = XtGetGC((Widget)new, GCmask, &GCvalues);

  Resize( new );
  
} /* Initialize */

/************************************************************************/
/* SetValues()								*/
/************************************************************************/
static Boolean SetValues(current, grequest, new)
KiviatWidget	current, grequest, new;
{
  int			i;
  Boolean		redraw = FALSE;
  XtGCMask		GCmask;
  XGCValues		GCvalues;

#ifdef DEBUG
printf("SetValues: current 0x%x; new 0x%x\n", current, new);
#endif
  /* change face border color */
  if((new->kiviat.face_border_color != current->kiviat.face_border_color)
      ||(new->core.background_pixel != current->core.background_pixel)) {
    GCmask = GCForeground | GCBackground;
    GCvalues.foreground = new->kiviat.face_border_color;
    GCvalues.background = new->core.background_pixel;
    XtReleaseGC(current, current->kiviat.face_borderGC);
    new->kiviat.face_borderGC = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }
  
  /* change fill color */
  if(new->kiviat.fill_color != current->kiviat.fill_color) {
    GCmask = GCForeground;
    GCvalues.foreground = new->kiviat.fill_color;
    XtReleaseGC(current, current->kiviat.fillGC);
    new->kiviat.fillGC = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }
  
  /* change axis color */
  if(new->kiviat.axis_color != current->kiviat.axis_color) {
    GCmask = GCForeground;
    GCvalues.foreground = new->kiviat.axis_color;
    XtReleaseGC(current, current->kiviat.axisGC);
    new->kiviat.axisGC = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }

  /* change history color */
  if(new->kiviat.history_color != current->kiviat.history_color) {
    GCmask = GCForeground;
    GCvalues.foreground = new->kiviat.history_color;
    XtReleaseGC(current, current->kiviat.historyGC);
    new->kiviat.historyGC = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }
  
  /* change erase color */
  if(new->core.background_pixel != current->core.background_pixel) {
    GCmask = GCForeground;
    GCvalues.foreground = new->core.background_pixel;
    XtReleaseGC(current, current->kiviat.eraseGC);
    new->kiviat.eraseGC = XtGetGC((Widget)current, GCmask, &GCvalues);
    redraw = TRUE;
  }
  
  /* change number of axes */
  if(new->kiviat.numaxes != current->kiviat.numaxes) {
    if(new->kiviat.numaxes > MAX_AXES) {
      XtWarning( "Kiviat: exceeded maximum number of axes" );
      new->kiviat.numaxes = MAX_AXES;
    }
    redraw = TRUE;
  }

  /* change width or height */
  if ( (new->core.width != current->core.width)
      || (new->core.height != current->core.height) ) {
     Resize(new);
     redraw = TRUE;
  }

  return(redraw);
  
} /* SetValues */

/************************************************************************/
/* Destroy()								*/
/************************************************************************/
static void Destroy(w)
KiviatWidget w;
{

#ifdef DEBUG
printf("Destroy: widget 0x%x\n", w);
#endif
  XtReleaseGC(w, w->kiviat.face_borderGC);
  XtReleaseGC(w, w->kiviat.fillGC);
  XtReleaseGC(w, w->kiviat.axisGC);
  XtReleaseGC(w, w->kiviat.historyGC);
  XtReleaseGC(w, w->kiviat.eraseGC);
  XtRemoveAllCallbacks( w, XtNselect );

} /* Destroy */

/************************************************************************/
/* Resize()								*/
/************************************************************************/
static void Resize(w) 
KiviatWidget	w;
{
#ifdef DEBUG
printf("Resize widget: 0x%x\n", w);
#endif
    w->kiviat.radius = ( Min( w->core.width, w->core.height ) - PADDING*2 ) / 2;
    w->kiviat.centerX = w->core.width / 2;
    w->kiviat.centerY = w->core.height / 2;
#ifdef DEBUG
printf( "Height %d, Width %d, Radius %d, centerX %d, centerY %d\n",
	w->core.height, w->core.width, w->kiviat.radius,
	w->kiviat.centerX, w->kiviat.centerY );
#endif DEBUG

} /* Resize */

/************************************************************************/
/* Redisplay()								*/
/************************************************************************/
static void Redisplay(w, event, region)
KiviatWidget	w;
XEvent	*event;				/* unused */
Region	region;				/* unused */
{
  char		buf[512];
  
#ifdef DEBUG
printf("Redisplay: widget = 0x%x\n", w);
#endif
  if ( XtIsRealized( w ) && w->core.visible ) {
     DrawKiviatFace(w);
     DrawKiviatAxes(w);
     DrawKiviatValues(w);
  }
} /* Redisplay */

/************************************************************************/
/* DrawKiviatFace()							*/
/************************************************************************/
static void DrawKiviatFace(w)
KiviatWidget	w;
{
  register int	i;
  int		perimx, perimy;
	
#ifdef DEBUG
printf("DrawKiviatFace: widget = 0x%x\n", w);
#endif
  XClearWindow( XtDisplay(w), XtWindow(w) );
  XDrawArc(XtDisplay(w), XtWindow(w), w->kiviat.face_borderGC,
	   w->kiviat.centerX - w->kiviat.radius,
	   w->kiviat.centerY - w->kiviat.radius,
	   2*w->kiviat.radius, 2*w->kiviat.radius,
	   0, 360*64);

} /* DrawKiviatFace */

/************************************************************************/
/* DrawKiviatAxes()							*/
/************************************************************************/
static void DrawKiviatAxes(w)
KiviatWidget	w;
{
  register int	i;
  int		perimx, perimy;
  double	fraction;
  double	angle, cosangle, sinangle;
	
#ifdef DEBUG
printf("DrawKiviatAxes : widget = 0x%x\n", w);
printf("  - number of axes = %d\n", w->kiviat.numaxes);
#endif
  for(i=0; i<w->kiviat.numaxes; i++) {
    fraction = ((double) i) / ((double) w->kiviat.numaxes);
    angle = NINETY + TWOPI - (TWOPI * fraction);
    cosangle = cos(angle);
    sinangle = sin(angle);
    perimx = w->kiviat.centerX +
      (int) ((double) w->kiviat.radius * cosangle);
    perimy = w->kiviat.centerY -
      (int) ((double) w->kiviat.radius * sinangle);
    XDrawLine(XtDisplay(w), XtWindow(w), w->kiviat.axisGC,
	      w->kiviat.centerX, w->kiviat.centerY, perimx, perimy);
  }

} /* DrawKiviatAxes */

/************************************************************************/
/* DrawKiviatValues()							*/
/************************************************************************/
static void DrawKiviatValues(w)
KiviatWidget	w;
{
  register int	i;
  double	fraction;
  double	angle, cosangle, sinangle;

#ifdef DEBUG
printf("DrawKiviatValues: widget = 0x%x\n", w);
#endif
  for(i=0; i<w->kiviat.numaxes; i++) {
    fraction =((double)i) / ((double) w->kiviat.numaxes);
    angle = NINETY + TWOPI - (TWOPI * fraction);
    cosangle = cos(angle);
    sinangle = sin(angle);
    w->kiviat.points[i].x = w->kiviat.centerX +
      (int)((double)(w->kiviat.radius)) *
	    cosangle * (double) Norm(w, w->kiviat.values[i]);
    w->kiviat.points[i].y = w->kiviat.centerY -
      (int)((double)(w->kiviat.radius)) *
	    sinangle * (double) Norm(w, w->kiviat.values[i]);

    w->kiviat.history_points[i].x = w->kiviat.centerX +
      (int)((double)(w->kiviat.radius)) *
	    cosangle * (double) Norm(w, w->kiviat.history_values[i]);
    w->kiviat.history_points[i].y = w->kiviat.centerY -
      (int)((double)(w->kiviat.radius)) *
	    sinangle * (double) Norm(w, w->kiviat.history_values[i]);
  }    

  if (w->kiviat.history) {
     if (w->kiviat.fill) {
        XFillPolygon(XtDisplay(w), XtWindow(w), w->kiviat.historyGC,
		 w->kiviat.history_points, w->kiviat.numaxes,
		 Nonconvex, CoordModeOrigin);
     }

     for(i=0; i<w->kiviat.numaxes-1; i++) {
       XDrawLine(XtDisplay(w), XtWindow(w), w->kiviat.historyGC,
	      w->kiviat.history_points[i].x, w->kiviat.history_points[i].y,
	      w->kiviat.history_points[i+1].x, w->kiviat.history_points[i+1].y);
     }

     XDrawLine(XtDisplay(w), XtWindow(w), w->kiviat.historyGC,
               w->kiviat.history_points[i].x, w->kiviat.history_points[i].y,
	       w->kiviat.history_points[0].x, w->kiviat.history_points[0].y);
  }

  if (w->kiviat.fill) {
    XFillPolygon(XtDisplay(w), XtWindow(w), w->kiviat.fillGC,
		 w->kiviat.points, w->kiviat.numaxes,
		 Nonconvex, CoordModeOrigin);
  }

  for(i=0; i<w->kiviat.numaxes-1; i++) {
    XDrawLine(XtDisplay(w), XtWindow(w), w->kiviat.fillGC,
	      w->kiviat.points[i].x, w->kiviat.points[i].y,
	      w->kiviat.points[i+1].x, w->kiviat.points[i+1].y);
  }

  XDrawLine(XtDisplay(w), XtWindow(w), w->kiviat.fillGC,
	    w->kiviat.points[i].x, w->kiviat.points[i].y,
	    w->kiviat.points[0].x, w->kiviat.points[0].y);

} /* DrawKiviatValues */

/************************************************************************/
/* EraseKiviatValues()							*/
/************************************************************************/
static void EraseKiviatValues(w)
KiviatWidget	w;
{
  register int	i;

#ifdef DEBUG
printf("EraseKiviatValues: widget = 0x%x\n", w);/
#endif
  if (w->kiviat.fill) {
    XFillPolygon(XtDisplay(w), XtWindow(w), w->kiviat.eraseGC,
		 w->kiviat.points, w->kiviat.numaxes,
		 Nonconvex, CoordModeOrigin);

    if (w->kiviat.history) {
       XFillPolygon(XtDisplay(w), XtWindow(w), w->kiviat.eraseGC,
		    w->kiviat.history_points, w->kiviat.numaxes,
		    Nonconvex, CoordModeOrigin);
    }
  }

  for(i=0; i<w->kiviat.numaxes-1; i++) {
    XDrawLine(XtDisplay(w), XtWindow(w), w->kiviat.eraseGC,
	      w->kiviat.points[i].x, w->kiviat.points[i].y,
	      w->kiviat.points[i+1].x, w->kiviat.points[i+1].y);

    if (w->kiviat.history) {
       XDrawLine(XtDisplay(w), XtWindow(w), w->kiviat.eraseGC,
	         w->kiviat.history_points[i].x, w->kiviat.history_points[i].y,
	         w->kiviat.history_points[i+1].x, w->kiviat.history_points[i+1].y);
    }
  }

  XDrawLine(XtDisplay(w), XtWindow(w), w->kiviat.eraseGC,
	    w->kiviat.points[i].x, w->kiviat.points[i].y,
	    w->kiviat.points[0].x, w->kiviat.points[0].y);

  if (w->kiviat.history) {
     XDrawLine(XtDisplay(w), XtWindow(w), w->kiviat.eraseGC,
	       w->kiviat.history_points[i].x, w->kiviat.history_points[i].y,
	       w->kiviat.history_points[0].x, w->kiviat.history_points[0].y);
  }

} /* EraseKiviatValues */

/************************************************************************/
/* Euclid() computes the distance between a pair of points		*/
/************************************************************************/
static double Euclid(x0, y0, x1, y1)
double	x0, y0, x1, y1;
{
	double	Dist;
#ifdef DEBUG
printf("Euclid\n");
#endif

	Dist = (x0 - x1) * (x0 - x1) + (y0 - y1) * (y0 - y1);
	Dist = sqrt (Dist);

	return (Dist);

} /* Euclid */

/************************************************************************/
/* KiviatSelect() processes button down event.				*/
/************************************************************************/
static void KiviatSelect(w, event)
KiviatWidget		w;
XButtonPressedEvent	*event;
{
  register int	i;
  double	fraction;
  double	x0 = (double) event ->x;
  double	y0 = (double) event ->y;
  double	x1, y1;
  double	angle, sinangle, cosangle;
  double	Dist, MinDist;
  int		MinLoc;

#ifdef DEBUG
printf("KiviatSelect: widget = 0x%x\n", w);
#endif

  MinDist = (double) w->core.width + (double) w->core.height;
  MinLoc = -1;

  for(i=0; i<w->kiviat.numaxes; i++) {
    fraction = ((double) i) / ((double) w->kiviat.numaxes);
    angle = NINETY + TWOPI - (TWOPI * fraction);
    cosangle = cos(angle);
    sinangle = sin(angle);
    x1 = (double) (w->kiviat.centerX) + ((double) w->kiviat.radius) * cosangle;
    y1 = (double) (w->kiviat.centerY) - ((double) w->kiviat.radius) * sinangle;

    Dist = Euclid (x0, y0, x1, y1);

    if (Dist < MinDist) {
       MinDist = Dist;
       MinLoc = i;
    }
  }
  w->kiviat.value_select = w->kiviat.values[MinLoc];
  w->kiviat.axis_select = MinLoc;

} /* KiviatSelect */

/************************************************************************/
/* KiviatNotify() processes button up event.				*/
/************************************************************************/
static void KiviatNotify(w, event)
KiviatWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("KiviatNotify: widget = 0x%x\n", w);
#endif
  XtCallCallbacks(w, XtNselect, (caddr_t) w->kiviat.value_select);

} /* KiviatNotify */

/************************************************************************/
/*                          PUBLIC PROCEDURES                           */
/************************************************************************/

/************************************************************************/
/* KiviatSetValues()							*/
/************************************************************************/
extern void KiviatSetValues(w, numvals, values)
KiviatWidget	w;
int		numvals;
int		*values;
{
  int	i;
  double tmp;
  double newval;

#ifdef DEBUG
printf("KiviatSetValues: widget = 0x%x\n", w);
#endif

  if ( numvals > w->kiviat.numaxes  ) {	/* greater number of axes */
	if ( numvals > MAX_AXES ) {
	 	XtWarning( "Kiviat: exceeded maximum number of axes" );
		numvals = MAX_AXES;
	} 
	w->kiviat.numaxes = numvals ;
	DrawKiviatFace( w );			/* clear face */
	DrawKiviatAxes( w );			/* new axes   */
  }

  EraseKiviatValues(w);				/* Clear old values */

  w->kiviat.count++;

  for (i=0; i<numvals; i++) {		/* draw all the data unless too much, 
					 * then only use first MAX_AXES - 1
					 * values. */
    if (values[i] < w->kiviat.min_value) {
      printf("KiviatSetValues: tried to set an axis value (%d) ", values[i]);
      printf("less than the min (%d)\n", w->kiviat.min_value);
      w->kiviat.values[i] = w->kiviat.min_value;
    }
    else if (values[i] > w->kiviat.max_value) {
      printf("KiviatSetValues: tried to set an axis value (%d) ", values[i]);
      printf("greater than the max (%d)\n", w->kiviat.max_value);
      w->kiviat.values[i] = w->kiviat.max_value;
    }
    else
      w->kiviat.values[i] = values[i];

      if (w->kiviat.maxhistory) {
         w->kiviat.history_values[i] = 
		Max(w->kiviat.history_values[i], values[i]);
      } else {				/* compute new average		*/
	 if (w->kiviat.count == 1) {
	   w->kiviat.history_values[i] = w->kiviat.values[i];
	 } else {
            tmp = w->kiviat.history_values[i] * (double)(w->kiviat.count - 1);
	    newval= (tmp + values[i]) / (double)(w->kiviat.count);
            w->kiviat.history_values[i] = newval;
	 }
      }
  }
  DrawKiviatAxes(w);
  DrawKiviatValues(w);
    
} /* KiviatSetValues */



/* ---------------------------------------------------------------------- */
/* KiviatGetSelectedAxis()						
*/
/* ---------------------------------------------------------------------- */

extern void KiviatGetSelectedAxis(w, axis)
KiviatWidget	w;
int		*axis;
{
	*axis = w->kiviat.axis_select;
}

