/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Allen D. Malony (malony@uicsrd.csrd.uiuc.edu)
 * Contributing Authors: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *                       Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 * 
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/matrix/RCS/Matrix.c,v 1.8 1994/02/25 04:46:50 aydt Exp $
 */

/* #define DEBUG */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/keysymdef.h>
#include "MatrixP.h"

/*============================================================================*
 * 	Macros used by the Matrix widget
 *============================================================================*/

#define norm( w, v )    \
	( ( (float)v - (float)w->matrix.min_value ) / 	               \
          ( (float)w->matrix.max_value - (float)w->matrix.min_value + 1 ) )

/*============================================================================*
 * 	Resource list
 *============================================================================*/

static XtResource resources[] = {
	/* core resources */
  	{ XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     	   XtOffset(MatrixWidget, core.width), XtRString, "200" },
  	{ XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     	   XtOffset(MatrixWidget, core.height), XtRString, "200" },
  	{ XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
     	   XtOffset(MatrixWidget, core.background_pixel), XtRString, "white" },
  	/* matrix resources */
  	{ XtNcolumns, XtCColumns, XtRInt, sizeof(int), 
     	   XtOffset(MatrixWidget, matrix.columns), XtRString, "10" },
  	{ XtNgridColor, XtCGridColor, XtRPixel, sizeof(Pixel),
     	   XtOffset(MatrixWidget, matrix.grid_color), XtRString, "black" },
  	{ XtNminValue, XtCMinValue, XtRInt, sizeof(int),
     	   XtOffset(MatrixWidget, matrix.min_value), XtRString, "0" },
  	{ XtNmaxValue, XtCMaxValue, XtRInt, sizeof(int),
     	   XtOffset(MatrixWidget, matrix.max_value), XtRString, "100" },
  	{ XtNrows, XtCRows, XtRInt, sizeof(int), 
     	   XtOffset(MatrixWidget, matrix.rows), XtRString, "10" },
  	{ XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     	   XtOffset(MatrixWidget, matrix.select), XtRCallback, (caddr_t)NULL },
  	{ XtNshowGrid, XtCShowGrid, XtRBoolean, sizeof (Boolean),
     	   XtOffset(MatrixWidget, matrix.show_grid), XtRString, "TRUE" },
};

/*============================================================================*
 * 	Action function declarations. 
 * 	Default translation table.
 * 	Actions table.
 *============================================================================*/

static void	MatrixSelect(), MatrixNotify();

static char defaultTranslations[] = 
  	"<Btn1Down>:	MatrixSelect()		\n\
  	 <Btn1Up>:	MatrixNotify()";

static XtActionsRec	actions[] = {
  	{ "MatrixSelect", (XtActionProc) MatrixSelect },
  	{ "MatrixNotify", (XtActionProc) MatrixNotify },
};

/*============================================================================*
 * 	Function declarations 
 *============================================================================*/
	
	/* Core methods 	    */
static void	ClassInitialize();
static void	Destroy();
static void	Initialize();
static void	Redisplay();
static void 	Resize();
static Boolean	SetValues();

	/* Matrix private functions */
static void	DrawGrid();
static void	DrawMatrix();
static void	DrawMatrixCell();
static void	FreeColorGCs(); 
static void	FreePixmapGCs();
static void	MapXYToRowCol();
static void	Realloc();
	
/*============================================================================*
 * 	Static allocation and initialization of Matrix class record
 *      Definition and initialization of class record pointer
 *============================================================================*/

MatrixClassRec matrixClassRec = {

	/* Core part */
	{
  	   &widgetClassRec,		/* superclass			*/
  	   "Matrix",			/* class_name			*/
  	   sizeof(MatrixRec),		/* widget_size			*/
  	   ClassInitialize,		/* class_initialize		*/
  	   NULL,			/* class_part_initialize 	*/
  	   FALSE,			/* class_inited			*/
  	   Initialize,			/* initialize			*/
  	   NULL,			/* initialize_hook		*/
  	   XtInheritRealize,		/* realize			*/
  	   actions,			/* actions			*/
  	   XtNumber(actions),		/* num_actions			*/
  	   resources,			/* resources			*/
  	   XtNumber(resources),		/* resource_count		*/
  	   NULL,			/* xrm_class			*/
  	   TRUE,			/* compress_motion		*/
  	   TRUE,			/* compress_exposure		*/
  	   TRUE,			/* compress_enterleave		*/
  	   FALSE,			/* visible_interest		*/
  	   Destroy,			/* destroy			*/
  	   Resize,			/* resize			*/
  	   Redisplay,			/* expose			*/
  	   SetValues,			/* set_values			*/
  	   NULL,			/* set_values_hook		*/
  	   XtInheritSetValuesAlmost,	/* set_values_almost		*/
  	   NULL,			/* get_values_hook		*/
  	   NULL,			/* accept_focus			*/
  	   XtVersion,			/* version			*/
  	   NULL,			/* callback_private		*/
  	   defaultTranslations,		/* tm_table			*/
  	   XtInheritQueryGeometry,	/* query_geometry          	*/
  	   XtInheritDisplayAccelerator,	/* display_accelerator    	*/
  	   NULL,			/* extension               	*/
	},
	/* Matrix part */
	{
	   0,				/* dummy 			*/
	}
};
	
WidgetClass matrixWidgetClass = (WidgetClass) &matrixClassRec;



/*============================================================================*
 * 	Function definitions for Core methods
 *============================================================================*/

/*----------------------------------------------------------------------------*
 *	static void ClassInitialize()
 *
 *	This method is called once by the X toolkit, the first time an
 *	instance of the Matrix class is created by the application.
 *	The class is only initialized once, and is shared among all
 *	instances of the class.  
 *	
 *	We use this to register non-standard type converters for the 
 *	Matrix class.
 *----------------------------------------------------------------------------*/

static void ClassInitialize()
{
  	XtAddConverter( XtRString, XtRBackingStore, XmuCvtStringToBackingStore,
		 NULL, 0);

} /* ClassInitialize() */


/*----------------------------------------------------------------------------*
 *	static void Initialize( request, new )
 *
 *	This method is called by the X toolkit to initialize the widget's
 *	instance variables.  It checks to be sure that public variables
 * 	have reasonable values and sets initial values for private
 *	instance variables.  The widget will not be realized yet (no window 
 *	will have been created) at the time this method is called.
 *
 *	The parameters <request> and <new> are widgets where <request>
 *	represents the widget before the superclass modified any of the
 *	variables, and <new> is the current state of the widget.  <request> 
 *	can be used to determine the original requested data.  The widget
 *	<new> will become the new widget.
 *----------------------------------------------------------------------------*/

static void Initialize( request, new )
MatrixWidget request, new;
{
	int		i;
	XtGCMask	GCmask;
	XGCValues	GCvalues;
	char		msgbuf[256];

#	ifdef DEBUG
	printf("MatrixWidget: Initialize\n");
#	endif

	/* 
	 * Verify that the window size is not zero.  
	 * The Core Initialize() method doesn't do this.		
	 */
	if ( new->core.width == 0 )
	  new->core.width = MINSIZE;
	if ( new->core.height == 0 )
	  new->core.height = MINSIZE;

	/*
	 * Verify that we have reasonable values for all the public
	 * instance variables (resources).
	 */

	if ( new->matrix.rows <= 0 ) {
		XtWarning( "Matrix: Rows must be greater than zero." );
		new->matrix.rows = 1;
	}

	if ( new->matrix.rows > MAXROWS ) {
		sprintf( msgbuf, 
			 "Matrix: Number of rows cannot exceed %d", MAXROWS );
		XtWarning( msgbuf );
		new->matrix.rows = MAXROWS;
	}

	if ( new->matrix.columns <= 0 ) {
		XtWarning( "Matrix: Columns must be greater than zero." );
		new->matrix.columns = 1;
	}
		
	if ( new->matrix.columns > MAXCOLUMNS ) {
		sprintf( msgbuf, "Matrix: Number of columns cannot exceed %d", 
				 MAXCOLUMNS );
		XtWarning( msgbuf );
		new->matrix.columns = MAXCOLUMNS;
	}

	if ( new->matrix.min_value >= new->matrix.max_value ) {
		XtWarning( "Matrix: Maximum must be greater than Minimum.");
		new->matrix.max_value = new->matrix.min_value + 1;
	}
	
	/*
	 * Initialize the private instance (state) variables, including
	 * the graphics contexts. So far the application hasn't had a 
	 * chance to set colors or pixmaps, so for now we put a single 
	 * color in the colorGC and set it to be the same as the grid color.
	 */
	new->matrix.values = NULL;

	GCmask = GCForeground | GCBackground;
	GCvalues.foreground = new->matrix.grid_color;
	GCvalues.background = new->core.background_pixel;
	new->matrix.gridGC = XtGetGC( (Widget)new, GCmask, &GCvalues );

	new->matrix.numpixmaps = 0;
	new->matrix.pixmapGCs = NULL;

	new->matrix.numcolors = 1;
	new->matrix.colorGCs = 
			(GC *) XtMalloc( new->matrix.numcolors * sizeof(GC) );

	GCmask = GCForeground | GCFillStyle;
	GCvalues.foreground = new->matrix.grid_color;
	GCvalues.fill_style = FillSolid;
	new->matrix.colorGCs[0] = XtGetGC( (Widget)new, GCmask, &GCvalues );

	/* 
	 * We call Resize here to set up state variables that relate to
	 * the display.  If Realize() were a local method, it would be
	 * better to call Resize() from there.  But, since we inherit the
	 * Realize() method, we do it here so that things are set properly
	 * when it does come time to do the drawing. 
	 */

	Resize( new );

} /* Initialize */


/*----------------------------------------------------------------------------*
	static void Destory()

	This method is called by the X toolkit before the Matrix widget 
	is destroyed.  
	
	We free space allocated to hold values, release graphics contexts,
	and remove our callbacks.
 *----------------------------------------------------------------------------*/

static void Destroy( w )
MatrixWidget w;
{
#	ifdef DEBUG
	printf( "MatrixWidget: Destroy\n" );
#	endif
	
	if ( w->matrix.values != NULL ) {
		XtFree( w->matrix.values );
	}

  	XtReleaseGC( w, w->matrix.gridGC );
  	FreeColorGCs( w );
  	FreePixmapGCs( w );
  	XtRemoveAllCallbacks( w, XtNselect );

} /* Destroy */


/*----------------------------------------------------------------------------*
 *	static void Resize( w )
 *
 *	This method is called when the parent widget resizes the widget.
 *	It is also called by Matrix::Initialize and Matrix::SetValues() to 
 *	update sizing-related when other instance variables they depend 
 *	on have changed.
 *
 *	Resize() recalculates the sizing-related state variables based on the 
 *	position and size of the window and characteristics of the matrix.
 *----------------------------------------------------------------------------*/

static void Resize( w ) 
MatrixWidget	w;

{
	int	gridw, gridh;

#	ifdef DEBUG
	printf( "MatrixWidget: Resize\n" );
#	endif

	/*
	 * We pad both sides and the bottom of the matrix with PADDING.
	 * Adjust the grid to be centered within those padded boundaries.
	 */

	w->matrix.elew = ( w->core.width - 2*PADDING ) / w->matrix.columns;
	w->matrix.eleh = ( w->core.height - PADDING ) / w->matrix.rows;

	gridw = w->matrix.elew * w->matrix.columns;
	gridh = w->matrix.eleh * w->matrix.rows;

	w->matrix.offsetx = ( w->core.width - gridw ) / 2;
	w->matrix.offsety = ( w->core.height - gridh - PADDING ) / 2;

} /* Resize */


/*----------------------------------------------------------------------------*
 *	static void Redisplay( w, event, region )
 *	
 *	This method is responsible for redrawing the widget whenever an Expose
 *	event arrives from the server. It may be called before the
 *	widget is realized, so we check that.
 *----------------------------------------------------------------------------*/

static void Redisplay( w, event, region )
MatrixWidget	w;
XEvent		*event;
Region		region;
{
#	ifdef DEBUG
	printf( "MatrixWidget: Redisplay\n" );
#	endif

	if (  XtIsRealized( w ) && w->core.visible  ) {
		DrawGrid( w );
		DrawMatrix( w );
	}

} /* Redisplay */


/*----------------------------------------------------------------------------*
 *	static Boolean SetValues( current, request, new )
 *	
 *	This method is called when the resource manager initializes the 
 *	widget's resources, or whenever the application calls XtSetValues to
 *	set the resources of the widget.  The method recalculates private
 *	state variables based on the new public instance variable values.
 *
 *	<current> contains the previous, unaltered state of the widget.
 *	<request> contains the values requested for the widget by a combination
 *	of the user's resource files, the widget default resources, and the 
 *	application.  <new> contains the state of the widget after all 
 *	superclass's SetValues() methods have been called.
 *----------------------------------------------------------------------------*/

static Boolean SetValues( current, request, new )
MatrixWidget	current, request, new;
{
	int		i;
	int		currentLimit, newLimit;
	XtGCMask	GCmask;
	XGCValues	GCvalues;
	Boolean		reSize = FALSE;
	Boolean		reDraw = FALSE;
	Boolean		reAlloc = FALSE;
	char		msgbuf[256];

#	ifdef DEBUG
	printf( "MatrixWidget: SetValues\n" );
#	endif

	/* change of width or height */
	if ( (new->core.width != current->core.width)  
	    || (new->core.height != current->core.height) ) {
	  	reSize = TRUE;
		reDraw = TRUE;
	}

	/* change show_grid */
	if ( new->matrix.show_grid != current->matrix.show_grid ) {
		reDraw = TRUE;
	}

	/* change grid color */
	if ( (new->matrix.grid_color != current->matrix.grid_color)
	    || (new->core.background_pixel != current->core.background_pixel) ){
	  	GCmask = GCForeground | GCBackground;
	  	GCvalues.foreground = new->matrix.grid_color;
	  	GCvalues.background = new->core.background_pixel;
	  	XtReleaseGC( current, current->matrix.gridGC );
	  	new->matrix.gridGC = XtGetGC( (Widget)new, GCmask, &GCvalues);
	  	reDraw = TRUE;
	}
	
	/* change number of rows */
	if ( new->matrix.rows != current->matrix.rows ) {
		if ( new->matrix.rows <= 0 ) {
		    XtWarning( "Matrix: Rows must be greater than zero." );
		    new->matrix.rows = 1;
		}
		if ( new->matrix.rows > MAXROWS ) {
		    sprintf( msgbuf, 
			     "Matrix: Number of rows cannot exceed %d", 
			     MAXROWS );
		    XtWarning( msgbuf );
		    new->matrix.rows = MAXROWS;
		}
		reAlloc = TRUE;
		reSize = TRUE;
		reDraw = TRUE;
	}

	/* change number of columns */
	if ( new->matrix.columns != current->matrix.columns ) {
		if ( new->matrix.columns <= 0 ) {
		    XtWarning( "Matrix: Columns must be greater than zero." );
		    new->matrix.columns = 1;
		}
		if ( new->matrix.columns > MAXCOLUMNS ) {
		    sprintf( msgbuf, 
			     "Matrix: Number of columns cannot exceed %d", 
			     MAXCOLUMNS );
		    XtWarning( msgbuf );
		    new->matrix.columns = MAXCOLUMNS;
		}
		reAlloc = TRUE;
		reSize = TRUE;
		reDraw = TRUE;
	}

	/* 
	 * Change minimum value. 
	 * We need to make sure that any "noValue" cells are still < newMin,
	 * and that any cells whose value was > currentMin but is < newMin are
	 * reset to newMin.
	 *
	 * Both current->matrix.values and new->matrix.values point to 
	 * the same array.  We use the current row and column values to 
	 * determine the size of array because the new ones may be
	 * different and the array itself hasn't been reAlloced yet. 
	 */
	if ( new->matrix.min_value != current->matrix.min_value ) {
		currentLimit = current->matrix.min_value;
		newLimit = new->matrix.min_value;

		for ( i = 0; i < current->matrix.rows*current->matrix.columns; 
			  						i++ ) {
		    if ( current->matrix.values[i] < currentLimit ) { 
			current->matrix.values[i] =  newLimit - 1;
		    } else if ( current->matrix.values[i] < newLimit ) {
			current->matrix.values[i] = newLimit;
		    }
		}
	}

	/* 
	 * Change maxmimum value.  
	 * If newMax < oldMax, make sure that any values > newMax
	 * are reset.  
	 *
	 * Both current->matrix.values and new->matrix.values point to 
	 * the same array.  We use the current row and column values to 
	 * determine the size of array because the new ones may be
	 * different and the array itself hasn't been reAlloced yet. 
	 */
	if ( new->matrix.max_value < current->matrix.max_value ) {
		newLimit = new->matrix.min_value;
		for ( i = 0; i < current->matrix.rows*current->matrix.columns; 
								        i++ ) {
		    if ( current->matrix.values[i] > newLimit ) { 
			current->matrix.values[i] = newLimit;
		    }
		}
	}

	/* reallocate array space, resize, and redraw as necessary */
	if ( reAlloc )
		Realloc( current, new );

	if ( reSize ) 
		Resize( new );

	return ( reDraw ); 
	
} /* SetValues */



/*============================================================================*
 * 	Function definitions for private methods
 *============================================================================*/

/*----------------------------------------------------------------------------*
 *	static void DrawGrid( w );
 *
 *	DrawGrid is called by the Matrix Redisplay() routine to clear
 *	the display area and, if the show_grid instance variable is true,
 *	to draw the grid lines. 
 *----------------------------------------------------------------------------*/

static void DrawGrid( w )
MatrixWidget	w;
{
	int 	i;

	/* Local copies of variables */
	int 	rows = w->matrix.rows;	
	int 	columns = w->matrix.columns;
	int 	offsetx = w->matrix.offsetx;
	int 	offsety = w->matrix.offsety;
	int 	elew = w->matrix.elew;
	int 	eleh = w->matrix.eleh;

#	ifdef DEBUG
	printf( "MatrixWidget: DrawGrid\n" );
#	endif

	XClearWindow( XtDisplay(w), XtWindow(w) );

	if ( w->matrix.show_grid ) {
		for ( i = 0; i < rows+1; i++ ) {
		    XDrawLine( XtDisplay(w), XtWindow(w), w->matrix.gridGC,
			       offsetx, offsety + i*eleh,
			       offsetx + columns*elew, offsety + i*eleh);
		}

		for( i = 0;  i < columns+1; i++ ) {
 		    XDrawLine( XtDisplay(w), XtWindow(w), w->matrix.gridGC,
			       offsetx + i*elew, offsety,
			       offsetx + i*elew, offsety + rows*eleh);
		}
	}

} /* DrawGrid */


/*----------------------------------------------------------------------------*
 *	static void DrawMatrix( w );
 *
 *	DrawMatrix is called by the Matrix Redisplay() and MatrixSetValues()
 *	routines to redraw all cells of the entire matrix.  If values <
 *	the minimum are seen, then that cell hasn't been set yet and is 
 *	cleared.  This should only be true if we use MatrixSetValue() to
 *	set cell values and the encounter a Redisplay().  When MatrixSetValues()
 *	is used, then the values outside of the allowed min,max range are 
 *	remapped and we do not have any "noValue" cells.
 *----------------------------------------------------------------------------*/

static void DrawMatrix( w )
MatrixWidget	w;
{
	int	i, j; 
	int	v;
	int	gcidx;
	int	grid = 0;
	int	elew = w->matrix.elew, eleh = w->matrix.eleh;
	int	offsetx = w->matrix.offsetx, offsety = w->matrix.offsety;
	int	rows = w->matrix.rows, columns = w->matrix.columns;
	int	*values = w->matrix.values;

#	ifdef DEBUG
	printf( "MatrixWidget: DrawMatrix\n" );
#	endif

	if ( values == NULL )
	  	return;

	if ( w->matrix.show_grid )
	  	grid = 1;

	for ( i = 0; i < rows; i++ ) {
	  	for ( j = 0; j < columns; j++ ) {
		    v = values[ i*columns + j ];
		    if ( v < w->matrix.min_value ) {
		        XClearArea( XtDisplay(w), XtWindow(w),
			            offsetx+j*elew+grid, offsety+i*eleh+grid,
				    elew-grid, eleh-grid, 0 );
		    } else {
	    	    	if ( w->matrix.numpixmaps > 0 ) {
			    gcidx = norm( w, v ) * w->matrix.numpixmaps;
			    XFillRectangle( XtDisplay(w), XtWindow(w),
		       			    w->matrix.pixmapGCs[gcidx],
		       			    offsetx+j*elew+grid, 
					    offsety+i*eleh+grid,
		       			    elew-grid, eleh-grid );
		    	} else if ( w->matrix.numcolors > 0 ) {
			    gcidx = norm( w, v ) * w->matrix.numcolors;
	  		    XFillRectangle( XtDisplay(w), XtWindow(w),
			 		    w->matrix.colorGCs[gcidx],
			 		    offsetx+j*elew+grid, 
					    offsety+i*eleh+grid,
			 		    elew-grid, eleh-grid );
	 		}
	     	    }
	  	}
	}

} /* DrawMatrix */


/*----------------------------------------------------------------------------*
 *	static void DrawMatrixCell( w, r, c );
 *
 *	DrawMatrixCell is called by Matrix::MatrixSetValue() to redraw a 
 *	single cell in the matrix.  We do not check for a value < minimum
 *	indicating a "noValue" cell, as the cell must have been set in
 *	MatrixSetValue() before the call here was made.
 *----------------------------------------------------------------------------*/

static void DrawMatrixCell(w, r, c)
MatrixWidget	w;
int		r, c;
{
	int	v;
	int	gcidx;
	int	grid = 0;
	int	elew = w->matrix.elew, eleh = w->matrix.eleh;
	int	offsetx = w->matrix.offsetx, offsety = w->matrix.offsety;

#	ifdef DEBUG
	printf( "MatrixWidget: DrawMatrixCell\n" );
#	endif

	if ( w->matrix.show_grid )
	  grid = 1;

	v = w->matrix.values[ r*w->matrix.columns + c ];
	if ( w->matrix.numpixmaps > 0 ) {
	    	gcidx = norm( w, v ) * w->matrix.numpixmaps;
	    	XFillRectangle( XtDisplay(w), XtWindow(w),
	    		        w->matrix.pixmapGCs[gcidx],
		       	        offsetx+c*elew+grid, 
				offsety+r*eleh+grid,
		       	    	elew-grid, eleh-grid );
	} else if ( w->matrix.numcolors > 0 ) {
	    	gcidx = norm( w, v ) * w->matrix.numcolors;
	      	XFillRectangle( XtDisplay(w), XtWindow(w),
		 	 	w->matrix.colorGCs[gcidx],
			    	offsetx+c*elew+grid, 
			    	offsety+r*eleh+grid,
	 		    	elew-grid, eleh-grid );
	}

} /* DrawMatrixCell */

/*----------------------------------------------------------------------------*
 *	static void FreeColorGCs( w );
 *
 *	FreeColorGCs is called by other routines of the Matrix widget
 *	to free the Color graphics contexts when appropriate.  It also
 *	sets numcolors to zero.
 *----------------------------------------------------------------------------*/

static void FreeColorGCs( w )
MatrixWidget	w;
{
	int	i;
	
#	ifdef DEBUG
	printf( "MatrixWidget: FreeColorGCs\n" );
#	endif

	if ( w->matrix.numcolors == 0 ) {
		return;
	}

	for ( i = 0; i < w->matrix.numcolors; i++ ) {
	    	XtReleaseGC( w, w->matrix.colorGCs[i] );
	}
	XtFree( w->matrix.colorGCs );
	w->matrix.numcolors = 0;

} /* FreeColorGCs */


/*----------------------------------------------------------------------------*
 *	static void FreePixmapGCs( w );
 *
 *	FreePixmapGCs is called by other routines of the Matrix widget
 *	to free the Pixmap graphics contexts when appropriate.  It also
 *	sets numpixmaps to zero.
 *----------------------------------------------------------------------------*/

static void FreePixmapGCs( w )
MatrixWidget	w;
{
	
	int	i;
	
#ifdef DEBUG
	printf( "MatrixWidget: FreePixmapGCs\n" );
#endif

	if ( w->matrix.numpixmaps == 0 ) {
		return;
	}

	for ( i = 0; i < w->matrix.numpixmaps; i++ ) {
	    	XtReleaseGC( w, w->matrix.pixmapGCs[i] );
	}
	XtFree( w->matrix.pixmapGCs );
	w->matrix.numpixmaps = 0;

} /* FreePixmapGCs */


/*----------------------------------------------------------------------------*
 *	static void MapXYToRowCol( w, x, y );
 *
 *	MapXYToRowCol() takes (x,y) coordinates in the matrix widget window
 * 	and maps them to matrix row, column values, updating row_select and
 *	column_select in the matrix state variables with the results.
 *	A -1 in either the row or column value indicates the respective x or y 
 *	coordinate was outside the matrix boundaries.
 *----------------------------------------------------------------------------*/

static void MapXYToRowCol( w, x, y )
MatrixWidget	w;
int		x, y;
{
	int	i;
	int	elew = w->matrix.elew, eleh = w->matrix.eleh;
	int	offsetx = w->matrix.offsetx, offsety = w->matrix.offsety;

#ifdef DEBUG
	printf( "MatrixWidget: MapXYToRowCol\n" );
#endif

	for ( i = 0; i < w->matrix.columns; i++ ) {
	  	if ( ( (offsetx + i*elew) <= x ) 
		      && ( x <= (offsetx + (i+1)*elew - 1) ) ) {
	    	    break;
		}
	}
	if ( i == w->matrix.columns )
	  	w->matrix.column_select = -1;
	else
	  	w->matrix.column_select = i;

	for ( i = 0; i < w->matrix.rows; i++ ) {
	  	if ( ( (offsety + i*eleh) <= y ) 
		     && ( y <= (offsety + (i+1)*eleh - 1) ) ) {
	     	    break;
		}
	}
	if ( i == w->matrix.rows )
	  	w->matrix.row_select = -1;
	else
	  	w->matrix.row_select = i;

} /* MapXYToRowCol */

/*----------------------------------------------------------------------------*
 *	static void Realloc( old, new );
 *
 *	Realloc is called by Matrix::SetValues when a new array dimension
 *	is seen.  It allocates space for the new matrix, copies values
 *	from the old matrix into the new, populates any "extra" cells with
 *	min_value-1, and releases the old matrix space.
 *----------------------------------------------------------------------------*/

static void Realloc( old, new )
MatrixWidget	old, new;
{
	int	r, c;
	int	oldr = old->matrix.rows;	/* local copies of vars */
	int	oldc = old->matrix.columns;
	int	newr = new->matrix.rows;
	int	newc = new->matrix.columns;
	int	noValue = new->matrix.min_value - 1;
	
#ifdef DEBUG
	printf( "MatrixWidget: Realloc\n" );
#endif

	if ( old->matrix.values == NULL ) {	/* no values to copy    */
		return;
	}

  	new->matrix.values = (int *) XtMalloc( newr * newc * sizeof(int) );

	for ( r = 0; r < newr; r++ ) {
		for ( c = 0; c < newc; c++ ) {
		    if ( r < oldr && c < oldc ) {
			new->matrix.values[r * newc + c] = 
					old->matrix.values[r * oldc + c];
		    } else {
			new->matrix.values[r * newc + c] = noValue;
		    }
		}
	}

	XtFree( old->matrix.values );

} /* Realloc */


/*============================================================================*
 * 	Function definitions for public methods
 *============================================================================*/

/*----------------------------------------------------------------------------*
 *	extern void MatrixGetRowCol( w, row, col )
 *
 *	MatrixGetRowCol is called by the application code to get the currently
 *	selected row and column.
 *
 *	<w> is the Matrix Widget.
 *	<row> is a pointer to an integer where the row value will be put.
 *	<col> is a pointer to an integer where the column value will be put.
 *----------------------------------------------------------------------------*/

extern void MatrixGetRowCol(w, row, col)
MatrixWidget	w;
int		*row, *col;
{
#ifdef DEBUG
	printf( "MatrixWidget: MatrixGetRowCol\n" );
#endif

	*row = w->matrix.row_select;
	*col = w->matrix.column_select;

} /* MatrixGetRowCol */


/*----------------------------------------------------------------------------*
 *	extern void MatrixSetColors( w, numcolors, colors )
 *
 *	MatrixSetColors is called by the application code to set up a color
 *	table for the Matrix widget.
 *
 *	<w> is the Matrix Widget.
 *	<numcolors> is the number of colors we will be setting
 *	<colors> is a pointer to a Pixel array with <numcolors> elements 
 *----------------------------------------------------------------------------*/

extern void MatrixSetColors( w, numcolors, colors )
MatrixWidget	w;
int		numcolors;
Pixel		*colors;
{
	int 		i;
	XGCValues	gcValues;
	XtGCMask	gcMask;

#ifdef DEBUG
	printf("MatrixWidget: MatrixSetColors\n");
#endif

	FreeColorGCs( w );
	FreePixmapGCs( w );		/* In case we were using pixmaps */

	w->matrix.numcolors = numcolors;
	w->matrix.colorGCs = 
			(GC *) XtMalloc( w->matrix.numcolors * sizeof(GC) );
	  
	gcValues.fill_style = FillSolid;
	gcMask = GCForeground | GCFillStyle;
	for ( i=0; i < numcolors; i++) {
	  	gcValues.foreground = colors[i];
	  	w->matrix.colorGCs[i] = XtGetGC( w, gcMask, &gcValues );
	}

	if (  XtIsRealized( w ) && w->core.visible  ) {
		DrawMatrix(w);
	}

} /* MatrixSetColors */

/*----------------------------------------------------------------------------*
 *	extern void MatrixSetPixmaps( w, numpixmaps, pixmaps )
 *
 *	MatrixSetPixmaps is called by the application code to set up a pixmap
 *	table for the Matrix widget.
 *
 *	<w> is the Matrix Widget.
 *	<numpixmaps> is the number of pixmaps we will be setting
 *	<pixmaps> is a pointer to a Pixmap array with <numpixmaps> elements 
 *----------------------------------------------------------------------------*/

extern void MatrixSetPixmaps( w, numpixmaps, pixmaps )
MatrixWidget	w;
int		numpixmaps;
Pixmap		*pixmaps;
{
	int		i;
	XGCValues	gcValues;
	XtGCMask	gcMask;

#ifdef DEBUG
	printf( "MatrixWidget: MatrixSetPixmaps\n" );
#endif

	FreePixmapGCs( w );
	FreeColorGCs( w );		/* In case we were using colors */ 

	w->matrix.numpixmaps = numpixmaps;
	w->matrix.pixmapGCs = 
			  (GC *) XtMalloc( w->matrix.numpixmaps * sizeof(GC) );

	gcValues.background = w->core.background_pixel;
	gcValues.foreground = w->matrix.grid_color;
	gcValues.fill_style = FillTiled;
	gcMask = GCBackground | GCForeground | GCFillStyle | GCTile;
	for ( i = 0; i < numpixmaps; i++) {
		gcValues.tile = pixmaps[i];
	  	w->matrix.pixmapGCs[i] = XtGetGC( w, gcMask, &gcValues );
	}

	if (  XtIsRealized( w ) && w->core.visible  ) {
		DrawMatrix(w);
	}

} /* MatrixSetPixmaps */


/*----------------------------------------------------------------------------*
 *	extern void MatrixSetValues( w, values )
 *
 *	MatrixSetValues is called by the application code to set the values
 *	for all cells in the Matrix.  There is no check that the number of
 *	integers pointed to by values equals (rows * columns) so be careful!
 *----------------------------------------------------------------------------*/

extern void MatrixSetValues( w, values )
MatrixWidget    w;
int		*values;
{
	int	i, j, v;
	int	rows = w->matrix.rows;
	int	cols = w->matrix.columns;
	int	minv = w->matrix.min_value;
	int	maxv = w->matrix.max_value;
	char	msgbuf[256];

#ifdef DEBUG
	printf( "MatrixWidget: MatrixSetValues\n" );
#endif

	if ( w->matrix.values == NULL ) {
	    w->matrix.values = (int *)XtMalloc( rows * cols * sizeof(int) );
	}

	for ( i = 0; i < rows; i++ ) {
	    for ( j = 0; j < cols; j++ ) {
		v = values[ i * cols + j ];
		if ( v < minv )  {
	           sprintf( msgbuf, 
		  "MatrixSetValues: Value %d for (%d,%d) too small - using %d",
			       v, i, j, minv );
		   XtWarning( msgbuf );
		   v = minv;
		} else if ( v > maxv ) {
		   sprintf( msgbuf, 
		  "MatrixSetValues: Value %d for (%d,%d) too large - using %d",
			       v, i, j, maxv );
		    XtWarning( msgbuf );
		    v = maxv;
		}
		w->matrix.values[i * cols + j] = v;
             }
	}

	if ( XtIsRealized( w ) && w->core.visible ) {
		DrawMatrix( w );
	}

} /* MatrixSetValues */

/*----------------------------------------------------------------------------*
 *	extern void MatrixSetValue( w, r, c, v )
 *
 *	MatrixSetValue is called by the application code to set the value
 *	of a single cell Matrix[r,c] to v. 
 *----------------------------------------------------------------------------*/

extern void MatrixSetValue( w, r, c, v )
MatrixWidget    w;
int		r, c;
int		v;
{
  	int 	rows = w->matrix.rows;
  	int 	cols = w->matrix.columns;
  	int	minv = w->matrix.min_value;
  	int	maxv = w->matrix.max_value;
	int	noValue = minv - 1;
	int	i, j;
	int	val;
	char	msgbuf[256];
#ifdef DEBUG
	printf("MatrixSetValue - r:%d c:%d v: %d\n", r, c, v);
#endif

  	if ( ( r > rows ) || ( c > cols ) ) {
     		sprintf( msgbuf, 
	           "MatrixSetValue: (%d,%d) outside display range of (%d,%d)",
        	                    r, c, rows, cols );
		XtWarning( msgbuf );
     		return;
  	}

  	if ( w->matrix.values == NULL ) {
         	w->matrix.values = (int *) XtMalloc( rows * cols * sizeof(int));
         	for ( i = 0; i < rows * cols; i++ ) {
               		w->matrix.values[i] = noValue;
                } 
      	}

	val = v;
        if ( val < minv )  {
		sprintf( msgbuf, 
		  "MatrixSetValue: Value %d for (%d,%d) too small - using %d",
			 val, r, c, minv );
		XtWarning( msgbuf );
		val = minv;
	} else if ( val > maxv ) {
		sprintf( msgbuf, 
	           "MatrixSetValue: Value %d for (%d,%d) too large - using %d",
			 val, r, c, maxv );
		XtWarning( msgbuf );
		val = maxv;
	}
	w->matrix.values[r * cols + c] = val;

	if ( XtIsRealized( w ) && w->core.visible ) {
		DrawMatrixCell( w, r, c );
	}
} /* MatrixSetValue */

/*============================================================================*
 * 	Function definitions for action methods
 *============================================================================*/

/*----------------------------------------------------------------------------*
 *	static void MatrixSelect( w, event)
 *
 *	MatrixSelect processes a button down event, calling MapXYToRowCol
 *	to set the widget's row_select and column_select state variables.
 *----------------------------------------------------------------------------*/

static void MatrixSelect( w, event )
MatrixWidget		w;
XButtonPressedEvent	*event;
{
#ifdef DEBUG
	printf( "MatrixWidget: MatrixSelect\n" );
#endif

	MapXYToRowCol( w, event->x, event->y );

} /* MatrixSelect */


/*----------------------------------------------------------------------------*
 *	static void MatrixNotify( w, event)
 *
 *	MatrixNotify processes a button up event, calling the Callback routine
 *	if one has been registered.
 *----------------------------------------------------------------------------*/
static void MatrixNotify( w, event )
MatrixWidget		w;
XButtonReleasedEvent	*event;
{
	XtCallCallbacks( w, XtNselect, event );

} /* MatrixNotify */



