/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author:  Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/piechart/RCS/Piechart.c,v 1.11 1994/02/25 04:47:07 aydt Exp $
 */

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include <X11/keysymdef.h>
#include <math.h>
#include "PiechartP.h"

#define Round(x)	(x >= 0.0 ? (int)(x + .5) : (int)(x - .5))
#define Min(a, b)	((a) < (b) ? (a) : (b))
#define offset(field)	XtOffset(PiechartWidget,field)
#define norm(w, v)	(((float)v-(float)w->piechart.min_value) /	\
			 ((float)w->piechart.max_value -		\
			  (float)w->piechart.min_value + 1))

static void	PiechartSelect(), PiechartNotify();
static char piechart_translations[] = "	\
  <Btn1Down>:	PiechartSelect()\n	\
  <Btn1Up>:	PiechartNotify()		\
";

static XtActionsRec	piechart_actions[] = {
  {"PiechartSelect",	(XtActionProc) PiechartSelect	},
  {"PiechartNotify",	(XtActionProc) PiechartNotify	},
};

static XtResource resources[] = {
  /* core resources */
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     offset(core.width), XtRString, "200"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     offset(core.height), XtRString, "100"},
  {XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
     offset(core.background_pixel), XtRString, "white"},
  /* piechart resources */
  {XtNgridColor, XtCGridColor, XtRPixel, sizeof(Pixel),
     offset(piechart.grid_color), XtRString, "black"},
  {XtNfaceBorderColor, XtCFaceBorderColor, XtRPixel, sizeof(Pixel),
     offset(piechart.face_border_color), XtRString, "black"},
  {XtNcolors, XtCColors, XtRPointer, sizeof(Pixel *),
     offset(piechart.colors), XtRString, NULL},
  {XtNnumColors, XtCNumColors, XtRInt, sizeof(int),
     offset(piechart.numcolors), XtRString, "0"},
  {XtNpixmaps, XtCPixmaps, XtRPointer, sizeof(Pixel *),
     offset(piechart.pixmaps), XtRString, NULL},
  {XtNnumPixmaps, XtCNumPixmaps, XtRInt, sizeof(int),
     offset(piechart.numpixmaps), XtRString, "0"},
  {XtNentries, XtCEntries, XtRInt, sizeof(int), 
     offset(piechart.entries), XtRString, "8" },
  {XtNshowGrid, XtCShowGrid, XtRBoolean, sizeof (Boolean),
     offset(piechart.show_grid), XtRString, "FALSE"},
  {XtNminValue, XtCMinValue, XtRInt, sizeof(int),
     offset(piechart.min_value), XtRString, "0"},
  {XtNmaxValue, XtCMaxValue, XtRInt, sizeof(int),
     offset(piechart.max_value), XtRString, "100"},
  {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     offset(piechart.select), XtRCallback, (caddr_t) NULL},
};

#undef offset

static void	ClassInitialize(), Initialize();
static void	Destroy(), Resize(), Redisplay();
static Boolean	SetValues();

static void	CreateColorGCs(), CreatePixmapGCs();
static void	DestroyColorGCs(), DestroyPixmapGCs();
static void	DrawGrid(), DrawPiechart(), DrawFace();
	
/* public procedures */
extern void	PiechartSetValues();
extern void	PiechartSetColors(), PiechartSetPixmaps();
extern void	PiechartGetSelectedPie();

PiechartClassRec piechartClassRec = {
  &widgetClassRec,		/* superclass			*/
  "Piechart",			/* class_name			*/
  sizeof(PiechartRec),		/* widget_size			*/
  ClassInitialize,		/* class_initialize		*/
  NULL,				/* class_part_initialize	*/
  FALSE,			/* class_inited			*/
  Initialize,			/* initialize			*/
  NULL,				/* initialize_hook		*/
  XtInheritRealize,		/* realize			*/
  piechart_actions,		/* actions			*/
  XtNumber(piechart_actions),	/* num_actions			*/
  resources,			/* resources			*/
  XtNumber(resources),		/* resource_count		*/
  NULL,				/* xrm_class			*/
  TRUE,				/* compress_motion		*/
  TRUE,				/* compress_exposure		*/
  TRUE,				/* compress_enterleave		*/
  FALSE,			/* visible_interest		*/
  Destroy,			/* destroy			*/
  Resize,			/* resize			*/
  Redisplay,			/* expose			*/
  SetValues,			/* set_values			*/
  NULL,				/* set_values_hook		*/
  XtInheritSetValuesAlmost,	/* set_values_almost		*/
  NULL,				/* get_values_hook		*/
  NULL,				/* accept_focus			*/
  XtVersion,			/* version			*/
  NULL,				/* callback_private		*/
  piechart_translations,		/* tm_table			*/
  XtInheritQueryGeometry,	/* query_geometry          	*/
  XtInheritDisplayAccelerator,	/* display_accelerator    	*/
  NULL				/* extension               	*/
};

WidgetClass piechartWidgetClass = (WidgetClass) &piechartClassRec;

static void ClassInitialize()
{
  XtAddConverter(XtRString, XtRBackingStore, XmuCvtStringToBackingStore,
		 NULL, 0);
}

/* ARGSUSED */
static void Initialize(request, new)
PiechartWidget request, new;
{
  int		i;
  XtGCMask	GCmask;
  XGCValues	GCvalues;

#ifdef DEBUG
printf("Initialize\n");
#endif
  for (i=0; i < MAXENTRIES; i++)
    new->piechart.values[i] = new->piechart.min_value;

  if (new->core.width == 0)
    new->core.width = MINSIZE;
  if (new->core.height == 0)
    new->core.height = MINSIZE;

  GCmask = GCForeground | GCBackground;
  GCvalues.background = new->core.background_pixel;
  GCvalues.foreground = new->piechart.grid_color;
  new->piechart.gridGC = XtGetGC((Widget)new, GCmask, &GCvalues);

  GCmask = GCForeground | GCBackground;
  GCvalues.background = new->core.background_pixel;
  GCvalues.foreground = new->piechart.face_border_color;
  new->piechart.face_borderGC = XtGetGC((Widget)new, GCmask, &GCvalues);

  GCmask = GCForeground;
  GCvalues.foreground = new->core.background_pixel;
  new->piechart.eraseGC = XtGetGC((Widget)new, GCmask, &GCvalues);

  Resize(new);

} /* Initialize */

/* ARGSUSED */
static Boolean SetValues(current, grequest, new)
PiechartWidget	current, grequest, new;
{
  int			i;
  Boolean		redraw = FALSE;
  XtGCMask		GCmask;
  XGCValues		GCvalues;

#ifdef DEBUG
printf("SetValues\n");
#endif
  /* change of width or height */
  if ((new->core.height != current->core.height)
      || (new->core.width != current->core.width)) {
    Resize(new);
    redraw = TRUE;
  }


  /* change grid color */
  if ((new->piechart.grid_color != current->piechart.grid_color)
      || (new->core.background_pixel != current->core.background_pixel)) {
    GCmask = GCForeground | GCBackground;
    GCvalues.foreground = new->piechart.grid_color;
    GCvalues.background = new->core.background_pixel;
    XtReleaseGC(current, current->piechart.gridGC);
    new->piechart.gridGC = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }

 /* change face border color */
  if((new->piechart.face_border_color != current->piechart.face_border_color)
      ||(new->core.background_pixel != current->core.background_pixel)) {
    GCmask = GCForeground | GCBackground;
    GCvalues.foreground = new->piechart.face_border_color;
    GCvalues.background = new->core.background_pixel;
    XtReleaseGC(current, current->piechart.face_borderGC);
    new->piechart.face_borderGC = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }
  
  /* change number of entries */
  if (new->piechart.entries != current->piechart.entries) {
    if (new->piechart.entries > MAXENTRIES) {
      XtWarning("PieChart: exceeded maximum number of entries");
      new->piechart.entries = MAXENTRIES;
    }
    redraw = TRUE;
  }

  /* change colors */
  if (new->piechart.colors != current->piechart.colors) {
    DestroyColorGCs(current);
    CreateColorGCs(new);
    redraw = TRUE;
  }

 /* change erase color */
  if(new->core.background_pixel != current->core.background_pixel) {
    GCmask = GCForeground;
    GCvalues.foreground = new->core.background_pixel;
    XtReleaseGC(current, current->piechart.eraseGC);
    new->piechart.eraseGC = XtGetGC((Widget)current, GCmask, &GCvalues);
    redraw = TRUE;
  }

  /* change of pixmaps */
  if (new->piechart.pixmaps != current->piechart.pixmaps) {
    DestroyPixmapGCs(current);
    CreatePixmapGCs(new);
    redraw = TRUE;
  }

  return(redraw);
  
} /* SetValues */


static void Destroy(w)
PiechartWidget w;
{
  int		i;

#ifdef DEBUG
printf("Destroy\n");
#endif
/*
  XtReleaseGC(current, w->piechart.face_borderGC);
  XtReleaseGC(current, w->piechart.gridGC);
  XtReleaseGC(current, w->piechart.eraseGC);
  DestroyColorGCs(w);
  DestroyPixmapGCs(w);
*/
  XtRemoveAllCallbacks(w, XtNselect);

} /* Destroy */

static void Resize(w) 
PiechartWidget	w;
{
#ifdef DEBUG
printf("Resize\n");
#endif

    w->piechart.radius= ( Min(w->core.width,w->core.height) - PADDING*2) / 2;
    w->piechart.centerX = w->core.width / 2;
    w->piechart.centerY = w->core.height / 2;


} /* Resize */

/* ARGSUSED */
static void Redisplay(w, event, region)
PiechartWidget	w;
XEvent	*event;				/* unused */
Region	region;				/* unused */
{
  
#ifdef DEBUG
printf("Redisplay\n");
#endif

  if (XtIsRealized( w) && w->core.visible ) {
     DrawFace(w);
     DrawGrid(w);
     DrawPiechart(w);
  }

} /* Redisplay */

static void CreateColorGCs(w)
PiechartWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreateColorGCs\n");
#endif

  gcValues.fill_style = FillSolid;
  gcMask = GCForeground | GCFillStyle;
  for (i=0; i<w->piechart.numcolors; i++) {
    gcValues.foreground = w->piechart.colors[i];
    w->piechart.colorGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreateColorGCs */

static void CreatePixmapGCs(w)
PiechartWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreatePixmapGCs\n");
#endif
  gcValues.background = w->core.background_pixel;
  gcValues.foreground = w->piechart.grid_color;
  gcValues.fill_style = FillTiled;
  gcMask = GCBackground | GCForeground | GCFillStyle | GCTile;
  for (i=0; i<w->piechart.numpixmaps; i++) {
    gcValues.tile = w->piechart.pixmaps[i];
    w->piechart.pixmapGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreatePixmapGCs */

static void DestroyColorGCs(w)
PiechartWidget	w;
{
  register	i;
  
#ifdef DEBUG
printf("DestroyColorGCs\n");
#endif
  if (w->piechart.colors != NULL)
    for (i=0; i<w->piechart.numcolors; i++) {
      XtReleaseGC(w, w->piechart.colorGCs[i]);
    }

} /* DestroyColorGCs */

static void DestroyPixmapGCs(w)
PiechartWidget	w;
{
  register	i;
  
#ifdef DEBUG
printf("DestroyPixmapGCs\n");
#endif
  if (w->piechart.pixmaps != NULL)
    for (i=0; i<w->piechart.numpixmaps; i++) {
      XtReleaseGC(w, w->piechart.pixmapGCs[i]);
    }

} /* DestroyPixmapGCs */

static void DrawGrid(w)
PiechartWidget	w;
{
  int		i;
  int		perimx, perimy;
  double	fraction;
  double	angle, cosangle, sinangle;
  double	total;
	
#ifdef DEBUG
printf("DrawGrid\n");
#endif

  if (w->piechart.show_grid) {

     if (w->piechart.entries <= 1)
	return;

     total = 0.0;
     for (i = 0; i < w->piechart.entries; i++) {
        total += (double) (w->piechart.values[i]);
     }

     fraction = 0.0;
     for(i=0; i<w->piechart.entries; i++) {
         angle = TWOPI * fraction;
         cosangle = cos(angle);
         sinangle = sin(angle);
         perimx = Round(w->piechart.centerX +
           ((double) w->piechart.radius) * cosangle);
         perimy = Round(w->piechart.centerY -
           ((double) w->piechart.radius) * sinangle);
         XDrawLine(XtDisplay(w), XtWindow(w), w->piechart.gridGC,
              w->piechart.centerX, w->piechart.centerY, perimx, perimy);

	 fraction += ((double) w->piechart.values[i]) / total;
     }
  }

} /* DrawGrid */

/************************************************************************/
/* DrawFace()                                                     */
/************************************************************************/
static void DrawFace(w)
PiechartWidget    w;
{
  register int  i;
  int           perimx, perimy;

#ifdef DEBUG
printf("DrawFace\n");
#endif
 XClearWindow( XtDisplay(w), XtWindow(w) );
 XFillArc(XtDisplay(w), XtWindow(w), w->piechart.eraseGC,
           w->piechart.centerX - w->piechart.radius,
           w->piechart.centerY - w->piechart.radius,
           2*w->piechart.radius, 2*w->piechart.radius,
           0, 360*64);

  XDrawArc(XtDisplay(w), XtWindow(w), w->piechart.face_borderGC,
           w->piechart.centerX - w->piechart.radius,
           w->piechart.centerY - w->piechart.radius,
           2*w->piechart.radius, 2*w->piechart.radius,
           0, 360*64);

} /* DrawFace */

static void DrawPiechart(w)
PiechartWidget	w;
{
  int		i, f1, f2;
  int		color, pixmap;
  double        fraction, oldfraction;
  double        total;

#ifdef DEBUG
printf("DrawPiechart\n");
#endif

  if (w->piechart.entries == 0)
    return;

  total = 0.0;
  for (i = 0; i < w->piechart.entries; i++) {
        total += (double) (w->piechart.values[i]);
  }

  fraction = 0.0;
  oldfraction = 0.0;

  for (i=0; i< w->piechart.entries; i++) {
      fraction = ((double) w->piechart.values[i]) / total;

      f1 = Round(oldfraction*64*360);
      f2 = Round(fraction*64*360);

      if (w->piechart.colors != NULL) {
	 color = i % (w->piechart.numcolors);
         XFillArc(XtDisplay(w), XtWindow(w), w->piechart.colorGCs[color],
              w->piechart.centerX - w->piechart.radius,
              w->piechart.centerY - w->piechart.radius,
              2*w->piechart.radius, 2*w->piechart.radius,
              f1, f2);
      } else if (w->piechart.pixmaps != NULL) {
	 pixmap = i % (w->piechart.numpixmaps);
	 XFillArc(XtDisplay(w), XtWindow(w), w->piechart.pixmapGCs[pixmap],
              w->piechart.centerX - w->piechart.radius,
              w->piechart.centerY - w->piechart.radius,
              2*w->piechart.radius, 2*w->piechart.radius,
              f1, f2);
      }


      oldfraction += fraction;
    }

} /* DrawPiechart */

/*********************/
/* public procedures */
/*********************/

extern void PiechartSetValues(w, entries, values)
PiechartWidget    w;
int		entries;
int		*values;
{
  register int	i;

#ifdef DEBUG
printf("PiechartSetValues\n");
#endif

 if (entries > w->piechart.entries ) {
    if (entries > MAXENTRIES ) {
       XtWarning("PieChart: exceeded maximum number of entries");
       entries = MAXENTRIES;
    }
    w->piechart.entries = entries;
    if (XtIsRealized( w) && w->core.visible ) {
        DrawFace( w );
        DrawGrid( w );
    }
 } 

 for (i=0; i<entries; i++) {
    if (values[i] < w->piechart.min_value) {
      printf("PiechartSetValues: tried to set an entry value (%d) ",values[i]);
      printf("less than the min (%d)\n", w->piechart.min_value);
      w->piechart.values[i] = w->piechart.min_value;
    }
    else if (values[i] > w->piechart.max_value) {
      printf("PiechartSetValues: tried to set an entry value (%d) ",values[i]);
      printf("greater than the max (%d)\n", w->piechart.max_value);
      w->piechart.values[i] = w->piechart.max_value;
    }
    else
      w->piechart.values[i] = values[i];
  }

  Redisplay(w);

} /* PiechartSetValues */

extern void PiechartSetValue(w, entry, v)
PiechartWidget    w;
int		entry;
int		v;
{
#ifdef DEBUG
printf("PiechartSetValue\n");
#endif

   if ((entry < 0) || (entry > w->piechart.entries)) {
	printf("PiechartSetValue:  tried to set a value outside range (%d)\n",
		entry);
	return;
   }

   if (v < w->piechart.min_value) {
      printf("PiechartSetValue: tried to set an entry value (%d) ", v);
      printf("less than the min (%d)\n", w->piechart.min_value);
      w->piechart.values[entry] = w->piechart.min_value;
    }
    else if (v > w->piechart.max_value) {
      printf("PiechartSetValue: tried to set an axis value (%d) ", v);
      printf("greater than the max (%d)\n", w->piechart.max_value);
      w->piechart.values[entry] = w->piechart.max_value;
    }
    else
      w->piechart.values[entry] = v;

  Redisplay(w);

} /* PiechartSetValue */

extern void PiechartSetColors(w, numcolors, colors)
PiechartWidget	w;
int		numcolors;
Pixel		*colors;
{
#ifdef DEBUG
printf("PiechartSetColors\n");
#endif
  DestroyColorGCs(w);
  w->piechart.numcolors = numcolors;
  w->piechart.colors = colors;
  CreateColorGCs(w);
  if (XtIsRealized( w) && w->core.visible ) {
    DrawPiechart(w);
  }

} /* PiechartSetColors */

extern void PiechartSetPixmaps(w, numpixmaps, pixmaps)
PiechartWidget	w;
int		numpixmaps;
Pixmap		*pixmaps;
{
#ifdef DEBUG
printf("PiechartSetPixmaps\n");
#endif
  DestroyPixmapGCs(w);
  w->piechart.numpixmaps = numpixmaps;
  w->piechart.pixmaps = pixmaps;
  CreatePixmapGCs(w);
  if (XtIsRealized( w) && w->core.visible ) {
    DrawPiechart(w);
  }

} /* PiechartSetPixmaps */


/************************************************************************/
/* PiechartSelect() processes button down event.			*/
/************************************************************************/
static void PiechartSelect(w, event)
PiechartWidget		w;
XButtonPressedEvent	*event;
{
  register int	i;
  double	fraction;
  double	x0 = (double) event ->x;
  double	y0 = (double) event ->y;
  double	x1, y1;
  double	angle, sinangle, cosangle;
  int		MinLoc;
  double	total, partial_sum = 0.0;
  double	last_angle, next_angle, current_angle;

#ifdef DEBUG
printf("PiechartSelect\n");
#endif

  MinLoc = -1;

  total = 0.0;
  for (i = 0; i < w->piechart.entries; i++)
    total += w->piechart.values[i];
  current_angle = (360.0/TWOPI) * 
                  atan2( (double)w->piechart.centerY - y0,
			 (double)x0 - w->piechart.centerX );
  if (current_angle < 0.0) current_angle += 360.0;

  for(i=0; i<w->piechart.entries; i++) {
    last_angle = 360.0 * (partial_sum / total);
    partial_sum += (double)w->piechart.values[i];    
    next_angle = 360.0 * (partial_sum / total);

    if ( (current_angle > last_angle) && (current_angle <= next_angle) ) {
       MinLoc = i;
    }        
  }
  w->piechart.value_select = w->piechart.values[MinLoc];
  w->piechart.pie_select = MinLoc;
  w->piechart.pie_select_pct = w->piechart.values[MinLoc]/total;
} /* PiechartSelect */

/************************************************************************/
/* PiechartNotify() processes button up event.				*/
/************************************************************************/
static void PiechartNotify(w, event)
PiechartWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("PiechartNotify\n");
#endif
  XtCallCallbacks(w, XtNselect, (caddr_t) w->piechart.value_select);

} /* PiechartNotify */



/************************************************************************/
/* PiechartGetSelectedPie() returns the currently-selected pie          */
/************************************************************************/

extern void PiechartGetSelectedPie(w, pie, pie_percentage)
PiechartWidget 	w;
int		*pie;
float		*pie_percentage;
{
	*pie = w->piechart.pie_select;
	*pie_percentage = w->piechart.pie_select_pct;
}
