/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/scatter3d/RCS/Scatter3D.c,v 1.14 1994/02/25 04:47:18 aydt Exp $
 */

#include <values.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include <X11/Shell.h>
#include <X11/Xaw/FormP.h>
#include <X11/Xaw/ScrollbarP.h>
#include <X11/Xaw/LabelP.h>
#include <X11/Xaw/CommandP.h>
#include <math.h>
#include "Scatter3DP.h"

#define Offset(field)		XtOffset(Scatter3DWidget, field)
#define Set_Arg(r, v)		XtSetArg(arglist[argi], r, v); argi++
#define fontheight(f)   (f->max_bounds.ascent+f->max_bounds.descent)

#define	ScaleX(x)	(double)(((x) - w->scatter3d.minXvalues) /	\
		(double)(w->scatter3d.maxXvalues - w->scatter3d.minXvalues))

#define	ScaleY(x)	(double)(((x) - w->scatter3d.minYvalues) /	\
		  (double)(w->scatter3d.maxYvalues - w->scatter3d.minYvalues))

#define	ScaleZ(x)	(double)(((x) - w->scatter3d.minZvalues) /	\
		 (double)(w->scatter3d.maxZvalues - w->scatter3d.minZvalues))

static char control_text[6][14] = {
	"X Axis Angle ",
	"Y Axis Angle ",
	"Z Axis Angle ",
	"Image Scale  ",
	"XY View Angle",
	"Z View Angle ",
};

/**********************/
/* private procedures */
/**********************/
static void		Initialize(), Destroy(), ClassInitialize();
static void		ResetCallback(), ApplyCallback(), OkCallback();
static void		CloseCallback(), JumpCallback(), Redisplay();
static void		CreateGCs(), DestroyGCs(), CreateColorGCs();
static void		DestroyColorGCs(), Resize(), Realize();
static void		DrawAxes(), DrawPoints(), DrawLine(), DrawPoint();
static Boolean		SetValues();
static void		ProjectPoint(), SphericalToCartesian();
static void		IdentityMatrix(), TransformMatrix(), RotateMatrix();
static void		Normalize(), CrossProduct(), ShearMatrix();
static double		DotProduct();
static void		ScaleMatrix(), TranslateMatrix();
static void		VectorMatrixMultiply(), MatrixMatrixMultiply();
static void		DumpMatrix();

/*********************/
/* public procedures */
/*********************/

extern void		Scatter3DSetValues(), Scatter3DSetColors();

static void		Scatter3DNotify();

static char scatter3d_translations[] = "	\
  <Btn1Up>:	Scatter3DNotify()	\
";

static XtActionsRec	scatter3d_actions[] = {
  {"Scatter3DNotify",	(XtActionProc) Scatter3DNotify	},
};

/***********************/
/* scatter3d resources */
/***********************/
static XtResource resources[] = {
  /* core resources */
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     Offset(core.width), XtRString, "200"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     Offset(core.height), XtRString, "200"},
  {XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(core.background_pixel), XtRString, "white"},
  /* scatter3d resources */
  {XtNfont, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
     Offset(scatter3d.font), XtRString, "fixed"},
  {XtNaxisXColor, XtCAxisXColor, XtRPixel, sizeof(Pixel),
     Offset(scatter3d.axisXcolor), XtRString, "red"},
  {XtNaxisYColor, XtCAxisYColor, XtRPixel, sizeof(Pixel),
     Offset(scatter3d.axisYcolor), XtRString, "green"},
  {XtNaxisZColor, XtCAxisZColor, XtRPixel, sizeof(Pixel),
     Offset(scatter3d.axisZcolor), XtRString, "blue"},
  {XtNaxisOColor, XtCAxisOColor, XtRPixel, sizeof(Pixel),
     Offset(scatter3d.axisOcolor), XtRString, "black"},
  {XtNapplyLabelColor, XtCApplyLabelColor, XtRPixel, sizeof(Pixel),
     Offset(scatter3d.apply_color), XtRString, "white"},
  {XtNcloseLabelColor, XtCCloseLabelColor, XtRPixel, sizeof(Pixel),
     Offset(scatter3d.close_color), XtRString, "white"},
  {XtNresetLabelColor, XtCResetLabelColor, XtRPixel, sizeof(Pixel),
     Offset(scatter3d.reset_color), XtRString, "white"},
  {XtNbackingStore, XtCBackingStore, XtRBackingStore, sizeof(int),
     Offset(scatter3d.backing_store), XtRString, "default"},
  {XtNautoScale, XtCAutoScale, XtRBoolean, sizeof(Boolean),
     Offset(scatter3d.autoscale), XtRString, "FALSE"},
  {XtNminXValue, XtCMinXValue, XtRInt, sizeof(int),
     Offset(scatter3d.minXvalues), XtRString, "0" },
  {XtNmaxXValue, XtCMaxXValue, XtRInt, sizeof(int),
     Offset(scatter3d.maxXvalues), XtRString, "100" },
  {XtNminYValue, XtCMinYValue, XtRInt, sizeof(int),
     Offset(scatter3d.minYvalues), XtRString, "0" },
  {XtNmaxYValue, XtCMaxYValue, XtRInt, sizeof(int),
     Offset(scatter3d.maxYvalues), XtRString, "100" },
  {XtNminZValue, XtCMinZValue, XtRInt, sizeof(int),
     Offset(scatter3d.minZvalues), XtRString, "0" },
  {XtNmaxZValue, XtCMaxZValue, XtRInt, sizeof(int),
     Offset(scatter3d.maxZvalues), XtRString, "100" },
  {XtNmarkerSize, XtCMarkerSize, XtRInt, sizeof(int),
     Offset(scatter3d.markersize), XtRString, "4" },
};

/**************************/
/* scatter3d class record */
/**************************/

Scatter3DClassRec scatter3dClassRec = {
  &widgetClassRec,		/* superclass			*/
  "Scatter3D",			/* class_name			*/
  sizeof(Scatter3DRec),		/* widget_size			*/
  ClassInitialize,		/* class_initialize		*/
  NULL,				/* class_part_initialize	*/
  FALSE,			/* class_inited			*/
  Initialize,			/* initialize			*/
  NULL,				/* initialize_hook		*/
  Realize,			/* realize			*/
  scatter3d_actions,		/* actions			*/
  XtNumber(scatter3d_actions),	/* num_actions			*/
  resources,			/* resources			*/
  XtNumber(resources),		/* resource_count		*/
  NULL,				/* xrm_class			*/
  TRUE,				/* compress_motion		*/
  TRUE,				/* compress_exposure		*/
  TRUE,				/* compress_enterleave		*/
  FALSE,			/* visible_interest		*/
  Destroy,			/* destroy			*/
  Resize,			/* resize			*/
  Redisplay,			/* expose			*/
  SetValues,			/* set_values			*/
  NULL,				/* set_values_hook		*/
  XtInheritSetValuesAlmost,	/* set_values_almost		*/
  NULL,				/* get_values_hook		*/
  NULL,				/* accept_focus			*/
  XtVersion,			/* version			*/
  NULL,				/* callback_private		*/
  scatter3d_translations,	/* tm_table			*/
  XtInheritQueryGeometry,	/* query_geometry          	*/
  XtInheritDisplayAccelerator,	/* display_accelerator    	*/
  NULL				/* extension               	*/
};

WidgetClass scatter3DWidgetClass = (WidgetClass) &scatter3dClassRec;

static void ClassInitialize()
{
  XtAddConverter(XtRString, XtRBackingStore, XmuCvtStringToBackingStore,
		 NULL, 0);
}

/**********************************/
/* scatter3d jump callback record */
/**********************************/
static XtCallbackRec jump_callback[] = {
  {JumpCallback, NULL},
  {NULL, NULL},
};

/*******************************************/
/* scatter3d reset command callback record */
/*******************************************/
static XtCallbackRec reset_callback[] = {
  {ResetCallback, NULL},
  {NULL, NULL},
};

/*******************************************/
/* scatter3d apply command callback record */
/*******************************************/
static XtCallbackRec apply_callback[] = {
  {ApplyCallback, NULL},
  {NULL, NULL},
};

/*******************************************/
/* scatter3d close command callback record */
/*******************************************/
static XtCallbackRec close_callback[] = {
  {CloseCallback, NULL},
  {NULL, NULL},
};

/************************************************************************/
/*                          PRIVATE PROCEDURES                          */
/************************************************************************/

/************************************************************************/
/* JumpCallback()							*/
/************************************************************************/
static void JumpCallback(w, client_data, call_data)
Scatter3DWidget	w;
caddr_t	client_data, call_data;
{
  int			which = (int) client_data;
  float			*percent = (float *) call_data;
  char			string[20];
  Arg			arglist[10];
  int			argi;
  Scatter3DWidget	back = (Scatter3DWidget) XtParent(w);
  Scatter3DWidget	back2 = (Scatter3DWidget) XtParent(back);
  Scatter3DWidget	parent = (Scatter3DWidget) XtParent(back2);

#ifdef DEBUG
printf("JumpCallback\n");
#endif

  parent ->scatter3d.configChanged = TRUE;

  switch (which) {
     case XBAR:
        parent->scatter3d.Xdegrees = (int) (360.0 * *percent);
        argi = 0;
	sprintf(string, "%3d", parent->scatter3d.Xdegrees);
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[XBARLABEL], arglist, argi);
#ifdef DEBUG
	printf("setting X degrees to %d\n",parent->scatter3d.Xdegrees);
#endif
	break;

     case YBAR:
	parent->scatter3d.Ydegrees = (int) (360.0 * *percent);
	argi = 0;
        sprintf(string, "%3d", parent->scatter3d.Ydegrees);
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[YBARLABEL], arglist, argi);
#ifdef DEBUG
	printf("setting Y degrees to %d\n",parent->scatter3d.Ydegrees);
#endif
	break;

     case ZBAR:
	parent->scatter3d.Zdegrees = (int) (360.0 * *percent);
	argi = 0;
        sprintf(string, "%3d", parent->scatter3d.Zdegrees);
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[ZBARLABEL], arglist, argi);
#ifdef DEBUG
	printf("setting Z degrees to %d\n",parent->scatter3d.Zdegrees);
#endif
	break;

     case ZOOM:
	parent->scatter3d.Zoomfraction = (double) *percent;
	argi = 0;
        sprintf(string, "%3.1f", parent->scatter3d.Zoomfraction);
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[ZOOMLABEL], arglist, argi);
#ifdef DEBUG
	printf("setting zoom to %f\n",parent->scatter3d.Zoomfraction);
#endif
	break;

     case THETA:
	parent->scatter3d.theta = (double) (360.0 * *percent);
        argi = 0;
	sprintf(string, "%3d", (int)(parent->scatter3d.theta));
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[THETALABEL], arglist, argi);
#ifdef DEBUG
	printf("setting theta to %8.2f\n",parent->scatter3d.theta);
#endif
	break;

     case PHI:
	parent->scatter3d.phi = (double) (360.0 * *percent);
        argi = 0;
	sprintf(string, "%3d", (int)(parent->scatter3d.phi));
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[PHILABEL], arglist, argi);
#ifdef DEBUG
	printf("setting phi to %8.2f\n",parent->scatter3d.phi);
#endif
	break;

     default:
	fprintf(stderr, "JumpCallback:  thumb calldata invalid\n");
	exit(1);
   }

} /* JumpCallback */

/************************************************************************/
/* ApplyCallback() applies the configuration changes to the display	*/
/************************************************************************/
static void ApplyCallback(w, client_data, call_data)
Widget	w;
caddr_t	client_data, call_data;
{
  Scatter3DWidget	new = (Scatter3DWidget) client_data;

#ifdef DEBUG
printf("ApplyCallback\n");
#endif

   new->scatter3d.configChanged = TRUE;

   Redisplay(new);

} /* ApplyCallback */

/************************************************************************/
/* CloseCallback() closes the popup display				*/
/************************************************************************/
static void CloseCallback(w, client_data, call_data)
Scatter3DWidget w;
caddr_t client_data, call_data;
{
  Widget	new = (Widget) client_data;

#ifdef DEBUG
printf("CloseCallback\n");
#endif

   XtPopdown(new);

} /* CloseCallback */

/************************************************************************/
/* ResetCallback() resets all the display options			*/
/************************************************************************/
static void ResetCallback(w, client_data, call_data)
Widget  w;
caddr_t client_data, call_data;
{
  Scatter3DWidget       parent = (Scatter3DWidget) client_data;
  char			string[20];
  int			argi;
  Arg			arglist[10];

#ifdef DEBUG
printf("ResetCallback\n");
#endif

  parent->scatter3d.configChanged = TRUE;
  parent->scatter3d.Xdegrees = 0;
  parent->scatter3d.Ydegrees = 0;
  parent->scatter3d.Zdegrees = 0;
  parent->scatter3d.Zoomfraction = DEFAULTSCALE;
  parent->scatter3d.theta = DEFAULTTHETA;
  parent->scatter3d.phi = DEFAULTPHI;

        argi = 0;
	sprintf(string, "%3d", parent->scatter3d.Xdegrees);
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[XBARLABEL], arglist, argi);
	XawScrollbarSetThumb(parent->scatter3d.controlW[XBARSCROLL],
                parent->scatter3d.Xdegrees, -1.0);

	argi = 0;
        sprintf(string, "%3d", parent->scatter3d.Ydegrees);
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[YBARLABEL], arglist, argi);
	XawScrollbarSetThumb(parent->scatter3d.controlW[YBARSCROLL],
                parent->scatter3d.Ydegrees, -1.0);


	argi = 0;
        sprintf(string, "%3d", parent->scatter3d.Zdegrees);
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[ZBARLABEL], arglist, argi);
        XawScrollbarSetThumb(parent->scatter3d.controlW[ZBARSCROLL], 
		parent->scatter3d.Zdegrees, -1.0);

	argi = 0;
        sprintf(string, "%3.1f", parent->scatter3d.Zoomfraction);
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[ZOOMLABEL], arglist, argi);
        XawScrollbarSetThumb(parent->scatter3d.controlW[ZOOMSCROLL], 
		parent->scatter3d.Zoomfraction, -1.0);

        argi = 0;
	sprintf(string, "%3d", (int)(parent->scatter3d.theta));
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[THETALABEL], arglist, argi);
	XawScrollbarSetThumb(parent->scatter3d.controlW[THETASCROLL],
                (parent->scatter3d.theta) / 360.0, -1.0);

	argi = 0;
        sprintf(string, "%3d", (int)(parent->scatter3d.phi));
        Set_Arg(XtNlabel, string);
        XtSetValues(parent->scatter3d.controlW[PHILABEL], arglist, argi);
	XawScrollbarSetThumb(parent->scatter3d.controlW[PHISCROLL],
                (parent->scatter3d.phi) / 360.0, -1.0);

} /* ResetCallback */


/************************************************************************/
/* Initialize()								*/
/************************************************************************/
static void Initialize(request, new)
Widget	request, new;
{
  char			string[20];
  int			i, j, argi, child = 0;
  int			tmp1, tmp2, tmp3, size;
  int			width = new->core.width;
  int			height = new->core.height;
  int			scroll_w, scroll_h;
  int			button_w, button_h;
  int			s = DefaultScreen(XtDisplay(new));
  int			hpad, vpad;
  Arg			arglist[20];
  Scatter3DWidget	w = (Scatter3DWidget) new;
  Display		*d = XtDisplay(new);
  Widget		w1, w2, w3;
  XtGCMask		GCmask;
  XGCValues		GCvalues;
  int			flag;
  float			x, y, z;

#ifdef DEBUG
printf("Initialize\n");
#endif

  if (w->core.width == 0)
    w->core.width = MIN_SIZE;
  if (w->core.height == 0)
    w->core.height = MIN_SIZE;

  GCmask = GCForeground | GCBackground;
  GCvalues.background = w->core.background_pixel;

  GCvalues.foreground = w->scatter3d.axisXcolor;
  w->scatter3d.axisGC[XCOLOR] = XtGetGC((Widget)w, GCmask, &GCvalues);
  GCvalues.foreground = w->scatter3d.axisYcolor;
  w->scatter3d.axisGC[YCOLOR] = XtGetGC((Widget)w, GCmask, &GCvalues);
  GCvalues.foreground = w->scatter3d.axisZcolor;
  w->scatter3d.axisGC[ZCOLOR] = XtGetGC((Widget)w, GCmask, &GCvalues);
  GCvalues.foreground = w->scatter3d.axisOcolor;
  w->scatter3d.axisGC[OTHERCOLOR] = XtGetGC((Widget)w, GCmask, &GCvalues);

  if (w->scatter3d.autoscale == TRUE) {
     w->scatter3d.minXvalues = MAXINT;
     w->scatter3d.minYvalues = MAXINT;
     w->scatter3d.minZvalues = MAXINT;
     w->scatter3d.maxXvalues = -MAXINT;
     w->scatter3d.maxYvalues = -MAXINT;
     w->scatter3d.maxZvalues = -MAXINT;
  }

  w->scatter3d.configChanged = TRUE;
  
  width = w->core.width;
  height = w->core.height;

  if (width > height)
     size = height;
  else
     size = width;

  w->scatter3d.PRP.val[0] = 0.5 * DEFAULTRADIUS;
  w->scatter3d.PRP.val[1] = 0.5 * DEFAULTRADIUS;
  w->scatter3d.PRP.val[2] = DEFAULTRADIUS;
  w->scatter3d.PRP.val[3] = 1.0;

  w->scatter3d.VPN.val[0] = 0.0;
  w->scatter3d.VPN.val[1] = 0.0;
  w->scatter3d.VPN.val[2] = 1.0;
  w->scatter3d.VPN.val[3] = 1.0;

  w->scatter3d.VUP.val[0] = 0.0;
  w->scatter3d.VUP.val[1] = 1.0;
  w->scatter3d.VUP.val[2] = 0.0;
  w->scatter3d.VUP.val[3] = 1.0;

  w->scatter3d.frontplane = 1.0;
  w->scatter3d.backplane = -1.0;

  w->scatter3d.umin = 0.0;
  w->scatter3d.umax = 1.0;
  w->scatter3d.vmin = 0.0;
  w->scatter3d.vmax = 1.0;

#ifdef DEBUG
printf("VPN: (%f,%f,%f)\n",w->scatter3d.VPN.val[0],
			   w->scatter3d.VPN.val[1],
			   w->scatter3d.VPN.val[2]);

printf("VUP: (%f,%f,%f)\n",w->scatter3d.VUP.val[0],
                           w->scatter3d.VUP.val[1],
                           w->scatter3d.VUP.val[2]);

printf("PRP: (%f,%f,%f)\n",w->scatter3d.PRP.val[0],
                           w->scatter3d.PRP.val[1],
                           w->scatter3d.PRP.val[2]);

printf("front and back clipping plans: (%f,%f)\n",
			   w->scatter3d.frontplane,
			   w->scatter3d.backplane);

printf ("(vmin, vmax): (%f,%f)\n", w->scatter3d.vmin, w->scatter3d.vmax);
printf ("(umin, umax): (%f,%f)\n", w->scatter3d.umin, w->scatter3d.umax);
#endif

  w->scatter3d.setcnt = 0;
  w->scatter3d.colorcnt = 0;

  for (i = 0; i < MAXSETS; i++) {
     w->scatter3d.ptcnt[i] = 0;

     for (j = 0; j < MAXPOINTS; j++) {
        w->scatter3d.Xvalues[i][j] = -MAXINT;
        w->scatter3d.Yvalues[i][j] = -MAXINT;
        w->scatter3d.Zvalues[i][j] = -MAXINT;
     }
  }

  w->scatter3d.Xdegrees = 0;
  w->scatter3d.Ydegrees = 0;
  w->scatter3d.Zdegrees = 0;
  w->scatter3d.Zoomfraction = DEFAULTSCALE;
  w->scatter3d.theta = DEFAULTTHETA;
  w->scatter3d.phi = DEFAULTPHI;

  vpad = 0.05 * (float) height;
  hpad = 0.05 * (float) width;
  scroll_h = (int) (((float) height / 2.0 - 4.0 * vpad) / 3.0);
  scroll_w = (int) ((float) width - 2.0 * hpad);

  tmp1 = XTextWidth(w->scatter3d.font, "Reset", strlen("Reset")) + hpad;
  tmp2 = XTextWidth(w->scatter3d.font, "Apply", strlen("Apply")) + hpad;
  
  if (tmp1 > tmp2)
     button_w = tmp1;
  else
     button_w = tmp2;

  tmp3 = XTextWidth(w->scatter3d.font, "Close", strlen("Close")) + hpad;

  if (tmp3 > button_w)
     button_w = tmp3;

  button_h = fontheight(w->scatter3d.font) + vpad;

#ifdef DEBUG
printf("w=%d, h=%d, ", width, height);
printf("scroll_w=%d, scroll_h=%d, ", scroll_w, scroll_h);
printf("button_w=%d, button_h=%d\n", button_w, button_h);
printf("vpad=%d, hpad=%d\n",vpad,hpad);
#endif

  w1 = NULL;

  /* create description window */
  argi = 0;
  w->scatter3d.popupW = XtCreatePopupShell("Scatter3D Configuration",
				topLevelShellWidgetClass, w, arglist, argi);
  close_callback[0].closure = (caddr_t) w->scatter3d.popupW;
  apply_callback[0].closure = (caddr_t) w;
  reset_callback[0].closure = (caddr_t) w;

  argi = 0;
  w->scatter3d.popupFormW = XtCreateManagedWidget("PopupForm", formWidgetClass,
					    w->scatter3d.popupW, arglist, argi);

  for (i=0; i<NUM_CONTROL; i++) {
    argi = 0;					/* create scroll type */
    Set_Arg(XtNfromVert, w1);
    Set_Arg(XtNvertDistance, vpad);
    Set_Arg(XtNlabel, control_text[i]);
    Set_Arg(XtNborderWidth, 0);
    Set_Arg(XtNresizable, TRUE);
    w2 = w1;
    w1 = XtCreateWidget("", labelWidgetClass, w->scatter3d.popupFormW,
			arglist, argi);
    w->scatter3d.controlW[child] = w1;
    child++;

    argi = 0;					/* create scroll minimum */
    Set_Arg(XtNfromVert, w2);
    Set_Arg(XtNvertDistance, vpad);
    Set_Arg(XtNfromHoriz, w1);
    Set_Arg(XtNhorizDistance, hpad);
    Set_Arg(XtNresizable, TRUE);
    Set_Arg(XtNborderWidth, 0);

    string[0] = NULL;
  
    if (i != 3)
       sprintf(string, "%3d", 0);

    if (i == 3)
       sprintf(string, "%3.1f", 0.0);

    Set_Arg(XtNwidth, XTextWidth(w->scatter3d.font,string,
			strlen(string))+hpad);
    Set_Arg(XtNlabel, string);

    w->scatter3d.controlW[child] =
	XtCreateWidget("", labelWidgetClass, w->scatter3d.popupFormW,
		arglist, argi);
    child++;

    argi = 0;					/* create scroll bar	 */
    jump_callback[0].closure = (caddr_t) i;
    Set_Arg(XtNwidth, scroll_w);
    Set_Arg(XtNheight, scroll_h);
    Set_Arg(XtNfromVert, w2);
    Set_Arg(XtNvertDistance, vpad);
    Set_Arg(XtNfromHoriz, w->scatter3d.controlW[child-1]);
    Set_Arg(XtNhorizDistance, hpad);
    Set_Arg(XtNresizable, TRUE);
    Set_Arg(XtNorientation, XtorientHorizontal);
    Set_Arg(XtNjumpProc, jump_callback);
    w->scatter3d.controlW[child] = XtCreateWidget("", scrollbarWidgetClass,
				w->scatter3d.popupFormW, arglist, argi);
    child++;

    if (i == 3)
       XawScrollbarSetThumb(w->scatter3d.controlW[child-1],
			     DEFAULTSCALE, -1.0);

    if (i == 4)
	XawScrollbarSetThumb(w->scatter3d.controlW[child-1],
			     (DEFAULTTHETA/360.0), -1.0);

    if (i == 5)
	XawScrollbarSetThumb(w->scatter3d.controlW[child-1],
			     (DEFAULTPHI/360.0), -1.0);

    argi = 0;					/* create scroll maximum */
    Set_Arg(XtNfromVert, w2);
    Set_Arg(XtNvertDistance, vpad);
    Set_Arg(XtNfromHoriz, w->scatter3d.controlW[child-1]);
    Set_Arg(XtNhorizDistance, hpad);
    Set_Arg(XtNresizable, TRUE);
    Set_Arg(XtNborderWidth, 0);
 
    string[0] = NULL;

    if (i != 3)
       sprintf(string, "%3d", 360);
    else
       sprintf(string, "%3.1f", 1.0); 

    Set_Arg(XtNlabel, string);
    Set_Arg(XtNwidth, XTextWidth(w->scatter3d.font, string,
			strlen(string))+hpad);

    w->scatter3d.controlW[child] =
	XtCreateWidget("", labelWidgetClass, w->scatter3d.popupFormW,
			arglist, argi);
    child++;

    argi = 0;					/* create scroll current */
    Set_Arg(XtNfromVert, w1);
    Set_Arg(XtNvertDistance, vpad);
    Set_Arg(XtNfromHoriz, NULL);
    Set_Arg(XtNhorizDistance, 2*hpad);
    Set_Arg(XtNresizable, TRUE);
    Set_Arg(XtNborderWidth, 0);

    string[0] = NULL;

    if (i < 3)
       sprintf(string, "%3d", 0);

    if (i == 3)
       sprintf(string, "%3.1f", DEFAULTSCALE);

    if (i == 4)
       sprintf(string, "%3d", (int)DEFAULTTHETA);

    if (i == 5)
       sprintf(string, "%3d", (int)DEFAULTPHI);

    Set_Arg(XtNlabel, string);
    Set_Arg(XtNwidth, XTextWidth(w->scatter3d.font, string,
		strlen(string))+hpad);

    w1 = XtCreateWidget("", labelWidgetClass, w->scatter3d.popupFormW,
		arglist, argi);

    w->scatter3d.controlW[child] = w1;
    child++;
  }  

#ifdef DEBUG
printf("creating Reset button \n");
#endif

  argi = 0;
  reset_callback[0].closure = (caddr_t) w;
  Set_Arg(XtNwidth, button_w);
  Set_Arg(XtNheight, button_h);
  Set_Arg(XtNlabel, "Reset");
  Set_Arg(XtNbackground, w->scatter3d.reset_color);
  Set_Arg(XtNfromVert, w1);
  Set_Arg(XtNvertDistance, 3 * vpad);
  Set_Arg(XtNresizable, TRUE);
  Set_Arg(XtNcallback, reset_callback);
  w2 = XtCreateWidget("", commandWidgetClass,
				       w->scatter3d.popupFormW, arglist, argi);
  w->scatter3d.controlW[child] = w2;
  child++;

#ifdef DEBUG
printf("creating Apply button \n");
#endif

  argi = 0;
  apply_callback[0].closure = (caddr_t) w;
  Set_Arg(XtNwidth, button_w);
  Set_Arg(XtNheight, button_h);
  Set_Arg(XtNlabel, "Apply");
  Set_Arg(XtNbackground, w->scatter3d.apply_color);
  Set_Arg(XtNfromHoriz, w2);
  Set_Arg(XtNhorizDistance, 3 * hpad);
  Set_Arg(XtNfromVert, w1); 
  Set_Arg(XtNvertDistance, 3 * vpad);
  Set_Arg(XtNresizable, TRUE);
  Set_Arg(XtNcallback, apply_callback);
  w3 = XtCreateWidget("", commandWidgetClass,
				       w->scatter3d.popupFormW, arglist, argi);
  w->scatter3d.controlW[child] = w3;
  child++;

#ifdef DEBUG
printf("creating Close button \n");
#endif

  argi = 0;
  apply_callback[0].closure = (caddr_t) w;
  Set_Arg(XtNwidth, button_w);
  Set_Arg(XtNheight, button_h);
  Set_Arg(XtNbackground, w->scatter3d.close_color);
  Set_Arg(XtNlabel, "Close");
  Set_Arg(XtNfromHoriz, w3);
  Set_Arg(XtNhorizDistance, 3 * hpad);
  Set_Arg(XtNfromVert, w1);
  Set_Arg(XtNvertDistance, 3 * vpad);
  Set_Arg(XtNresizable, TRUE);
  Set_Arg(XtNcallback, close_callback);
  w->scatter3d.controlW[child] = XtCreateWidget("", commandWidgetClass,
                                       w->scatter3d.popupFormW, arglist, argi);
  child++;

#ifdef DEBUG
  printf("managing children in Initialize\n");
#endif

  XtManageChildren(w->scatter3d.controlW, child);

#ifdef DEBUG
   printf ("exiting Initialize\n");
#endif

} /* Initialize */

/**************************************************************************/
/* Redisplay()								  */
/**************************************************************************/

static void Redisplay(gw, event, region)
Widget	gw;
XEvent	*event;				/* unused */
Region	region;				/* unused */
{
  Scatter3DWidget	w = (Scatter3DWidget) gw;
  double		angle;
  double		scale;
  double		x, y, z;
  matrix3D		tmpMatrix;
  
#ifdef DEBUG
printf("Redisplay\n");
#endif

  if (!XtIsRealized((Widget) w))
     return;

  if (w->scatter3d.configChanged == TRUE) {
     w->scatter3d.configChanged = FALSE;

     IdentityMatrix(&(w->scatter3d.transMatrix));

     scale = w->scatter3d.Zoomfraction + DEFAULTSCALE;
     TranslateMatrix(w, -0.5, -0.5, -0.5);
     ScaleMatrix(w, scale, scale, scale);
     TranslateMatrix(w, 0.5, 0.5, 0.5);

     angle = (double)(w->scatter3d.Xdegrees) / 360.0 * TWOPI;
     TranslateMatrix(w, -0.5, -0.5, -0.5);
     RotateMatrix(w, angle, XAXIS);
     TranslateMatrix(w, 0.5, 0.5, 0.5);

     TranslateMatrix(w, -0.5, -0.5, -0.5);
     angle = (double)(w->scatter3d.Ydegrees) / 360.0 * TWOPI;
     RotateMatrix(w, angle, YAXIS);
     TranslateMatrix(w, 0.5, 0.5, 0.5);

     angle = (double)(w->scatter3d.Zdegrees) / 360.0 * TWOPI;
     TranslateMatrix(w, -0.5, -0.5, -0.5);
     RotateMatrix(w, angle, ZAXIS);
     TranslateMatrix(w, 0.5, 0.5, 0.5);


     SphericalToCartesian(DEFAULTRADIUS/4.0, w->scatter3d.theta,
		          w->scatter3d.phi, &x, &y, &z);

     w->scatter3d.VRP.val[0] = -x;
     w->scatter3d.VRP.val[1] = -y;
     w->scatter3d.VRP.val[2] = -z;
     w->scatter3d.VRP.val[3] = 1.0;

#ifdef DEBUG
     printf("VRP: (%f,%f,%f)\n",w->scatter3d.VRP.val[0],
                           w->scatter3d.VRP.val[1],
                           w->scatter3d.VRP.val[2]);
#endif

     IdentityMatrix(&tmpMatrix); /* convert to left-handed coordinates	*/
     tmpMatrix.val[2][2] = -1.0;
     MatrixMatrixMultiply(w->scatter3d.transMatrix, tmpMatrix,
		          &(w->scatter3d.transMatrix));

     TransformMatrix(w);
  }

  XFillRectangle( XtDisplay(w), w->scatter3d.drawPixmap,
		  w->scatter3d.pixmapGC, 0, 0, w->core.width, w->core.height);
  DrawAxes(w);
  DrawPoints(w);
  XCopyArea( XtDisplay(w), w->scatter3d.drawPixmap, XtWindow(w),
	     w->scatter3d.pixmapGC, 
	     0, 0, w->core.width, w->core.height, 0, 0 );
} /* Redisplay */

static void DrawPoints(w)
Scatter3DWidget w;
{
	int	i, j;
	int	size;
	int	numsets;
	point3D	pt;

#ifdef DEBUG
	printf("DrawPoints\n");
#endif

  	if (w->core.width > w->core.height)
     	   size = w->core.height - PADDING;
  	else
     	   size = w->core.width - 2 * PADDING;

	numsets = w->scatter3d.setcnt;

	for (i = 0; i < numsets; i++)
	   for (j = 0; j < w->scatter3d.ptcnt[i]; j++) {
	      pt.val[0] = ScaleX(w->scatter3d.Xvalues[i][j]);
	      pt.val[1] = ScaleY(w->scatter3d.Yvalues[i][j]);
	      pt.val[2] = ScaleZ(w->scatter3d.Zvalues[i][j]);
	      pt.val[3] = 1.0;

	      DrawPoint(w, size, pt, i);
	   }
}

static void DrawAxes(w)
Scatter3DWidget w;
{
      	point3D pt3dStart;
	point3D pt3dEnd;
	int	i, size;

#ifdef DEBUG
	printf("DrawAxes\n");
#endif

	if (w->core.height > w->core.width)
	   size = w->core.width - PADDING;
	else
	   size = w->core.height - 2 * PADDING;

	/* Note:  We draw the X, Y, and Z axes from the origin in color	*/
	/*	  to indicate the origin when the wire frame is rotated	*/

	for (i = 0; i < 4; i++)			/* point (0, 0, 0)	*/
           pt3dStart.val[i] = 0.0;

	pt3dStart.val[3] = 1.0;

	for (i = 0; i < 4; i++)
           pt3dEnd.val[i] = 0.0;

	pt3dEnd.val[1] = 1.0;			/* point 0, 1, 0)	*/
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, YCOLOR);

	for (i = 0; i < 4; i++)			/* point (1, 0, 0)	*/
           pt3dEnd.val[i] = 0.0;

        pt3dEnd.val[0] = 1.0;
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, XCOLOR);

 	for (i = 0; i < 4; i++)			/* point (0, 0, 1)	*/
           pt3dEnd.val[i] = 0.0;

        pt3dEnd.val[2] = 1.0;
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, ZCOLOR);

	for (i = 0; i < 4; i++)			/* point (1, 0, 0)	*/
           pt3dStart.val[i] = 0.0;

	pt3dStart.val[0] = 1.0;
	pt3dStart.val[3] = 1.0;

	for (i = 0; i < 4; i++)			/* point (1, 1, 0)	*/
           pt3dEnd.val[i] = 0.0;

	pt3dEnd.val[0] = 1.0;
	pt3dEnd.val[1] = 1.0;
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, OTHERCOLOR);

 	for (i = 0; i < 4; i++)			/* point (1, 0, 1)	*/
           pt3dEnd.val[i] = 0.0;

	pt3dEnd.val[0] = 1.0;
        pt3dEnd.val[2] = 1.0;
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, OTHERCOLOR);

	pt3dStart.val[0] = 1.0;			/* point (1, 1, 0)	*/
	pt3dStart.val[1] = 1.0;
	pt3dStart.val[2] = 0.0;
	pt3dStart.val[3] = 1.0;

	pt3dEnd.val[0] = 1.0;
	pt3dEnd.val[1] = 1.0;
	pt3dEnd.val[2] = 1.0;
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, OTHERCOLOR);

	for (i = 0; i < 4; i++)			/* point (0, 1, 0)	*/
           pt3dStart.val[i] = 0.0;

	pt3dStart.val[1] = 1.0;
	pt3dStart.val[3] = 1.0;

	for (i = 0; i < 4; i++)
           pt3dEnd.val[i] = 0.0;

	pt3dEnd.val[0] = 1.0;
	pt3dEnd.val[1] = 1.0;
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, OTHERCOLOR);

	for (i = 0; i < 4; i++)			/* point (0, 0, 1)	*/
           pt3dStart.val[i] = 0.0;

	pt3dStart.val[2] = 1.0;
	pt3dStart.val[3] = 1.0;

	pt3dEnd.val[0] = 0.0;
	pt3dEnd.val[1] = 1.0;
	pt3dEnd.val[2] = 1.0;
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, OTHERCOLOR);

	pt3dStart.val[0] = 0.0;			/* point (0, 1, 1)	*/
	pt3dStart.val[1] = 1.0;
	pt3dStart.val[2] = 1.0;
	pt3dStart.val[3] = 1.0;

	pt3dEnd.val[0] = 1.0;			/* point (1, 1, 1)	*/
	pt3dEnd.val[1] = 1.0;
	pt3dEnd.val[2] = 1.0;
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, OTHERCOLOR);

	for (i = 0; i < 4; i++)			/* point (0, 0, 1)	*/
           pt3dStart.val[i] = 0.0;

	pt3dStart.val[2] = 1.0;
	pt3dStart.val[3] = 1.0;

	pt3dEnd.val[0] = 1.0;			/* point (1, 0, 1)	*/
	pt3dEnd.val[1] = 0.0;
	pt3dEnd.val[2] = 1.0;
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, OTHERCOLOR);

	pt3dStart.val[0] = 0.0;			/* point (0, 1, 1)	*/
	pt3dStart.val[1] = 1.0;
	pt3dStart.val[2] = 1.0;
	pt3dStart.val[3] = 1.0;

	for (i = 0; i < 4; i++)			/* point (0, 1, 0)	*/
           pt3dEnd.val[i] = 0.0;

	pt3dEnd.val[1] = 1.0;
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, OTHERCOLOR);

	pt3dStart.val[0] = 1.0;			/* point (1, 1, 1)	*/
	pt3dStart.val[1] = 1.0;
	pt3dStart.val[2] = 1.0;
	pt3dStart.val[3] = 1.0;

	pt3dEnd.val[0] = 1.0;			/* point (1, 0, 1)	*/
	pt3dEnd.val[1] = 0.0;
	pt3dEnd.val[2] = 1.0;
	pt3dEnd.val[3] = 1.0;

	DrawLine(w, size, pt3dStart, pt3dEnd, OTHERCOLOR);
}

/************************************************************************/
/* DrawLine() 	draw a line projected onto the viewing plane		*/
/************************************************************************/

static void DrawLine(w, size, start, end, colorIndex)
Scatter3DWidget w;
int		size;
point3D		start, end;
int		colorIndex;
{
	point2D	pt2dStart;
	point2D	pt2dEnd;

#ifdef DEBUG
	printf("DrawLine\n");
	printf("DrawLine from (%f,%f,%f) to (%f,%f,%f)\n",
		start.val[0], start.val[1], start.val[2],
		end.val[0], end.val[1], end.val[2]);
#endif

	ProjectPoint(w, start, size, &pt2dStart);
	ProjectPoint(w, end, size, &pt2dEnd);

      	XDrawLine(XtDisplay(w),
		  w->scatter3d.drawPixmap, w->scatter3d.axisGC[colorIndex],
		(int)pt2dStart.val[0] + PADDING,
		size + PADDING - (int)pt2dStart.val[1],
		(int)pt2dEnd.val[0] + PADDING,
		size + PADDING - (int)pt2dEnd.val[1]);
/*
      	XDrawLine(XtDisplay(w), XtWindow(w), w->scatter3d.axisGC[colorIndex],
		(int)pt2dStart.val[0] + PADDING,
		size + PADDING - (int)pt2dStart.val[1],
		(int)pt2dEnd.val[0] + PADDING,
		size + PADDING - (int)pt2dEnd.val[1]);
*/
}

/************************************************************************/
/* DrawPoint() 	draw a point on the viewing plane			*/
/************************************************************************/

static void DrawPoint(w, size, pt, set)
Scatter3DWidget w;
int		size;
point3D		pt;
int		set;
{
	point2D	pt2dLoc;
	int	x, y;

#ifdef DEBUG
	printf("DrawPoint\n");
	printf("DrawPoint (%f,%f,%f)\n", pt.val[0], pt.val[1], pt.val[2]);
#endif

	ProjectPoint(w, pt, size, &pt2dLoc);

/*
      	XDrawPoint(XtDisplay(w), XtWindow(w), w->scatter3d.pointGCs[set],
		(int)pt2dLoc.val[0] + PADDING,
		size + PADDING - (int)pt2dLoc.val[1]);
*/
	x = pt2dLoc.val[0] + PADDING - w->scatter3d.markersize/2;
	y = size + PADDING - pt2dLoc.val[1] - w->scatter3d.markersize/2;

 	XFillArc(XtDisplay(w), w->scatter3d.drawPixmap,
           w->scatter3d.pointGCs[set],
           x, y, w->scatter3d.markersize, w->scatter3d.markersize,
           0, 360*64);
}

/************************************************************************/
/* Destroy()                                                             */
/************************************************************************/
static void Destroy(gw)
Widget gw;
{
  int		i;
  Scatter3DWidget	w = (Scatter3DWidget) gw;

#ifdef DEBUG
printf("Destroy\n");
#endif

  for (i = 0; i < 4; i++)
     XtDestroyGC(w->scatter3d.axisGC[i]);

  DestroyColorGCs(w);
  XtRemoveAllCallbacks(w, XtNselect);

} /* Destroy */

static void CreateColorGCs(w, cnt)
Scatter3DWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreateColorGCs\n");
#endif

  gcValues.fill_style = FillSolid;
  gcMask = GCForeground | GCFillStyle;

  for (i=0; i< cnt; i++) {
    gcValues.foreground = w->scatter3d.point_color[i];
    w->scatter3d.pointGCs[i] = XtGetGC(w, gcMask, &gcValues);
  }

} /* CreateColorGCs */

static void DestroyColorGCs(w)
Scatter3DWidget	w;
{
  register	i;
  
#ifdef DEBUG
printf("DestroyColorGCs\n");
#endif

    for (i=0; i<w->scatter3d.colorcnt; i++)
      XtDestroyGC(w->scatter3d.pointGCs[i]);

} /* DestroyColorGCs */

/************************************************************************/
/* SetValues()								*/
/************************************************************************/
/* ARGSUSED */
static Boolean SetValues(gcurrent, grequest, gnew)
Widget	gcurrent, grequest, gnew;
{
  int			i;
  register Scatter3DWidget	current = (Scatter3DWidget) gcurrent;
  register Scatter3DWidget	new = (Scatter3DWidget) gnew;
  Boolean		redraw = FALSE;
  XtGCMask		GCmask;
  XGCValues		GCvalues;
  Boolean               realized = XtIsRealized(gcurrent);

#ifdef DEBUG
printf("SetValues\n");
#endif
  /* change of width or height */
  if ((new->core.height != current->core.height)
      || (new->core.width != current->core.width)) {
    Resize(new);
    redraw = TRUE;
  }

  /* change axes colors */
  if ((new->scatter3d.axisXcolor != current->scatter3d.axisXcolor)
      || (new->core.background_pixel != current->core.background_pixel)) {
    GCmask = GCForeground | GCBackground;
    GCvalues.foreground = new->scatter3d.axisXcolor;
    GCvalues.background = new->core.background_pixel;
    XtDestroyGC(current->scatter3d.axisGC[XCOLOR]);
    new->scatter3d.axisGC[XCOLOR] = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }

  if ((new->scatter3d.axisYcolor != current->scatter3d.axisYcolor)
      || (new->core.background_pixel != current->core.background_pixel)) {
    GCmask = GCForeground | GCBackground;
    GCvalues.foreground = new->scatter3d.axisYcolor;
    GCvalues.background = new->core.background_pixel;
    XtDestroyGC(current->scatter3d.axisGC[YCOLOR]);
    new->scatter3d.axisGC[YCOLOR] = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }

  if ((new->scatter3d.axisZcolor != current->scatter3d.axisZcolor)
      || (new->core.background_pixel != current->core.background_pixel)) {
    GCmask = GCForeground | GCBackground;
    GCvalues.foreground = new->scatter3d.axisZcolor;
    GCvalues.background = new->core.background_pixel;
    XtDestroyGC(current->scatter3d.axisGC[ZCOLOR]);
    new->scatter3d.axisGC[ZCOLOR] = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }

  if ((new->scatter3d.axisOcolor != current->scatter3d.axisOcolor)
      || (new->core.background_pixel != current->core.background_pixel)) {
    GCmask = GCForeground | GCBackground;
    GCvalues.foreground = new->scatter3d.axisOcolor;
    GCvalues.background = new->core.background_pixel;
    XtDestroyGC(current->scatter3d.axisGC[OTHERCOLOR]);
    new->scatter3d.axisGC[OTHERCOLOR] = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }

  /* change colors */
  if (new->scatter3d.point_color != current->scatter3d.point_color) {
    DestroyColorGCs(current);
    CreateColorGCs(new, new->scatter3d.colorcnt);
    redraw = TRUE;
  }

  if (redraw && realized) {
    Redisplay(new);
  }

  return(FALSE);
  
} /* SetValues */

static void Realize(gw, GCmask, attrs)
Widget			gw;
XtGCMask		*GCmask;
XSetWindowAttributes	*attrs;
{
  Scatter3DWidget	w = (Scatter3DWidget) gw;
  XtGCMask		pixmapGCmask;
  XGCValues		pixmapGCvalues;
  
#ifdef DEBUG
printf("Realize\n");
#endif

  *GCmask |= CWBitGravity;
  attrs->bit_gravity = ForgetGravity;
  switch(w->scatter3d.backing_store) {
  case Always:
  case NotUseful:
  case WhenMapped:
    *GCmask |=CWBackingStore;
    attrs->backing_store = w->scatter3d.backing_store;
    break;
  }
  XtCreateWindow(gw, InputOutput,(Visual *)CopyFromParent,
		 *GCmask, attrs);

  pixmapGCmask = GCBackground;
  pixmapGCvalues.background = w->core.background_pixel;
  w->scatter3d.pixmapGC = XtGetGC((Widget)w, pixmapGCmask, &pixmapGCvalues);
  w->scatter3d.drawPixmap = XCreatePixmap( XtDisplay(gw), XtWindow(gw),
				           w->core.width, w->core.height, 
				           DefaultDepth( XtDisplay(gw),
				      XScreenNumberOfScreen( XtScreen(gw) )));
  Resize(gw);
} /* Realize */

static void Resize(gw) 
Widget	gw;
{
  Scatter3DWidget	w = (Scatter3DWidget) gw;
  int		width;
  int		height;

#ifdef DEBUG
printf("Resize\n");
#endif

  if (XtIsRealized(gw)) {
     XFreePixmap( XtDisplay(gw), w->scatter3d.drawPixmap);
     w->scatter3d.drawPixmap = XCreatePixmap( XtDisplay(gw), XtWindow(gw),
				           w->core.width, w->core.height, 
				           DefaultDepth( XtDisplay(gw),
				            DefaultScreen( XtDisplay(gw) )));
  }

} /* Resize */

extern void Scatter3DSetColors(w, numcolors, colors)
Scatter3DWidget	w;
int		numcolors;
Pixel		*colors;
{
int	i;

#ifdef DEBUG
printf("Scatter3DSetColors\n");
#endif

  DestroyColorGCs(w);

  w->scatter3d.colorcnt = numcolors;
  w->scatter3d.point_color = colors;

  CreateColorGCs(w, numcolors);
  Redisplay(w);

} /* Scatter3DSetColors */

/************************************************************************/
/* Scatter3DSetValues()							*/
/************************************************************************/
extern void Scatter3DSetValues(w, ptcnt, dataset, Xvalues, Yvalues, Zvalues)
Scatter3DWidget	w;
int		ptcnt;
int		*Xvalues;
int		*Yvalues;
int		*Zvalues;
int		dataset;
{
  int	i;
  double mag;
  int	minX, minY, minZ;
  int	maxX, maxY, maxZ;

#ifdef DEBUG
printf("Scatter3DSetValues\n");
#endif

  if ((ptcnt < 0) || (ptcnt > MAXPOINTS)) {
     fprintf(stderr, "Scatter3DSetValues:  Incorrect number of values");
     fprintf(stderr, " (points, dataset): (%d,%d)\n",ptcnt,dataset);
     w->scatter3d.ptcnt[dataset] = MAXPOINTS;
  } else if ((dataset < 0) || (dataset > MAXSETS)) {
     fprintf(stderr, "Scatter3DSetValues:  Incorrect number of values");
     fprintf(stderr, " (points, dataset): (%d,%d)\n", ptcnt, dataset);
     w->scatter3d.setcnt = MAXSETS;
  } else
    w->scatter3d.ptcnt[dataset] = ptcnt;

  if (dataset >= w->scatter3d.setcnt)
     w->scatter3d.setcnt = dataset + 1;

  if (w->scatter3d.autoscale == TRUE) {
     minX = w->scatter3d.minXvalues;
     minY = w->scatter3d.minYvalues;
     minZ = w->scatter3d.minZvalues;
     maxX = w->scatter3d.maxXvalues;
     maxY = w->scatter3d.maxYvalues;
     maxZ = w->scatter3d.maxZvalues;
  }

  for (i=0; i < ptcnt; i++) {
      w->scatter3d.Xvalues[dataset][i] = Xvalues[i];
      w->scatter3d.Yvalues[dataset][i] = Yvalues[i];
      w->scatter3d.Zvalues[dataset][i] = Zvalues[i];

      if (w->scatter3d.autoscale == TRUE) {
         if (Xvalues[i] < minX)
	    minX = Xvalues[i];

         if (Xvalues[i] > maxX)
	    maxX = Xvalues[i];

         if (Yvalues[i] < minY)
	    minY = Yvalues[i];

         if (Yvalues[i] > maxY)
	    maxY = Yvalues[i];

         if (Zvalues[i] < minZ)
	    minZ = Zvalues[i];

         if (Zvalues[i] > maxZ)
	    maxZ = Zvalues[i];
      }
  }

  if (w->scatter3d.autoscale == TRUE) {
     w->scatter3d.minXvalues = minX;
     w->scatter3d.minYvalues = minY;
     w->scatter3d.minZvalues = minZ;
     w->scatter3d.maxXvalues = maxX;
     w->scatter3d.maxYvalues = maxY;
     w->scatter3d.maxZvalues = maxZ;
  }
  Redisplay(w);
    
} /* Scatter3DSetValues */

/************************************************************************/
/* Scatter3DNotify() processes button up event.				*/
/************************************************************************/
static void Scatter3DNotify(w, event)
Scatter3DWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("Scatter3DNotify\n");
#endif

  XtPopup(w->scatter3d.popupW, XtGrabNone);

} /* Scatter3DNotify */

/************************************************************************/
/* IdentityMatrix(m) creates an identity transformation matrix		*/
/************************************************************************/
static void IdentityMatrix(m)
matrix3D *m;
{
	int	i, j;

#ifdef DEBUG
printf("IdentityMatrix\n");
#endif

  for (i = 0; i < 4; i++) {
     for (j = 0; j < 4; j++)
        m-> val[i][j] = 0.0;

     m -> val[i][i] = 1.0;
  }
}

/************************************************************************/
/* TransformMatrix(w) creates a transformation matrix			*/
/************************************************************************/
static void TransformMatrix(w)
Scatter3DWidget	w;
{
	int		i;
	double		tmp, Zmin, scaleX, scaleY, scaleZ;
	point3D		VRPPrime, DOP, Rx, Ry, Rz, CrossforRx;
	matrix3D	tmpMatrix;

#ifdef DEBUG
printf("TransformMatrix\n");
#endif

/* NOTE:  The creation of a transformation matrix follows Foley, van Dam,
	  Feiner, and Hughes (chapter 6, section 6.5.2).  Briefly, the
	  algorithm is:

		1.  Translate the View Reference Point (VRP) to the origin
		2.  Rotate the View Reference Coordinate (VRC) system such
		    that the n axis (VPN) becomes the z axis, the u axis
		    becomes the x axis, and the v axis becomes the y axis.
		3.  Translate such that the center of projection (COP), given
		    by the Perspective Reference Point (PRP), is at the origin.
	   	4.  Shear such that the center line of the view volume becomes
		    the z axis.
		5.  Scale such that the view volume becomes the canonical
		    perspective view volume, a truncated right pyramid.
		6.  Prepare for clipping in homogeneous coordinates.
		7.  Project onto the view plane (i.e., map onto a viewport).

	IMPORTANT:  We use row vectors for points, unlike Foley et al
		    who use column vectors.  This means that all matrices
		    are transposed from the Foley et al description.
*/

#ifdef DEBUG
printf("Initial Matrix:\n");
DumpMatrix(w);
#endif

	/* Step 1:  Translate the VRP to the origin			*/

	TranslateMatrix(w, -(double)(w->scatter3d.VRP.val[0]),
                           -(double)(w->scatter3d.VRP.val[1]),
                           -(double)(w->scatter3d.VRP.val[2]));

#ifdef DEBUG
printf("Step 1 (VRP translated to origin):\n");
DumpMatrix(w);
#endif

	/* Step 2:  Rotate the VRC to align axes			*/

	Normalize(&Rz, w->scatter3d.VPN);
	CrossProduct(&CrossforRx, w->scatter3d.VUP, Rz);
	Normalize(&Rx, CrossforRx);

	CrossProduct(&Ry, Rz, Rx);

	IdentityMatrix(&tmpMatrix);

	for (i = 0; i < 3; i++) {
	   tmpMatrix.val[i][0] = Rx.val[i]; 
	   tmpMatrix.val[i][1] = Ry.val[i];
	   tmpMatrix.val[i][2] = Rz.val[i];
	}

	MatrixMatrixMultiply(w->scatter3d.transMatrix, tmpMatrix,
		       &(w->scatter3d.transMatrix));

#ifdef DEBUG
printf("Step 2 (VRC rotated to align axes):\n");
DumpMatrix(w);
#endif

	/* Step 3:  Translate PRP to the origin				*/

	TranslateMatrix(w, -(double)(w->scatter3d.PRP.val[0]),
		           -(double)(w->scatter3d.PRP.val[1]),
		           -(double)(w->scatter3d.PRP.val[2]));

#ifdef DEBUG
printf("Step 3 (PRP translated to origin):\n");
DumpMatrix(w);
#endif

	/* Step 4:  Shear so that the center line of the view volume	*/
	/*	    becomes the z axis					*/

	DOP.val[0] = (w->scatter3d.umax + w->scatter3d.umin) / 2.0
				- w->scatter3d.PRP.val[0];
	DOP.val[1] = (w->scatter3d.vmax + w->scatter3d.vmin) / 2.0
				- w->scatter3d.PRP.val[1];

	DOP.val[2] = -w->scatter3d.PRP.val[2];
	DOP.val[3] = 1.0;

	ShearMatrix(w, -DOP.val[0]/DOP.val[2], -DOP.val[1]/DOP.val[2]);

#ifdef DEBUG
printf("Step 4 (Shear to bring view volume to z axis):\n");
DumpMatrix(w);
#endif


	/* Step 5:  Scale so that the view volume becomes a canonical	*/
	/*	    perspective view volume				*/

	IdentityMatrix(&tmpMatrix);
	tmpMatrix.val[2][0] = -DOP.val[0] / DOP.val[2];
	tmpMatrix.val[2][1] = -DOP.val[1] / DOP.val[2];

	VRPPrime.val[3] = 1.0;
	for (i = 0; i < 3; i++)
	   VRPPrime.val[i] = w->scatter3d.VRP.val[i]
				- w->scatter3d.PRP.val[i];

#ifdef DEBUG
printf("VRPPrime: (%f,%f,%f,%f)\n",VRPPrime.val[0], VRPPrime.val[1],
	VRPPrime.val[2], VRPPrime.val[3]);
#endif

	VectorMatrixMultiply(tmpMatrix, VRPPrime, &VRPPrime);

	tmp = w->scatter3d.umax - w->scatter3d.umin;
	tmp = tmp * (VRPPrime.val[2] + w->scatter3d.backplane);
	scaleX = (2.0 * VRPPrime.val[2]) / tmp;

 	tmp = w->scatter3d.vmax - w->scatter3d.vmin;
        tmp = tmp * (VRPPrime.val[2] + w->scatter3d.backplane);
        scaleY = (2.0 * VRPPrime.val[2]) / tmp;

	scaleZ = -1.0 / (VRPPrime.val[2] + w->scatter3d.backplane);

	ScaleMatrix(w, scaleX, scaleY, scaleZ);

#ifdef DEBUG
printf("Step 5 (Scale to make canonical perspective view volume):\n");
DumpMatrix(w);
#endif

	/* Step 6:  Prepare for clipping in homogeneous coordinates	*/

	Zmin = -(VRPPrime.val[2] + w->scatter3d.frontplane);
	Zmin = Zmin / (VRPPrime.val[2] + w->scatter3d.backplane);

	IdentityMatrix(&tmpMatrix);
	tmpMatrix.val[2][2] = 1.0 / (1.0 - Zmin);
	tmpMatrix.val[3][2] = Zmin / (1.0 - Zmin);

	MatrixMatrixMultiply(w->scatter3d.transMatrix, tmpMatrix,
			     &(w->scatter3d.transMatrix));

#ifdef DEBUG
printf("Step 6 (Prepare for clipping in homogeneous coordinates):\n");
DumpMatrix(w);
#endif

	/* Step 7:  Map onto a viewport					*/

	TranslateMatrix(w, 1.0, 1.0, 0.0);
	ScaleMatrix(w, 0.5, 0.5, 1.0);

#ifdef DEBUG
printf("Step 7 (Map to a viewport):\n");
DumpMatrix(w);
#endif
}

/************************************************************************/
/* Normalize() returns a normalized vector				*/
/************************************************************************/

static void Normalize(result, input)
point3D *result;
point3D input;
{
	int	i;
	double	factor;

#ifdef DEBUG
printf("Normalize\n");
#endif

	factor = 0.0;

	for (i = 0; i < 3; i++) {
	   factor += (input.val[i] * input.val[i]);
	}

	factor = sqrt(factor);

	if (factor == 0.0) {
	   fprintf(stderr, "*** Normalize: vector has zero length ***\n");
	   factor = 1.0;
	}

	for (i = 0; i < 3; i++) {
	   result -> val[i] = input.val[i] / factor;
	}
}

/************************************************************************/
/* DotProduct() returns the dot product of two vectors			*/
/************************************************************************/

static double DotProduct(input1, input2)
point3D input1, input2;
{
	int	i;
	double	sum;

#ifdef DEBUG
printf("DotProduct\n");
#endif

	sum = 0.0;

        for (i = 0; i < 3; i++) {
           sum += input1.val[i] * input2.val[i];
        }

	return (sum);
}
/************************************************************************/
/* CrossProduct() returns the cross product of two vectors                  */
/************************************************************************/

static void CrossProduct(result, input1, input2)
point3D *result;
point3D input1, input2;
{
#ifdef DEBUG
printf("CrossProduct\n");
#endif

	result -> val[0] = input1.val[1] * input2.val[2]
				- input1.val[2] * input2.val[1];

	result -> val[1] = input1.val[2] * input2.val[0]
				- input1.val[0] * input2.val[2];

	result -> val[2] = input1.val[0] * input2.val[1]
				- input1.val[1] * input2.val[0];
}




/************************************************************************/
/* VectorMatrixMultiply() multiplies a point by the transformation	*/
/*			  matrix					*/
/************************************************************************/
static void VectorMatrixMultiply(m, pt, newpt)
matrix3D m;
point3D  pt;
point3D *newpt;
{
	int	i, j;
	point3D	tmppt;

#ifdef DEBUG
printf("VectorMatrixMultiply\n");
#endif

	for (i = 0; i < 4; i++) {
	   tmppt.val[i] = 0.0;

	   for (j = 0; j < 4; j++)
	      tmppt.val[i] += pt.val[j] * (m.val[j][i]);
	}

	for (i = 0; i < 4; i++)
	   newpt -> val[i] = tmppt.val[i];
}

/************************************************************************/
/* MatrixMatrixMultiply() multiplies two transformation matrices	*/
/************************************************************************/
static void MatrixMatrixMultiply(m1, m2, result)
matrix3D m1, m2;
matrix3D *result;
{
	int		i, j, k;
	matrix3D	m3;

#ifdef DEBUG
printf("MatrixMatrixMultiply\n");
#endif

	for (i = 0; i < 4; i++) {
	   for (j = 0; j < 4; j++) {
	      m3.val[i][j] = 0.0;

	      for (k = 0; k < 4; k++) {
	         m3.val[i][j] += m1.val[i][k] *  m2.val[k][j];
	      }
	   }
	}

	for (i = 0; i < 4; i++)
	   for (j = 0; j < 4; j++)
	      result -> val[i][j] = m3.val[i][j];
}

/************************************************************************/
/* RotateMatrix()	updates transformation matrix with rotation	*/
/************************************************************************/
static void RotateMatrix(w, angle, axis)
Scatter3DWidget	w;
double	angle;
int	axis;
{
	int	i, j;

	double	 cosA;
	double	 sinA;
	matrix3D m;

#ifdef DEBUG
printf("RotateMatrix\n");
printf("rotation angle %f\n",angle);
#endif

	IdentityMatrix(&m);

	cosA = cos(angle);
	sinA = sin(angle);

	switch (axis) {
	   case XAXIS:
	      m.val[1][1] = cosA;
	      m.val[1][2] = sinA;
	      m.val[2][1] = -sinA;
	      m.val[2][2] = cosA;
	      break;

	   case YAXIS:
	      m.val[0][0] = cosA;
	      m.val[0][2] = -sinA;
	      m.val[2][0] = sinA;
	      m.val[2][2] = cosA;
	      break;

 	   case ZAXIS:
              m.val[0][0] = cosA;
              m.val[0][1] = sinA;
              m.val[1][0] = -sinA;
              m.val[1][1] = cosA;
              break;
	}

	MatrixMatrixMultiply(w->scatter3d.transMatrix, m,
			     &(w->scatter3d.transMatrix));
}

/************************************************************************/
/* ScaleMatrix() scales the transformation matrix by specified value	*/
/************************************************************************/
static void ScaleMatrix(w, scaleX, scaleY, scaleZ)
Scatter3DWidget	w;
double	scaleX, scaleY, scaleZ;
{

matrix3D	m;

#ifdef DEBUG
printf("ScaleMatrix\n");
printf("Scale factors: (%f,%f,%f)\n", scaleX, scaleY, scaleZ);
#endif

	IdentityMatrix(&m);

	m.val[0][0] = scaleX;
	m.val[1][1] = scaleY;
	m.val[2][2] = scaleZ;

	MatrixMatrixMultiply(w->scatter3d.transMatrix, m,
                             &(w->scatter3d.transMatrix));

}

/************************************************************************/
/* ShearMatrix() shears the transformation matrix by specified value	*/
/************************************************************************/
static void ShearMatrix(w, shearX, shearY)
Scatter3DWidget	w;
double	shearX, shearY;
{

matrix3D	m;

#ifdef DEBUG
printf("ShearMatrix\n");
printf("Shear factors: (%f,%f)\n", shearX, shearY);
#endif

	IdentityMatrix(&m);

	m.val[2][0] = shearX;
	m.val[2][1] = shearY;

	MatrixMatrixMultiply(w->scatter3d.transMatrix, m,
                             &(w->scatter3d.transMatrix));

}

/************************************************************************/
/* TranslateMatrix() translates the transformation matrix by		*/
/*			the specified value				*/
/************************************************************************/
static void TranslateMatrix(w, x, y, z)
Scatter3DWidget	w;
double	x, y, z;
{

matrix3D	m;

#ifdef DEBUG
printf("TranslateMatrix\n");
printf("translating to (%f,%f,%f)\n",x, y, z);
#endif

	IdentityMatrix(&m);

	m.val[3][0] = x;
	m.val[3][1] = y;
	m.val[3][2] = z;

	MatrixMatrixMultiply(w->scatter3d.transMatrix, m,
                             &(w->scatter3d.transMatrix));
}

/************************************************************************/
/* ProjectPoint() projects a point onto 2D				*/
/************************************************************************/
static void ProjectPoint(w, pt, scale, newval)
Scatter3DWidget	w;
point3D pt;
point2D *newval;
int	scale;
{
	int	i, j;
	point3D	tmppt;
	point3D tpt;

#ifdef DEBUG
printf("ProjectPoint\n");
#endif

	for (i = 0; i < 4; i++)
	   tpt.val[i] = 0.0;

	tpt.val[3] = 1.0;

	VectorMatrixMultiply(w->scatter3d.transMatrix, pt, &tpt);

#ifdef DEBUG
	printf("translated values (%f,%f,%f,%f)\n",tpt.val[0],tpt.val[1],
			tpt.val[2], tpt.val[3]);
#endif

	newval -> val[0] = scale * tpt.val[0] / tpt.val[3];
	newval -> val[1] = scale * tpt.val[1] / tpt.val[3];

#ifdef DEBUG
	printf("Project from (%f", pt.val[0]);

	for (i = 1; i < 4; i++)
	   printf(",%f", pt.val[i]);

	printf(") to (%f,%f)\n", newval -> val[0], newval -> val[1]);
#endif
}

static void SphericalToCartesian(radius, theta, phi, x, y, z)
double radius;
double theta, phi;
double *x, *y, *z;
{
	*x = radius * sin(phi / 360.0) * cos(theta / 360.0);
	*y = radius * sin(phi / 360.0) * sin(theta / 360.0);
	*z = radius * cos(phi / 360.0);
}

static void DumpMatrix(w)
Scatter3DWidget w;
{
	int	i;
	int	j;

	for (i = 0; i < 4; i++) {
	   for (j = 0; j < 4; j++) {
	      printf("%8.4f ", w->scatter3d.transMatrix.val[i][j]); 
	   }
	   printf("\n");
	}
}

