/*
 *  $Id: mpir.h,v 1.24 1994/06/08 20:49:10 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/* include file for the MPIR implementation of MPI */

/***********************************************************************
*                                                                      *
*   mpir.h                                                             *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <stdio.h>

#ifndef _MPIR_INCLUDE
#define _MPIR_INCLUDE

#include "mpid.h" 
/*****************************************************************************
*                           MPIR DATA STRUCTURES                             *
*****************************************************************************/


struct MPIR_DATATYPE {
    MPIR_NODETYPE dte_type;	/* type of datatype element this is */
    MPIR_COOKIE			/* Cookie to help detect valid item */
    MPIR_BOOL    committed;	/* whether committed or not */
    Int          is_contig;	/* whether entirely contiguous */
    Int              basic;	/* Is this a basic type */
    Int          permanent;	/* Is this a permanent type */
    Int             ub, lb;	/* upper/lower bound of type */
    Int             extent;	/* extent of this datatype */
    Int               size;	/* size of type */
    Int           elements;	/* number of basic elements */
    Int          ref_count;	/* nodes depending on this node */
    Int              align;	/* alignment needed for start of datatype */
    Int                pad;	/* padding for each element of type */
    Int          far *pads;	/* padding for STRUCT types */
    Int              count;	/* replication count */
    MPI_Aint        stride;	/* stride, for VECTOR and HVECTOR types */
    Int           blocklen;	/* blocklen, for VECTOR and HVECTOR types */
    Int       far *indices;	/* array of indices, for (H)INDEXED */
    Int     far *blocklens;	/* array of blocklens for (H)INDEXED */
    MPI_Datatype old_type; 	/* type this type is built of, if 1 */
    MPI_Datatype far *old_types;/* array of types, for STRUCT */
};

typedef enum {
    MPIR_SEND,
    MPIR_RECV
} MPIR_OPTYPE;

/* MPIR_COMMON is a subset of the handle structures that contain common
   elements.  MAKE SURE THAT ANY CHANGES IN THE HANDLES PRESERVES
   THIS ORDER! */
typedef struct {
    MPIR_OPTYPE handle_type;
    MPIR_COOKIE                 /* Cookie to help detect valid item */
    Int         partner;        /* source or destination, depending on type */
    Int         tag;
    Int         contextid;
    MPIR_BOOL   completed;
    MPI_Datatype datatype;
    Int         persistent;     /* is handle persistent (created with 
				   MPI_..._init)? */
    Int         active;         /* relavent only for persistent handles,
				   indicates that the handle is active */
    void far *bufadd;		/* address of buffer */
    Int buflen;			/* length of buffer at bufadd */
    Int count;			/* requested count */

    char far *bufpos;		/* position of next byte to be transferred */
    Int  transfer_count;	/* count of bytes transferred to/from device */
    Int  totallen;		/* length in local bytes of message */

    } MPIR_COMMON;

typedef struct {
    MPIR_OPTYPE handle_type;	/* send or receive */
    MPIR_COOKIE			/* Cookie to help detect valid item */
    Int         dest;		/* destination process for message */
    Int         tag;		/* tag */
    Int         contextid;  	/* context id */
    MPIR_BOOL   completed;  	/* whether operation is completed or not */
    MPI_Datatype datatype;  	/* basic or derived datatype */
    Int         persistent;
    Int         active;
    void far *bufadd;		/* address of buffer */
    Int buflen;			/* length of buffer at bufadd */
    Int count;			/* requested count */
    char far *bufpos;		/* position of next byte to be transferred */
    Int  transfer_count;	/* count of bytes transferred to/from device */
    Int  totallen;		/* length in local bytes of message */

    MPIR_Mode mode;		/* mode: STANDARD, READY, SYNC */
    Int  lrank;                 /* Rank in sending group */
    MPID_SHANDLE dev_shandle;   /* device's version of send handle */
} MPIR_SHANDLE;

typedef MPIR_SHANDLE far *LPMPIR_SHANDLE;
typedef LPMPIR_SHANDLE far *LPPMPIR_SHANDLE;

typedef struct {
    MPIR_OPTYPE handle_type;    /* send or receive */
    MPIR_COOKIE			/* Cookie to help detect valid item */
    Int         source;		/* source process message */
    Int         tag;		/* tag */
//    Int         contextid;	/* context id */
    MPIR_CONTEXT  contextid;	/* context id */
    MPIR_BOOL   completed;	/* Whether this receive has been completed */
    MPI_Datatype datatype;	/* basic or derived datatype */
    Int         persistent;
    Int         active;
    void far *bufadd;		/* address of buffer */
    Int buflen;			/* length of buffer at bufadd */
    Int count;			/* requested count */
    char far *bufpos;		/* position of next byte to be transferred */
    Int  transfer_count;	/* count of bytes transferred to/from device */
    Int  totallen;		/* length in local bytes of message */

    void far *p;		/* Pointer to unexpected data */
    Int  len;			/* Length of this data ? */
    Int  actcount;		/* number of items actually read */
    MPID_RHANDLE dev_rhandle;   /* device's version of recv handle */
} MPIR_RHANDLE; 

typedef MPIR_RHANDLE far *LPMPIR_RHANDLE;
typedef LPMPIR_RHANDLE far *LPPMPIR_RHANDLE;

#define MPIR_HANDLES_DEFINED

union MPIR_HANDLE {
    MPIR_OPTYPE type;           /* send or receive */
    MPIR_COMMON  chandle;       /* common fields */
    MPIR_SHANDLE shandle;
    MPIR_RHANDLE rhandle;
};

typedef enum {
    MPIR_QSHANDLE,
    MPIR_QRHANDLE
} MPIR_QEL_TYPE;

typedef struct _MPIR_QEL {	/* queue elements */
    void far *ptr;		/* queues can contain anything  */
    MPIR_QEL_TYPE qel_type;	/* but we may need to know what */
    struct _MPIR_QEL far *prev; /* previous queue element */
    struct _MPIR_QEL far *next; /* next queue element */
} MPIR_QEL;

typedef struct {		/* header for queues of things like handles */
    MPIR_QEL far *first;	/* first queue element */
    MPIR_QEL far *last;		/* last queue element */
    Int      currlen;		/* current length */
    Int      maxlen;		/* maximum since beginning of run */
} MPIR_QHDR;

typedef struct _MPIR_FDTEL {	/* flat datatype elements */
    MPIR_NODETYPE      type;    /* one of the elementary data types */
    Int                disp;    /* displacement */
    struct _MPIR_FDTEL far *next;   /* pointer to next element */
} MPIR_FDTEL;

typedef MPIR_FDTEL far *LPMPIR_FDTEL;
typedef LPMPIR_FDTEL far *LPPMPIR_FDTEL;
typedef LPPMPIR_FDTEL far *LPPPMPIR_FDTEL;

/*****************************************************************************
*                                GLOBAL DATA                                 *
*****************************************************************************/

/* memory management for fixed-size blocks */
extern void far *MPIR_shandles;	/* sbcnst MPIR_SHANDLES */
extern void far *MPIR_rhandles;	/* sbcnst MPIR_RHANDLES */
extern void far *MPIR_dtes;	/* sbcnst datatype elements */
extern void far *MPIR_qels;	/* sbcnst queue elements */
extern void far *MPIR_fdtels;	/* sbcnst flat datatype elements */
extern void far *MPIR_hbts;	/* sbcnst height balanced tree roots for cacheing */
extern void far *MPIR_hbt_els;	/* sbcnst height balanced tree nodes for cacheing */

/* queues */
extern MPIR_QHDR MPIR_posted_recvs;
extern MPIR_QHDR MPIR_unexpected_recvs;

/* MPIR process id (from device) */
extern Int MPIR_tid;

/* Fortran logical values */
extern Int MPIR_F_TRUE, MPIR_F_FALSE;
#define MPIR_TO_FLOG(a) ((a) ? MPIR_F_TRUE : MPIR_F_FALSE)
/* 
   Note on true and false.  This code is only an approximation.
   Some systems define either true or false, and allow some or ALL other
   patterns for the other.  This is just like C, where 0 is false and 
   anything not zero is true.  Modify this test as necessary for your
   system.
 */
#define MPIR_FROM_FLOG(a) ( (a) == MPIR_F_TRUE ? 1 : 0 )

/* 
   Standardized error testing

   Many of the MPI routines take arguments of the same type.  These
   macros provide tests for these objects.

   It is intended that the tests for a valid opaque object such as 
   a communicator can check to insure that the object is both a communicator
   and that it is valid (hasn't been freed).  They can also test for
   null pointers.

   These are not used yet; we are still looking for the best ways to 
   define them.

   The intent is to use them in this manner:

   if (MPIR_TEST_...() || MPIR_TEST_... || ... ) 
        return MPIR_ERROR( comm, errno, "Error in MPI_routine" );

   The hope is, that in the NO_ERROR_CHECKING case, the optimizer will
   be smart enough to remove the code.
 */
#ifdef MPIR_NO_ERROR_CHECKING
#define MPIR_TEST_SEND_TAG(comm,tag)      0
#define MPIR_TEST_RECV_TAG(comm,tag)      0
#define MPIR_TEST_SEND_RANK(comm,rank)    0
#define MPIR_TEST_RECV_RANK(comm,rank)    0
#define MPIR_TEST_COUNT(comm,count)       0
#define MPIR_TEST_OP(comm,op)             0
#define MPIR_TEST_GROUP(comm,group)       0
#define MPIR_TEST_COMM(comm,comm1)        0
#define MPIR_TEST_REQUEST(comm,request)   0
#define MPIR_TEST_DATATYPE(comm,datatype) 0
#define MPIR_TEST_ALIAS(b1,b2)            0
#define MPIR_TEST_ARG(arg)                0

#else
#define MPIR_TEST_SEND_TAG(comm,tag) \
    ( ((tag) < 0 ) && ((errno = MPI_ERR_TAG ) != 0))
    /* This requires MPI_ANY_TAG == -1 */
#define MPIR_TEST_RECV_TAG(comm,tag) \
    ( ((tag) < MPI_ANY_TAG) &&  ((errno = MPI_ERR_TAG ) != 0))
    /* This exploits MPI_ANY_SOURCE==-2, MPI_PROC_NULL==-1 */
#define MPIR_TEST_SEND_RANK(comm,rank) \
    ( ((rank) < MPI_PROC_NULL || (rank) >= (comm)->group->np)\
           && ((errno = MPI_ERR_RANK) != 0))
    /* This requires min(MPI_PROC_NULL,MPI_ANY_SOURCE)=-2 */
#define MPIR_TEST_RECV_RANK(comm,rank) \
    (((rank) < -2 || (rank) >= (comm)->group->np) &&  ((errno = MPI_ERR_RANK) != 0))
#define MPIR_TEST_COUNT(comm,count) ( ((count) < 0) && ((errno = MPI_ERR_COUNT) != 0))
#define MPIR_TEST_OP(comm,op)       ( (!(op)) && ((errno = MPI_ERR_OP ) != 0))
#define MPIR_TEST_GROUP(comm,group) ( (!(group)) && ((errno = MPI_ERR_GROUP ) != 0))
#define MPIR_TEST_COMM(comm,comm1)  ( (!(comm1)) && ((errno = MPI_ERR_COMM ) != 0))
#define MPIR_TEST_REQUEST(comm,request) \
    ( (!(request)) && ((errno = MPI_ERR_REQUEST) != 0))
#define MPIR_TEST_DATATYPE(comm,datatype) \
    ( (!(datatype)) && ((errno = MPI_ERR_TYPE ) != 0))
#define MPIR_TEST_ALIAS(b1,b2)      ( ((b1)==(b2)) && ((errno = MPI_ERR_BUFFER) != 0))
#define MPIR_TEST_ARG(arg)  (!(arg) && ((errno = MPI_ERR_ARG) != 0) )
#endif 

Int	MPIR_UnPackMessage(char far *, Int , MPI_Datatype , Int , MPI_Request );
Int	MPIR_PackMessage (char far *, Int , MPI_Datatype , Int , MPI_Request );
Int	MPIR_EndPackMessage (MPI_Request request);

Int	MPIR_Send_init( void far *, Int , MPI_Datatype , Int , Int , 
			MPI_Comm , MPI_Request , Int , Int );

#endif
