
/***********************************************************************
*                                                                      *
*   p4_mon.c                                                           *
*   p4 1.4 for MS-Windows 3.1                                          *
*   current version: 0.99b          07/16/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is the WIN31 version of the p4 Parallel Programming System    *
*   developed at Argonne National Laboratory.  Note their COPYRIGHT.   *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/p4 )                          *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include "p4.h"
#include "p4_sys.h"

/*
 *  WIN provides non-preemptive multitasking only 
 *  calling this function will process messages for current task
 *  (especially msg sent from outside to abort) and, at the same time,
 *  keeps other tasks running
 */

P4VOID	FAR PASCAL _export P4WinMessageLoop(void)
{
    MSG msg;
    
    while (PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE)) 
    {
        if (!GetMessage (&msg, NULL, 0, 0))
            p4_error ("Received WM_QUIT - Aborting", p4_get_my_id ());
        TranslateMessage (&msg) ;
        DispatchMessage (&msg) ;
    }
}    


P4VOID FAR PASCAL _export p4_lock_init (p4_lock_t far *l)
{
    *l = 0; // unlocked
}


P4VOID FAR PASCAL _export p4_lock (p4_lock_t far *l)
{   
    while (*l != 0)
    {
    	if (*l != 1)
    		p4_error ("lock unequal 0,1 !", 0);
    		
	/* WIN: If locked, let others unlock it */
		P4WinMessageLoop ();
    }                  
    *l = 1; // locked now
}
     
     
P4VOID FAR PASCAL _export p4_unlock (p4_lock_t far *l)
{
    *l = 0; // unlocked
}
    



Int FAR PASCAL _export p4_moninit (p4_monitor_t far *m, Int i)
{
    Int j;
    struct p4_mon_queue far *q;

    p4_lock_init(&(m->mon_lock)); 

    if (i)
    {
	m->qs = (struct p4_mon_queue far *) p4_shmalloc(sizeof(struct p4_mon_queue) * i);
	if (m->qs == NULL)
	{
	    p4_dprintf("OOPS! p4_moninit: p4_shmalloc failed ***\n");
	    return (-1);
	}
	for (j = 0; j < i; j++)
	{
	    q = m->qs + j;
	    q->count = 0;
	    p4_lock_init(&(q->delay_lock));  
	    p4_lock(&(q->delay_lock));      
	}
    }
    else
		m->qs = NULL;
    return (0);
}

P4VOID FAR PASCAL _export p4_menter(p4_monitor_t far *m)
{
    ALOG_LOG(p4_local->my_id,REQUEST_MONITOR_ENTRY,m,"");
    p4_lock(&(m->mon_lock));
    ALOG_LOG(p4_local->my_id,ENTER_MONITOR,m,"");
}

P4VOID FAR PASCAL _export p4_mexit(p4_monitor_t far *m)
{
    ALOG_LOG(p4_local->my_id,OPEN_DOOR,m,"");
    ALOG_LOG(p4_local->my_id,EXIT_MONITOR,m,"");
    p4_unlock(&(m->mon_lock));
}

P4VOID FAR PASCAL _export p4_mdelay(p4_monitor_t far *m, Int i)
{
    struct p4_mon_queue far *q;

    q = m->qs + i;
    q->count++;
    ALOG_LOG(p4_local->my_id,ENTER_DELAY_QUEUE,m,"");
    ALOG_LOG(p4_local->my_id,OPEN_DOOR,m,"");
    p4_unlock(&(m->mon_lock));	// step out of monitor
    p4_lock(&(q->delay_lock));	
    ALOG_LOG(p4_local->my_id,EXIT_DELAY_QUEUE,m,"");
}

P4VOID FAR PASCAL _export p4_mcontinue(p4_monitor_t far *m, Int i)
{
    struct p4_mon_queue far *q;

    q = m->qs + i;
    if (q->count)
    {
	q->count--;
	ALOG_LOG(p4_local->my_id,SECRET_EXIT_MONITOR,m,"");
	p4_unlock(&(q->delay_lock));
    }
    else
    {
	ALOG_LOG(p4_local->my_id,OPEN_DOOR,m,"");
	ALOG_LOG(p4_local->my_id,EXIT_MONITOR,m,"");
	p4_unlock(&(m->mon_lock));
    }
}

Int num_in_mon_queue (p4_monitor_t far *m, Int i)
{
    struct p4_mon_queue far *q;

    q = m->qs + i;
    return (q->count);
}


/* ------------------  getsub monitor -------------------- */

Int FAR PASCAL _export p4_getsub_init(p4_getsub_monitor_t far *gs)
{
    gs->sub = 0;
    return (p4_moninit(&(gs->m), 1));
}

P4VOID FAR PASCAL _export p4_getsubs(p4_getsub_monitor_t far *gs, 
				 Int far *s, Int max, Int nprocs, Int stride)
{
    p4_menter(&(gs->m));
    if (gs->sub <= max)
    {
		*s = gs->sub;
		gs->sub += stride;
		p4_mexit(&(gs->m));
    }
    else
    {
		*s = -1;
		if (num_in_mon_queue(&(gs->m), 0) < nprocs - 1)
		    p4_mdelay(&(gs->m), 0);
		else
		    gs->sub = 0;
		p4_mcontinue(&(gs->m), 0);
    }
}


/* ------------------  barrier monitor -------------------- */

Int FAR PASCAL _export p4_barrier_init(p4_barrier_monitor_t far *b)
{
    return (p4_moninit(&(b->m), 1));  // 1 monitor queue
}

P4VOID FAR PASCAL _export p4_barrier (p4_barrier_monitor_t far *b, Int nprocs)
{
    p4_menter(&(b->m));
    if (num_in_mon_queue(&(b->m), 0) < nprocs - 1)
	p4_mdelay(&(b->m), 0);
    p4_mcontinue(&(b->m), 0);
}


/* ------------------  askfor monitor -------------------- */

Int p4_askfor_init(p4_askfor_monitor_t *af)
{

    af->pgdone = 0;
    af->pbdone = 0;
    /* alog assumes only one askfor per program */
    ALOG_LOG(p4_local->my_id,PBDONE,0,"");
    ALOG_LOG(p4_local->my_id,PGDONE,0,"");
    ALOG_LOG(p4_local->my_id,UPDATE_NUM_SUBPROBS,0,"");
    return (p4_moninit(&(af->m), 1));
}

Int FAR PASCAL _export p4_askfor (p4_askfor_monitor_t far *af, 
					Int nprocs, Int (far *getprob_fxn) (), 
					P4VOID far *problem, P4VOID (far *reset_fxn) ())
{
    Int rc;

    p4_menter(&(af->m));
    if (!(af->pgdone) && af->pbdone)
    {
	if (num_in_mon_queue(&(af->m), 0) < nprocs - 1)
	    p4_mdelay(&(af->m), 0);
    }
    else
    {
	while (!(af->pgdone) && !(af->pbdone))
	{
	    if ((rc = (*getprob_fxn) (problem)) == 0)
	    {
		p4_mcontinue(&(af->m), 0);
		return (rc);
	    }
	    else
	    {
		if (num_in_mon_queue(&(af->m), 0) == nprocs - 1)
		{
		    af->pbdone = 1;
		    ALOG_LOG(p4_local->my_id,PBDONE,1,"");
		}
		else
		    p4_mdelay(&(af->m), 0);
	    }
	}
    }
    if (af->pgdone)
    {
	rc = (-1);
	p4_mcontinue(&(af->m), 0);
    }
    else
    {
	rc = af->pbdone;
	if (num_in_mon_queue(&(af->m), 0) == 0)
	{
	    (*reset_fxn) ();
	    af->pbdone = 0;
	}
	p4_mcontinue(&(af->m), 0);
    }
    return (rc);
}

P4VOID FAR PASCAL _export p4_update(p4_askfor_monitor_t far *af, 
				Int (far *putprob_fxn) (), P4VOID far *problem)
{
    p4_menter(&(af->m));
    if (putprob_fxn(problem))
	p4_mcontinue(&(af->m), 0);
    else
	p4_mexit(&(af->m));
}

P4VOID FAR PASCAL _export p4_probend(p4_askfor_monitor_t far *af, Int code)
{
    p4_menter(&(af->m));
    af->pbdone = code;
    ALOG_LOG(p4_local->my_id,PBDONE,code,"");
    p4_mexit(&(af->m));
}

P4VOID FAR PASCAL _export p4_progend(p4_askfor_monitor_t far *af)
{
    p4_menter(&(af->m));
    af->pgdone = 1;
    ALOG_LOG(p4_local->my_id,PGDONE,1,"");
    p4_mcontinue(&(af->m), 0);
}

Int FAR PASCAL _export p4_create(Int (far *fxn) ())
{

    p4_error("Sorry, cannot fork - p4_create not implemented for WIN31\n", 0);

    return (0);
}
