
/***********************************************************************
*                                                                      *
*   p4_utils.c                                                         *
*   p4 1.4 for MS-Windows 3.1                                          *
*   current version: 0.99b          07/16/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is the WIN31 version of the p4 Parallel Programming System    *
*   developed at Argonne National Laboratory.  Note their COPYRIGHT.   *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/p4 )                          *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include "p4.h"
#include "p4_sys.h"
#include <stdlib.h>
#include <string.h>
#include <memory.h>

LPPSTR bm_argv;		// access to master parameters, used in read_procgroup
struct p4_global_data far *p4_global;
struct local_data *p4_localtab;
static char szPkgName[8];

int FAR PASCAL LibMain (HANDLE hInstance, WORD wDataSeg, WORD wHeapSize,
                        LPSTR lpszCmdLine)
{
	if (wHeapSize > 0)
	    UnlockData (0) ;  
	
	p4_global = NULL;
	p4_localtab = malloc (P4_MAXPROCS * sizeof (struct local_data));
	memset (p4_localtab, 0, P4_MAXPROCS * sizeof (struct local_data));     
//	sync = 0;            
	_fstrcpy (szPkgName, "p4");	// by default we assume a native p4 application
	
	return 1;
}


int FAR PASCAL _export WEP (int nParam)
{
	// try to free all global memory used
	
	p4_shfree (p4_global->procgroup);
	p4_shfree (p4_global->global_barrier.m.qs); // cluster_barrier
/* #pragma message ("Free shmem cluster barriers") done by WIN */
	
	p4_shfree (p4_global);
    return 1;
}	


char far * FAR PASCAL _export p4_version(void)
{
    return((char far *)P4_PATCHLEVEL); 
}


char far * p4_machine_type (void)
{
	return ((char far *)P4_MACHINE_TYPE);
}


P4VOID FAR PASCAL _export P4WinSetPkgName (char far *pkgName)
{
	int len = _fstrlen (pkgName);
	
	if (len > sizeof (szPkgName) - 1)
		len = sizeof (szPkgName) - 1;
		
	_fstrncpy (szPkgName, pkgName, len);
	szPkgName[len] = '\0';
}			
	

int FAR PASCAL _export p4_initenv(int far *argc, LPPSTR argv)
{
//    int i;
    int rc;
/* slaves were always forked - they never called p4_initenv but they exited from it
   WIN : slaves enter p4_initenv() too
         but there are no remote masters anymore 
*/         
    P4BOOL am_slave = FALSE;
                   
    if (NULL != p4_global)  // p4_global already initialized
        am_slave = TRUE;	// I am a slave
    
    if (am_slave)
    {
        rc = slave_start (argc, argv); 
		ALOG_SETUP(p4_local->my_id,ALOG_TRUNCATE);
	}
    else
    {   
    	bm_argv = argv;	// used in read_procgroup to find executable path
		rc = bm_start(argc, argv);  // master
        ALOG_MASTER(0,ALOG_TRUNCATE);
        ALOG_DEFINE(BEGIN_USER,"beg_user","");
        ALOG_DEFINE(END_USER,"end_user","");
        ALOG_DEFINE(BEGIN_SEND,"beg_send","");
        ALOG_DEFINE(END_SEND,"end_send","");
        ALOG_DEFINE(BEGIN_RECV,"beg_recv","");
        ALOG_DEFINE(END_RECV,"end_recv","");
        ALOG_DEFINE(BEGIN_WAIT,"beg_wait","");
        ALOG_DEFINE(END_WAIT,"end_wait","");
    }
    ALOG_LOG(p4_local->my_id, BEGIN_USER,0,"");
    return (rc);
}


char far * FAR PASCAL _export p4_shmalloc(unsigned long n)
{
    char far *rc;

    if ((rc = MD_shmalloc(n)) == NULL)
		p4_dprintf("p4_shmalloc returning NULL; request = %ld bytes\n",n);
    
    return (rc);
}


P4VOID FAR PASCAL _export p4_shfree(void far *p)
{
    MD_shfree(p);
}


Int FAR PASCAL _export p4_num_cluster_ids(void)
{
    return (p4_global->
    		clustertab[CLUSTER_IDX(p4_get_my_id())].local_slave_count + 1);
}


Int FAR PASCAL _export p4_num_total_ids(void)
{
    return (p4_global->num_in_proctable);
}

Int FAR PASCAL _export p4_num_total_slaves(void)
{
    return (p4_global->num_in_proctable - 1);
}


P4VOID FAR PASCAL _export p4_global_barrier (Int type)
{
    Int dummy[1];

    dummy[0] = 0;
    p4_global_op (type, (char far *) dummy, (Int)1, (Int)sizeof(Int),
    			  p4_int_sum_op, (Int)P4INT);
}


P4VOID FAR PASCAL _export p4_get_cluster_masters(Int far *numids, Int far ids[])
{
    int node;

    ids[0] = 0;		// big master
    *numids = 1;
    
    for (node = 1; node < p4_global->num_in_proctable; node++)
    {
		if (SLAVE_IDX(node) != 0)
		    continue;
		ids[(*numids)++] = node;
    }
}


P4VOID FAR PASCAL _export p4_get_cluster_ids(Int far *start, Int far *end)
{ 
	struct p4_cluster_data far * pc;
	
	pc = &(p4_global->clustertab[CLUSTER_IDX(p4_get_my_id())]);
    *start = pc->low_cluster_id;
    *end   = pc->hi_cluster_id;
}


Int FAR PASCAL _export p4_get_my_id(void)
{
    return (get_my_p4_local()->my_id);
}


Int FAR PASCAL _export p4_get_my_cluster_id (void)
{
	return (p4_global->proctable[p4_get_my_id ()].slave_idx);
}
  
  
P4BOOL FAR PASCAL _export p4_am_i_cluster_master(void)
{
	return (p4_global->proctable[p4_get_my_id ()].slave_idx == 0);
}


P4BOOL in_same_cluster(Int i, Int j)
{
    return (p4_global->proctable[i].cluster_idx == 
    		p4_global->proctable[j].cluster_idx);
}

                                              
P4VOID FAR PASCAL _export p4_cluster_shmem_sync (LPSTR far *cluster_shmem)
{   
    Int my_id = p4_get_my_id();
    
    if (my_id == 0)  /* cluster master */
		p4_global->clustertab[CLUSTER_IDX(my_id)].cluster_shmem = *cluster_shmem;  

    p4_barrier(&(p4_global->clustertab[CLUSTER_IDX(my_id)].cluster_barrier),
    			p4_num_cluster_ids());
    
    if (my_id != 0)
		*cluster_shmem = p4_global->clustertab[CLUSTER_IDX(my_id)].cluster_shmem;
}


#if defined(USE_XX_SHMALLOC)
/* This is not machine dependent code but is only used on some machines */

/*
  Memory management routines from ANSI K&R C, modified to manage
  a single block of shared memory.
  Have stripped out all the usage monitoring to keep it simple.

  To initialize a piece of shared memory:
    xx_init_shmalloc(char *memory, unsigned nbytes)

  Then call xx_shmalloc() and xx_shfree() as usual.
*/  
/* WIN again trouble with pointers and segments > 64K
   huge pointers necessary - but handling of these in this code is difficult
   easiest solution - we use GlobalAlloc
*/

#define LOG_ALIGN 6         	// multiples of 64 bytes
#define ALIGNMENT (1 << LOG_ALIGN)

/* ALIGNMENT is assumed below to be bigger than sizeof(p4_lock_t) +
   sizeof(Header *), so do not reduce LOG_ALIGN below 4 */

union header
{
    struct
    {
	union header far *ptr;	/* next block if on free list */
	unsigned Int size;		/* size of this block */
    } s;
    char align[ALIGNMENT];	/* Align to ALIGNMENT byte boundary */
};

typedef union header Header;

typedef Header far *LPHEADER;
typedef LPHEADER far *LPPHEADER;
static LPPHEADER freep;		/* pointer to pointer to start of free list */
static p4_lock_t far *shmem_lock;	/* Pointer to lock */

P4VOID xx_init_shmalloc (LPSTR memory, unsigned long nbytes)
/*
  memory points to a region of shared memory nbytes long.
  initialize the data structures needed to manage this memory
*/
{
    Int nunits = nbytes >> LOG_ALIGN;
    LPHEADER region = (LPHEADER) memory;

    /* Quick check that things are OK */

    if (ALIGNMENT != sizeof(Header) ||
	    ALIGNMENT < (sizeof(LPHEADER) + sizeof(p4_lock_t)))
    {
        p4_dprintfl(00,"%d %d\n",sizeof(Header),sizeof(p4_lock_t));    // 00 ??
		p4_error("xx_init_shmem: Alignment %d is wrong", ALIGNMENT);
    }

    if (!region)
		p4_error("xx_init_shmem: Passed null pointer", 0);

    if (nunits < 2)
		p4_error("xx_init_shmem: Initial region is ridiculously small",
		 						(Int) nbytes);

    /*
     * Shared memory region is structured as follows
     * 
     * 1) (Header *) freep ... free list pointer 2) (p4_lock_t) shmem_lock ...
     * space to hold lock 3) padding up to alignment boundary 4) First header
     * of free list
     */

    freep = (LPPHEADER) region;			/* Free space pointer in first block  */
    shmem_lock = (p4_lock_t *) (freep + 1);	/* Lock still in first block */
    (region + 1)->s.ptr = *freep = region + 1;	/* Data in rest */
    (region + 1)->s.size = nunits - 1;	/* One header consumed already */

    p4_lock_init(shmem_lock);                /* Initialize the lock */
}
  

char far *xx_shmalloc(unsigned long nbytes)
{
    LPHEADER p, prevp;
    char far *address = NULL;
    unsigned long nunits;

    /* Force entire routine to be single threaded */
    p4_lock(shmem_lock);

    nunits = ((nbytes + sizeof(Header) - 1) >> LOG_ALIGN) + 1;

    prevp = *freep;
    for (p = prevp->s.ptr;; prevp = p, p = p->s.ptr)
    {
		if (p->s.size >= nunits)
		{			/* Big enuf */
		    if (p->s.size == nunits)	/* exact fit */
				prevp->s.ptr = p->s.ptr;
		    else
		    {			/* allocate tail end */
				p->s.size -= nunits;
				p += p->s.size;
				p->s.size = nunits;
		    }
		    
		    *freep = prevp;
		    address = (char far *)(p + 1);
		    break;
		}
	
		if (p == *freep)
		{			
		/* wrapped around the free list ... no fit found */
		    address = NULL;
		    break;
		}
    }

    /* End critical region */
//    (P4VOID) p4_unlock(shmem_lock);
    p4_unlock(shmem_lock);

    if (address == NULL)
		p4_dprintf("xx_shmalloc: returning NULL; requested %ld bytes\n",nbytes);
    
    return address;
}
     
     
P4VOID xx_shfree(void far *ap)
{
    LPHEADER bp, p;

    /* Begin critical region */
//    (P4VOID) p4_lock(shmem_lock);
    p4_lock(shmem_lock);

    if (!ap)
		return;			/* Do nothing with NULL pointers */

    bp = (LPHEADER) ap - 1;	/* Point to block header */

    for (p = *freep; !(bp > p && bp < p->s.ptr); p = p->s.ptr)
		if (p >= p->s.ptr && (bp > p || bp < p->s.ptr))
	    	break;		/* Freed block at start of end of arena */

    if (bp + bp->s.size == p->s.ptr)
    {				/* join to upper neighbour */
		bp->s.size += p->s.ptr->s.size;
		bp->s.ptr = p->s.ptr->s.ptr;
    }
    else
		bp->s.ptr = p->s.ptr;

    if (p + p->s.size == bp)
    {				/* Join to lower neighbour */
		p->s.size += bp->s.size;
		p->s.ptr = bp->s.ptr;
    }
    else
		p->s.ptr = bp;

    *freep = p;

    /* End critical region */
//    (P4VOID) p4_unlock(shmem_lock);
    p4_unlock(shmem_lock);
}
#endif // USE_XX_SHMALLOC


P4VOID setup_conntab(void)
{   
    struct local_data *p4_local;
	int i;
    Int my_id;

	p4_local = get_my_p4_local ();                                
    p4_dprintfl(60, "setup_conntab: myid=%ld\n", p4_local->my_id);
    
    my_id = p4_get_my_id();

    for (i = 0; i < p4_global->num_in_proctable; i++)
    {
		if (i == my_id)
		    p4_local->conntypetab[i] = CONN_ME;
		else 
		    p4_local->conntypetab[i] = CONN_LOCAL;
    }
    p4_dprintfl(60, "conntab after setup_conntab:\n");
    dump_conntab(60);
}

// private to p4_wait_for_end and zap_p4_processes                           
static char szAppName[16];
static int nProcsFinished = 0;
                           
void FAR PASCAL _export p4_wait_for_end (void)
{
    struct local_data *p4_local;

	p4_local = get_my_p4_local ();                                
    ALOG_LOG(p4_local->my_id,END_USER,0,"");
    ALOG_OUTPUT;

	nProcsFinished++;
    if (p4_get_my_id() != 0)	// slaves finish
    {
	    p4_dprintfl(90, "exit wait_for_end \n");
    	return;
    }

    /* Wait for all slave processes to die */
    p4_dprintfl(90, "enter wait_for_end nss=%d\n",p4_global->n_started_slaves);

	while (nProcsFinished < p4_global->n_started_slaves)
	{
		P4WinMessageLoop ();
	}
	
    MessageBox (hwnd, "All processes finished", "P4 INFO", MB_ICONINFORMATION);
	p4_dprintfl(90, "exit wait_for_end \n");
    return ;
}

          
P4VOID FAR PASCAL _export zap_p4_processes(void)
{
    int i;
    HWND hWndSlave;
    
    if (p4_global == NULL)
        return; 
    
	for (i = 1; i <= p4_global->n_started_slaves; i++)
	{
	// WIN Does wsprintf() really not allow more than one argument ???
		_fstrcpy (szAppName, szPkgName);
		wsprintf (szAppName + _fstrlen (szAppName), "-Rank %d", i);
	//	printf ("%s\n", szAppName);
		if ((hWndSlave = FindWindow ("WinMPI", szAppName)) != NULL)
		{
			if (GetCurrentTask () != GetWindowTask (hWndSlave))
				PostMessage (hWndSlave, WM_DESTROY, NULL, NULL);
		}
	}		
}
 
 
P4VOID get_qualified_hostname (char far *str) // WIN31 Not very important 
{
    _fstrcpy (str,"win_node");
}
