/*
 *  $Id: scatterv.c,v 1.17 1994/06/07 21:29:52 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   scattev.c                                                          *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop

#ifndef lint
static char vcid[] = "$Id: scatterv.c,v 1.17 1994/06/07 21:29:52 gropp Exp $";
#endif /* lint */

#ifdef __BORLANDC__
/* BC cant find local header files - BUG ??? */
#include "..\src\coll\coll.h"
#else
#include "coll.h"
#endif /* __BORLANDC__ */

/*@

MPI_Scatterv - Scatters a buffer in parts to all tasks in a group

Input Parameters:
. sendbuf - address of send buffer (choice, significant only at root) 
. sendcounts - integer array (of length group size) 
specifying the number of elements to send to each processor  
. displs - integer array (of length group size). Entry 
 i  specifies the displacement (relative to sendbuf  from
which to take the outgoing data to process  i  
. sendtype - data type of send buffer elements (handle) 
. recvcount - number of elements in receive buffer (integer) 
. recvtype - data type of receive buffer elements (handle) 
. root - rank of sending process (integer) 
. comm - communicator (handle) 

Output Parameter:
. recvbuf - address of receive buffer (choice) 

@*/
Int MPI_Scatterv ( void far *sendbuf, Int far *sendcnts, Int far *displs, 
				   MPI_Datatype sendtype,
				   void far *recvbuf, Int recvcnt, MPI_Datatype recvtype,
				   Int root, MPI_Comm comm)
{
  MPI_Status status;
  Int        rank, size;
  Int        errno = MPI_SUCCESS;
  Int        flag;

  if (MPIR_TEST_COMM(comm,comm) || MPIR_TEST_DATATYPE(comm,sendtype)) 
      return MPIR_ERROR(comm,errno,"Error in MPI_SCATTER" );

  /* Get size and rank */
  MPI_Comm_size ( comm, &size );
  MPI_Comm_rank ( comm, &rank );

  /* Check for invalid arguments */
  if ( ( (root            <  0)              && ((errno = MPI_ERR_ROOT) != 0))  || 
       ( (root            >= size)           && ((errno = MPI_ERR_ROOT) != 0))  ||
	   ( ((recvcnt>0)&&(recvbuf==(void *)0)) && ((errno = MPI_ERR_BUFFER) != 0)) )
    return MPIR_ERROR( comm, errno, "Error in MPI_SCATTERV" );
  
  /* Check for intra-communicator */
  MPI_Comm_test_inter ( comm, &flag );
  if (flag) 
    return MPIR_ERROR(comm, MPI_ERR_COMM,
					  "Inter-communicator invalid in MPI_SCATTERV");
				  
  /* Switch communicators to the hidden collective */
  comm = comm->comm_coll;

  /* Lock for collective operation */
  MPID_THREAD_LOCK(comm);

  /* If I'm the root, then scatter */
  if ( rank == root ) {
    MPI_Aint extent;
    Int      i;

    MPI_Type_extent(sendtype, &extent);
    for ( i=0; i<root; i++ )
      MPI_Send( (void far *)((Int)sendbuf+displs[i]*extent), sendcnts[i], 
        sendtype, i, MPIR_SCATTERV_TAG, comm);

    MPI_Sendrecv((void far *)((Int)sendbuf+displs[rank]*extent), sendcnts[rank], 
                 sendtype, rank, MPIR_SCATTERV_TAG, recvbuf, recvcnt, recvtype, 
                 rank, MPIR_SCATTERV_TAG, comm, &status);

    for ( i=root+1; i<size; i++ )
      MPI_Send( (void far *)((Int)sendbuf+displs[i]*extent), sendcnts[i], 
        sendtype, i, MPIR_SCATTERV_TAG, comm);
  }
  else
    MPI_Recv(recvbuf,recvcnt,recvtype,root,MPIR_SCATTERV_TAG,comm,&status);

  /* Unlock for collective operation */
  MPID_THREAD_UNLOCK(comm);

  return (errno);
}


