/*
 *  $Id: comm_split.c,v 1.27 1994/06/07 21:24:30 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   comm_sp.c                                                          *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop

#include <malloc.h>

#define MPIR_Table_color(table,i) table[(i)]
#define MPIR_Table_key(table,i)   table[((i)+size)]
#define MPIR_Table_next(table,i)  table[((i)+(2*size))]

/*@

MPI_Comm_split - Creates new communicators based on colors and keys

Input Parameters:
. comm - communicator (handle) 
. color - control of subset assignment (integer) 
. key - control of rank assigment (integer) 

Output Parameter:
. newcomm - new communicator (handle) 

Algorithm:

The current algorithm used has quite a few (read: a lot of) inefficiencies 
that can be removed.  Here's what we do for now:
$ 1) A table is built of colors, and keys (has a next field also).
$ 2) The tables of all processes are merged using MPI_Allreduce
$ 3) Two contexts are allocated for all the comms to be created.  These
     same two contexts can be used for all created communicators since
     the communicators will not overlap.
$ 4) If the local process has a color of MPI_UNDEFINED, it can return
     a NULL comm. 
$ 5) The table entries that match the local process color are sorted 
     by key/rank. 
$ 6) A group is created from the sorted list and a communicator is created
     with this group and the previously allocated contexts.
@*/
Int MPI_Comm_split ( MPI_Comm  comm, Int color, Int key, MPI_Comm far *comm_out)
{
  Int           size, rank, head, new_size, i;
  Int           far *table, far *table_in;
  MPIR_CONTEXT  context;
  Int           far *group_list;
  MPI_Group     comm_group, group;
  MPI_Comm      new_comm;
  Int           errno = MPI_SUCCESS;

  /* If we don't have a communicator we don't have anything to do */
  if ( MPIR_TEST_COMM(comm,comm) ||
	   ( (comm->comm_type == MPIR_INTER) && ((errno = MPI_ERR_COMM) != 0) ) )
  {
    (*comm_out) = MPI_COMM_NULL;
    return MPIR_ERROR( comm, errno, "Error in MPI_COMM_SPLIT" );
  }

  /* Create and initialize split table. */
  (void) MPI_Comm_size ( comm, &size );
  (void) MPI_Comm_rank ( comm, &rank );
  table = (Int far *) MPI_CALLOC ( 2 * 3 * size, sizeof(Int) );
  table_in = table + (3 * size);
  if (!table) 
	return MPIR_ERROR( comm, MPI_ERR_EXHAUSTED,
					  "Out of space in MPI_COMM_SPLIT" );
  MPIR_Table_color(table_in,rank) = color;
  MPIR_Table_key(table_in,rank)   = key;

  /* Combine the split table. I only have to combine the colors and keys */
  (void) MPI_Allreduce(table_in, table, size * 2, MPI_INT, MPI_SUM, comm);

  /* Allocate 2 contexts */
  (void) MPIR_Context_alloc( comm, 2, &context );

  /* If my color is MPI_UNDEFINED, then I'm not in a comm and can */
  /* stop here since there are no more communications with others */
  /* I'll even go ahead and free the 2 contexts I allocated above */
  if ( MPIR_Table_color(table,rank) == MPI_UNDEFINED ) {
    MPI_FREE(table);
    (void) MPIR_Context_dealloc( comm, 2, context );
    (*comm_out) = MPI_COMM_NULL;
    return (errno);
  }

  /* Sort the table */
  (void) MPIR_Sort_split_table ( size, rank, table, &head, &new_size );
     
  /* Create group of processes that share my color */
  group_list = (Int far *) MPI_MALLOC ( new_size * sizeof(Int) );
  for ( i=0; i<new_size; i++, head=MPIR_Table_next(table,head) )
    group_list[i] = head;
  (void) MPI_Comm_group ( comm, &comm_group );
  (void) MPI_Group_incl ( comm_group, new_size, group_list, &group );
  (void) MPI_Group_free ( &comm_group );
  MPI_FREE(table);
  MPI_FREE(group_list);

  /* Make communicator using contexts allocated */
  new_comm                = (*comm_out) = MPI_NEW(struct MPIR_COMMUNICATOR);
  if (!new_comm)
	return MPIR_ERROR( comm, MPI_ERR_EXHAUSTED, 
					  "Out of space in MPI_COMM_SPLIT" );
  new_comm->comm_type     = MPIR_INTRA;
  new_comm->group         = group;
  MPIR_Group_dup ( group, &(new_comm->local_group) );
  new_comm->send_context  = new_comm->recv_context = context;
  new_comm->comm_cache    = 0;
  new_comm->topology.type = MPI_UNDEFINED;
  new_comm->ref_count     = 1;
  new_comm->permanent     = 0;
  new_comm->error_handler = comm->error_handler;

  (void) MPIR_Attr_create_tree ( new_comm );
  (void) MPIR_Comm_make_coll( new_comm, MPIR_INTRA );

  return (errno);
}


/*+

MPIR_Sort_split_table - sort split table using a yuckie sort (YUCK!).  I'll
                        switch to a more efficient sort one of these days.

+*/
#define MPIR_EOTABLE -1
void MPIR_Sort_split_table ( Int size, Int rank, Int far *table, 
				Int far *head, Int far *list_size)
{
  Int i, j, prev;
  Int color = MPIR_Table_color(table,rank);
  
  /* Initialize head and list size */
  (*head)      = MPIR_EOTABLE;
  (*list_size) = 0;

  /* Sort out only colors == to my rank's color */
  for ( i=0; i<size; i++ ) {
    for (prev = MPIR_EOTABLE, j=(*head);
		 j != MPIR_EOTABLE; prev = j, j = MPIR_Table_next(table,j)) {
      if ( MPIR_Table_color(table,i) != color )
        continue;
      if ( (j==MPIR_EOTABLE) || (MPIR_Table_key(table,i)<MPIR_Table_key(table,j)) )
	break;
    }
    if ( MPIR_Table_color(table,i) == color) {
      (*list_size)++;
      MPIR_Table_next(table,i) = j;
      if (prev == MPIR_EOTABLE)
        (*head) = i;
      else
        MPIR_Table_next(table,prev) = i;
    }
  }
}






