/*
 *  $Id: comm_util.c,v 1.18 1994/06/07 21:24:31 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   comm_uti.c                                                         *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include "mpiimpl.h"
#include "mpisys.h"
#include <malloc.h>

/*

MPIR_Comm_make_coll - make a hidden collective communicator
                      from an inter- or intra-communicator assuming
		      that an appropriate number of contexts
		      have been allocated.  An inter-communicator
		      collective can only be made from another
		      inter-communicator.

*/
Int MPIR_Comm_make_coll ( MPI_Comm comm, MPIR_COMM_TYPE comm_type)
{
	MPI_Comm new_comm         = MPI_NEW(struct MPIR_COMMUNICATOR);
	
	if (!new_comm) 
		return MPIR_ERROR(comm, MPI_ERR_EXHAUSTED,
							"Error creating new communicator" );
	
	new_comm->comm_type       = comm_type;
	MPIR_Attr_dup_tree ( comm, new_comm );
	new_comm->comm_cache      = 0;
	new_comm->send_context    = comm->send_context + 1;
	new_comm->recv_context    = comm->recv_context + 1;
	new_comm->error_handler   = comm->error_handler;
	new_comm->ref_count       = 1;
	new_comm->permanent       = 0;
	
	if (comm_type == MPIR_INTRA) 
	{
		MPIR_Group_dup ( comm->local_group, &(new_comm->group) );
		MPIR_Group_dup ( comm->local_group, &(new_comm->local_group) );
	}
	else 
	{
		MPIR_Group_dup ( comm->group, &(new_comm->group) );
		MPIR_Group_dup ( comm->local_group, &(new_comm->local_group) );
	}
	
	new_comm->comm_coll       = new_comm;  /* a circular reference to myself */
	comm->comm_coll           = new_comm;
	
	return(MPI_SUCCESS);
}


/*+

MPIR_Comm_N2_prev - retrieve greatest power of two < size of Comm.

+*/
Int MPIR_Comm_N2_prev ( MPI_Comm comm, Int far *N2_prev)
{
	(*N2_prev) = comm->group->N2_prev;
	
	return (MPI_SUCCESS);
}


/*+
  MPIR_Dump_comm - utility function to dump a communicator 
+*/
Int MPIR_Dump_comm ( MPI_Comm comm)
{
//  Int  i;
  Int  rank;

  MPI_Comm_rank ( MPI_COMM_WORLD, &rank );

  printf("[%d] ----- Dumping communicator -----\n", rank );
  if (comm->comm_type == MPIR_INTRA) {
    printf("[%d] Intra-communicator\n",rank);
    printf("[%d] Group\n",rank);
    MPIR_Dump_group ( comm->group );
  }
  else {
    printf("[%d]\tInter-communicator\n",rank);
    printf("[%d] Local group\n",rank);
    MPIR_Dump_group ( comm->local_group );
    printf("[%d] Remote group\n",rank);
    MPIR_Dump_group ( comm->group );
  }
  printf ("[%d] Ref count = %d\n",rank,comm->ref_count);
  printf ("[%d] Send = %d   Recv =%d\n",
          rank,comm->send_context,comm->recv_context);
  printf ("[%d] permanent = %d\n",rank,comm->permanent);
  return (MPI_SUCCESS);
}

/*+
  MPIR_Intercomm_high - determine a high value for an
                        inter-communicator
+*/
Int MPIR_Intercomm_high ( MPI_Comm  comm, Int far *high)
{
  MPI_Status status;
  MPI_Comm   inter = comm->comm_coll;
  MPI_Comm   intra = inter->comm_coll;
  Int        rank, rhigh;

  MPI_Comm_rank ( comm, &rank );

  /* Node 0 determines high value */
  if (rank == 0) {

    /* "Normalize" value for high */
    if (*high)
      (*high) = 1;
    else
      (*high) = 0;

    /* Get the remote high value from remote node 0 and determine */
    /* appropriate high */
    MPI_Sendrecv(  high, 1, MPI_INT, 0, 0, 
                 &rhigh, 1, MPI_INT, 0, 0, inter, &status);
    if ( (*high) == rhigh ) {
      if ( comm->group->lrank_to_grank[0] < 
           comm->local_group->lrank_to_grank[0] )
        (*high) = 1;
      else
        (*high) = 0;
    }
  }

  /* Broadcast high value to all */
  MPI_Bcast ( high, 1, MPI_INT, 0, intra );
  return (MPI_SUCCESS);
}
