/*
 *  $Id: group_util.c,v 1.13 1994/06/07 21:24:44 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   grp_ut.c                                                           *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop
#include <malloc.h>

MPI_Group MPIR_CreateGroup( Int np)
{
    MPI_Group  new;
    Int        i;

    new = MPI_NEW(struct MPIR_GROUP);
    
    if (!new) 
	return 0;
    new->np = np;
    
    if (np > 0) 
    {
	new->lrank_to_grank = (Int far *) MPI_MALLOC( np * sizeof(Int) );
	if (!new->lrank_to_grank) 
	    return NULL;
    }
    else 
	new->lrank_to_grank = 0;
	
//    new->set_mark   = (Int *)0;
    new->set_mark   = NULL;
    new->local_rank = MPI_UNDEFINED;
    new->ref_count  = 1;
    new->permanent  = 0;
    MPIR_Powers_of_2 ( np, &(new->N2_next), &(new->N2_prev) );

    for (i=0; i<np; i++) 
	new->lrank_to_grank[i] = -1;

    return new;
}

void MPIR_FreeGroup( MPI_Group group)
{
  if (group->lrank_to_grank) {
      MPI_FREE( group->lrank_to_grank );
      }
  if ( group->set_mark ) {
      MPI_FREE( group->set_mark );
      }
  MPI_FREE( group );
}

void MPIR_SetToIdentity( MPI_Group g)
{
  Int np, i;

  np = g->np;
  for (i=0; i<np; i++) 
    g->lrank_to_grank[i] = i;

  MPID_myrank(&(g->local_rank));
  if (g->local_rank > np)
    g->local_rank = MPI_UNDEFINED;
}

/*+

MPIR_Group_dup -

+*/                                      
Int MPIR_Group_dup( MPI_Group group, MPI_Group far *new_group)
{
  (*new_group) = group;
  if ( group != MPI_GROUP_NULL )
    group->ref_count++;
  return(MPI_SUCCESS);
}


/*+

MPIR_Dump_group - dump group information

+*/
void MPIR_Dump_group ( MPI_Group group)
{
  Int i, rank;
  MPI_Comm_rank ( MPI_COMM_WORLD, &rank );

  printf ( "\t[%d] group       = %d\n", rank, group );
  if (group != NULL) {
    printf ( "\t[%d] np          = %d\n", rank, group->np );
    printf ( "\t[%d] local rank  = %d\n", rank, group->local_rank );
    printf ( "\t[%d] local rank -> global rank mapping\n", rank );
    for ( i=0; i<group->np; i++ )
      printf ( "\t [%d]   %d             %d\n", rank, i, group->lrank_to_grank[i] );
  }
}

/*+

MPIR_Dump_ranks - dump an array of ranks

+*/
void MPIR_Dump_ranks ( Int n, Int far *ranks)
{
  int i;

  printf ( "\tnumber of ranks = %d\n", n );
  printf ( "\t n     rank\n" );
  for ( i=0; i<n; i++ )
    printf ( "\t %d      %d\n", i, ranks[i] );
}

/*+

MPIR_Dump_ranges - dump an array of ranges

+*/
void MPIR_Dump_ranges ( Int n, Int far *ranges)
{
  int i;

  printf ( "\tnumber of ranges = %d\n", n );
  printf ( "\t first    last    stride\n" );
  for ( i=0; i<n; i++ )
  printf ( "\t %d      %d        %d       %d\n", i, ranges[i*3],
          ranges[(i*3)+1], ranges[(i*3)+2] );
}


/*+

MPIR_Powers_of_2 - given a number N, determine the previous and next
                   powers of 2

+*/
Int MPIR_Powers_of_2 ( Int  N, Int far *N2_next, Int far *N2_prev)
{
  Int high     = 131072;
  Int low      = 1;

  while( (high > N) && (low < N) ) {
    high >>= 1; low  <<= 1;
  }

  if(high <= N) {
    if(high == N)   /* no defect, power of 2! */
      (*N2_next) = N;
	else
      (*N2_next) = high << 1;
  }
  else {/* condition low >= N satisfied */
    if(low == N)	/* no defect, power of 2! */
      (*N2_next) = N;
	else
      (*N2_next) = low;
  }

  if( N == (*N2_next) ) /* power of 2 */
	(*N2_prev) = N;
  else
	(*N2_prev) = (*N2_next) >> 1;

  return (MPI_SUCCESS);
}

/*+

MPIR_Group_N2_prev - retrieve greatest power of two < size of group.

+*/
Int MPIR_Group_N2_prev ( MPI_Group  group, Int far *N2_prev)
{
  (*N2_prev) = group->N2_prev;
  return (MPI_SUCCESS);
}






