/*
 *  $Id: recv.c,v 1.15 1994/06/07 21:22:19 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   recv.c                                                             *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop


#ifndef lint
static char vcid[] = "$Id: recv.c,v 1.15 1994/06/07 21:22:19 gropp Exp $";
#endif /* lint */

/*@
    MPI_Recv - Basic receive

Output Parameter:
. buf - initial address of receive buffer (choice) 
. status - status object (Status) 

Input Parameters:
. count - number of elements in receive buffer (integer) 
. datatype - datatype of each receive buffer element (handle) 
. source - rank of source (integer) 
. tag - message tag (integer) 
. comm - communicator (handle) 

@*/
Int MPI_Recv( void far *buf, Int count, MPI_Datatype datatype, Int source, 
				Int tag, MPI_Comm comm, MPI_Status far *status)
{
    MPI_Request 	request;
    MPIR_RHANDLE 	rhandle;
    Int         	errno;
//    Int				is_available;

    /* 
       Because this is a very common routine, we show how it can be
       optimized to be run "inline"; In addition, this lets us exploit
       features in the ADI to simplify the execution of blocking receive
       calls.
     */
    if (source != MPI_PROC_NULL)
    {
        if (MPIR_TEST_COMM(comm,comm) || MPIR_TEST_COUNT(comm,count) ||
		    MPIR_TEST_DATATYPE(comm,datatype) || 
		    MPIR_TEST_RECV_TAG(comm,tag) ||
		    MPIR_TEST_RECV_RANK(comm,source))
		{ 
	    	return MPIR_ERROR(comm, errno, "Error in MPI_RECV" );
	    }

        request = (MPI_Request)&rhandle;
		request->type	    = MPIR_RECV;
		if (source == MPI_ANY_SOURCE)
		    rhandle.source  = source;
		else
		    rhandle.source  = comm->group->lrank_to_grank[source];
		    
		rhandle.tag	    = tag;
		rhandle.contextid   = comm->recv_context;
		rhandle.datatype    = datatype;
		rhandle.bufadd	    = buf;
		rhandle.count	    = count;
		rhandle.completed   = MPIR_NO;
		MPID_alloc_recv_handle(&(rhandle.dev_rhandle));
		if (rhandle.datatype->is_contig) 
		{
		    rhandle.dev_rhandle.start				= buf;
	            rhandle.dev_rhandle.bytes_as_contig	= count * datatype->extent;
		    rhandle.bufpos		                    = 0;
		}
#ifdef MPID_RETURN_PACKED
		else 
		{
		    if ((errno = 
		MPIR_SetupUnPackMessage( buf, count, datatype, source, request )) != 0)
			{
			MPIR_ERROR( comm, errno, 
				    "Could not pack message in MPI_RECV" );
			}
	    }
#else
		else 
		    rhandle.dev_rhandle.start = 0;
#endif

#ifdef MPID_blocking_recv
		/* If ADI supports blocking receive directly, use that */
		MPID_blocking_recv( &rhandle );
#else
	        /* ADI will handle queueing of MPIR recv handle in posted-recv queue */
		MPID_set_recv_is_nonblocking( &(rhandle.dev_rhandle), 1 );
		MPID_post_recv( &rhandle, &(is_available) );
		while ( request->rhandle.completed == MPIR_NO ) 
		{
		    MPID_check_device( MPID_BLOCKING );
		}
		MPID_complete_recv( &rhandle );
#endif
		status->MPI_SOURCE     = rhandle.source;
		status->MPI_TAG        = rhandle.tag;
		status->count	       = rhandle.actcount;  
		status->count_in_bytes = rhandle.totallen;
#ifdef MPID_RETURN_PACKED
		if (rhandle.bufpos) 
		    errno = MPIR_UnPackMessage( buf, count, datatype, 
					        source, request );
#endif
    }
    else 
    {
		/* See MPI standard section 3.11 */
		status->MPI_SOURCE     = MPI_PROC_NULL;
		status->MPI_TAG        = MPI_ANY_TAG;
		status->count	       = 0;
		status->count_in_bytes = 0;
	}
    return MPI_SUCCESS;
}

