/*
 *  $Id: sendrecv_rep.c,v 1.5 1994/06/07 21:22:24 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   sendrer.c                                                          *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop
#include <memory.h>
#include <malloc.h>

#ifndef lint
static char vcid[] = "$Id: sendrecv_rep.c,v 1.5 1994/06/07 21:22:24 gropp Exp $";
#endif /* lint */

/*@
    MPI_Sendrecv_replace - Sends and receives using a single buffer

Input Parameters:
. count - number of elements in send and receive buffer (integer) 
. datatype - type of elements in send and receive buffer (handle) 
. dest - rank of destination (integer) 
. sendtag - send message tag (integer) 
. source - rank of source (integer) 
. recvtag - receive message tag (integer) 
. comm - communicator (handle) 

Output Parameters:
. buf - initial address of send and receive buffer (choice) 
. status - status object (Status) 
@*/
Int MPI_Sendrecv_replace( void far *buf, Int count, MPI_Datatype datatype, 
							Int dest, Int sendtag, Int source, Int recvtag, 
							MPI_Comm comm, MPI_Status far *status)
{
    Int          errno = MPI_SUCCESS;
    Int          buflen;
    void         far *rbuf;
    MPI_Status   status_array[2];
    MPI_Request  req[2];

    /* Check for invalid arguments */
    if ( MPIR_TEST_COMM(comm,comm) || MPIR_TEST_DATATYPE(comm,datatype) ||
		 MPIR_TEST_COUNT(comm,count) )
		return MPIR_ERROR( comm, errno, "Error in MPI_SENDRECV_REPL" );
    /* Let the other send/recv routines find the remaining errors. */

    /* Allocate a temporary buffer that is long enough to receive the 
       message even if it has holes in it.  Perhaps a better way to 
       do this is if contiguous, then as here, else use pack/unpack
       to send contiguous data... 
     */
    if (count == 0 || datatype->is_contig) {
	buflen = datatype->extent * count;
	if ((errno = MPI_Isend ( buf,  count, datatype, dest,   
				   sendtag, comm, &req[0] )) != 0) return errno;
	if (buflen > 0) {
	    rbuf = (void far *)MPI_MALLOC( buflen );
	    if (!rbuf) {
		return MPIR_ERROR( comm, MPI_ERR_EXHAUSTED, 
				  "Error in MPI_SENDRECV_REPL" );
		}
	    }
	else
	    rbuf = (void far *)0;
	
	if ((errno = MPI_Irecv ( rbuf, count, datatype, source, 
				recvtag, comm, &req[1] )) != 0) return errno;
	errno = MPI_Waitall ( 2, req, status_array );
	if (rbuf) {
	    _fmemcpy( buf, rbuf, (int)buflen );
	    MPI_FREE( rbuf );
	    }
	(*status) = status_array[1];
	}
    else {
	Int position;
	/* non-contiguous data will be packed and unpacked */
	MPI_Pack_size( count, datatype, comm, &buflen );
	if (buflen > 0) {
	    rbuf = (void far *)MPI_MALLOC( buflen );
	    if (!rbuf) {
		return MPIR_ERROR( comm, MPI_ERR_EXHAUSTED, 
				  "Error in MPI_SENDRECV_REPL" );
		}
	    }
	else
//	    rbuf = (void  *)0;
	    rbuf = NULL;

	position = 0;
	MPI_Pack( buf, count, datatype, rbuf, buflen, &position, comm );
	errno = MPI_Sendrecv_replace( rbuf, buflen, MPI_PACKED, dest, sendtag, 
			  source, recvtag, comm, status );
	if (errno) {
	    if (rbuf) 
	        MPI_FREE( rbuf );
	    return errno;
	    }
	position = 0;
	MPI_Unpack( rbuf, buflen, &position, buf, count, datatype, comm );
	if (rbuf) {
	    MPI_FREE( rbuf );
	    }
	/* Still need to update status value? */
	}
    return errno;
}
