/*
 *  $Id: unpack.c,v 1.4 1994/06/07 21:22:42 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   unpack.c                                                           *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop
#include <memory.h>

#ifndef lint
static char vcid[] = "$Id: unpack.c,v 1.4 1994/06/07 21:22:42 gropp Exp $";
#endif /* lint */

#ifndef MPIR_TRUE
#define MPIR_TRUE  1
#define MPIR_FALSE 0
#endif


/*@
    MPI_Unpack - Unpack a datatype into contiguous memory

Input Parameters:
. inbuf - input buffer start (choice) 
. insize - size of input buffer, in bytes (integer) 
. position - current position in bytes (integer) 
. outcount - number of items to be unpacked (integer) 
. datatype - datatype of each output data item (handle) 
. comm - communicator for packed message (handle) 

Output Parameter:
. outbuf - output buffer start (choice) 

@*/   
Int MPIR_Unpack (void far *buf, Int count, MPI_Datatype type, void far *in );

Int MPI_Unpack ( void far *inbuf, Int insize, Int far *position, 
				void far *outbuf, Int outcount, MPI_Datatype type, MPI_Comm comm)
{
	Int size, pad, errno = MPI_SUCCESS;
	
	/* NOT ENOUGH ERROR CHECKING AT PRESENT */
	if (MPIR_TEST_COMM(comm,comm) || MPIR_TEST_DATATYPE(comm,type) ||
		MPIR_TEST_COUNT(comm,insize) || MPIR_TEST_ARG(position) ||
		( (*position < 0 ) && ((errno = MPI_ERR_ARG) ) != 0) )
	{ 
		return MPIR_ERROR(comm,errno,"Error in MPI_UNPACK" );
	}
	  
	/* What kind of padding is necessary? */
	pad = (type->align - ((*position) % type->align)) % type->align;
	
	/* Is the inbuf big enough for the type that's being unpacked? */
	MPIR_Pack_size ( outcount, type, comm, &size );
	if (((*position) + pad + size) > insize)
		return MPIR_ERROR(comm, MPI_ERR_LIMIT, 
							"Input buffer too small in MPI_UNPACK");
	
	/* Figure the pad and adjust position */
	(*position) += pad;
//	errno = MPIR_Unpack( outbuf, outcount, type, (Int)inbuf + (*position));
	errno = MPIR_Unpack( outbuf, outcount, type, (char far *)inbuf + (*position));
	(*position) += size;

	return (errno);
}


/*
   This code assumes that we can use char * pointers (previous code 
   incremented pointers by considering them integers, which is even 
   less portable).  Systems that, for example, use word-oriented pointers
   may want to use different code.

   This code is used in dmpi/dmpipk.c to unpack data from a device that
   only supports contiguous messages.
 */
Int MPIR_Unpack ( void far *buf, Int count, MPI_Datatype type, void far *in)
{
	Int i,j;
//	Int k;
	Int pad = 0;
	char far *tmp_buf;
	char far *lbuf = (char far *)buf, far *lin = (char far *)in;
	
	/* Unpack contiguous data */
	if (type->is_contig) 
	{
		_fmemcpy ( buf, in, (int)(type->size * count) );
		return (MPI_SUCCESS);
	}
	
	/* For each of the count arguments, unpack data */
	switch (type->dte_type) 
	{
		/* Contiguous types */
		case MPIR_CONTIG:
			MPIR_Unpack ( buf, count * type->count, type->old_type, in );
			break;
		
		/* Vector types */
		case MPIR_VECTOR:
		case MPIR_HVECTOR:
			if (count > 1)
				pad = (type->align - (type->size % type->align)) % type->align;
			tmp_buf = lbuf;
			for (i=0; i<count; i++) 
			{
				lbuf = tmp_buf;
				for (j=0; j<type->count; j++) 
				{
					MPIR_Unpack (lbuf, type->blocklen, type->old_type, lin);
					lbuf  += (type->stride);
					if ((j+1) != type->count)
						lin += ((type->blocklen * type->old_type->size) + type->pad);
				}
				lin += ((type->blocklen * type->old_type->size) + pad);
				tmp_buf += type->extent;
			}
			break;
		
		/* Indexed types */
		case MPIR_INDEXED:
		case MPIR_HINDEXED:
		if (count > 1)
			pad = (type->align - (type->size % type->align)) % type->align;
		for (i=0; i<count; i++) 
		{
			for (j=0;j<type->count; j++) 
			{
				tmp_buf  = lbuf + type->indices[j];
				MPIR_Unpack (tmp_buf, type->blocklens[j], type->old_type, lin);
				if ((j+1) != type->count)
					lin += ((type->blocklens[j]*type->old_type->size)+type->pad);
			}
			lin += ((type->blocklens[j]*type->old_type->size) + pad);
			lbuf += type->extent;
		}
		break;
		
		/* Struct type */
		case MPIR_STRUCT:
		if (count > 1)
			pad = (type->align - (type->size % type->align)) % type->align;
		for (i=0; i<count; i++) 
		{
			for (j=0;j<type->count; j++) 
			{
				tmp_buf  = lbuf + type->indices[j];
				MPIR_Unpack(tmp_buf,type->blocklens[j],type->old_types[j],lin);
				if ((j+1) != type->count)
				lin += ((type->blocklens[j] * type->old_types[j]->size) +
				type->pads[j]);
			}
			lin+=((type->blocklens[type->count-1]*
			type->old_types[type->count-1]->size)+pad);
			lbuf +=type->extent;
		}
		break;
		
		default:
			return MPIR_ERROR (MPI_COMM_WORLD, MPI_ERR_TYPE, 
								"Error in MPI_UNPACK");
//			break;
	}
	
	/* Everything fell through, must have been successful */
	return (MPI_SUCCESS);
}
