/*
 *  $Id: cart_create.c,v 1.8 1994/06/14 16:04:04 bridges Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   crt_cre.c                                                          *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop

/*@

MPI_Cart_create - Makes a new communicator to which topology information
                  has been attached

Input Parameters:
. comm_old - input communicator (handle) 
. ndims - number of dimensions of cartesian grid (integer) 
. dims - integer array of size ndims specifying the number of processes in each dimension 
. periods - logical array of size ndims specifying whether the grid is periodic (true ) or not (false ) in each dimension 
. reorder - ranking may be reordered (true ) or not (false ) (logical) 

Output Parameter:
. comm_cart - communicator with new cartesian topology (handle) 

Algorithm:
We ignore the periods and reorder info currently.

@*/
/*ARGSUSED*/
Int MPI_Cart_create ( MPI_Comm comm_old, Int ndims, Int far *dims, 
		      Int far *periods, Int reorder, MPI_Comm far *comm_cart)
{
  Int range[1][3];
  MPI_Group group_old, group;
  Int i, rank, num_ranks = 1;
  Int errno = MPI_SUCCESS;
  Int flag, size;

  reorder = reorder;
  /* Check validity of arguments */
  if (MPIR_TEST_COMM(comm_old,comm_old) || MPIR_TEST_ARG(comm_cart) ||
      MPIR_TEST_ARG(periods)  ||
	  ((ndims     <  1)             && ((errno = MPI_ERR_DIMS) != 0)) ||
	  ((dims      == NULL)          && ((errno = MPI_ERR_DIMS) != 0)))
    return MPIR_ERROR( comm_old, errno, "Error in MPI_CART_CREATE" );

  /* Check for Intra-communicator */
  MPI_Comm_test_inter ( comm_old, &flag );
  if (flag)
    return MPIR_ERROR(comm_old, MPI_ERR_COMM,
                      "Inter-communicator invalid in MPI_CART_CREATE");

  /* Determine number of ranks in topology */
  for ( i=0; i<ndims; i++ )
    num_ranks    *= (dims[i]>0)?dims[i]:-dims[i];
  if ( num_ranks < 1 ) {
    (*comm_cart)  = MPI_COMM_NULL;
    return MPIR_ERROR( comm_old, MPI_ERR_TOPOLOGY, 
		      "Error in MPI_CART_CREATE" );
  }

  /* Is the old communicator big enough? */
  MPI_Comm_size (comm_old, &size);
  if (num_ranks > size) 
	return MPIR_ERROR(comm_old, MPI_ERR_ARG, 
					  "Topology size too big in MPI_CART_CREATE");
	
  /* Make new comm */
  range[0][0] = 0; range[0][1] = num_ranks - 1; range[0][2] = 1;
  MPI_Comm_group ( comm_old, &group_old );
  MPI_Group_range_incl ( group_old, 1, range, &group );
  MPI_Comm_create  ( comm_old, group, comm_cart );

  /* Store topology information in new communicator */
  if ( (*comm_cart) != MPI_COMM_NULL ) {
    (*comm_cart)->topology.type                  = MPI_CART;
    (*comm_cart)->topology.tag.cart.nnodes       = num_ranks;
    (*comm_cart)->topology.tag.cart.ndims        = ndims;
    (*comm_cart)->topology.tag.cart.dims         = 
      (Int far *)MPI_MALLOC( sizeof(Int) * 3 * ndims );
    (*comm_cart)->topology.tag.cart.periods      = 
      (*comm_cart)->topology.tag.cart.dims + ndims;
    (*comm_cart)->topology.tag.cart.position     = 
      (*comm_cart)->topology.tag.cart.periods + ndims;
    for ( i=0; i<ndims; i++ ) {
      (*comm_cart)->topology.tag.cart.dims[i]    = dims[i];
      (*comm_cart)->topology.tag.cart.periods[i] = periods[i];
    }

    /* Compute my position */
    MPI_Comm_rank ( (*comm_cart), &rank );
    for ( i=0; i < ndims; i++ ) {
      num_ranks = num_ranks / dims[i];
      (*comm_cart)->topology.tag.cart.position[i]  = rank / num_ranks;
      rank   = rank % num_ranks;
    }
  }
  return (errno);
}
