/*
 *  $Id: cart_sub.c,v 1.12 1994/06/14 18:07:29 doss Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/***********************************************************************
*                                                                      *
*   crt_sub.c                                                          *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop

/*@

MPI_Cart_sub - Partitions a communicator into subgroups which 
               form lower-dimensional cartesian subgrids

Input Parameters:
. comm - communicator with cartesian structure (handle) 
. remain_dims - the  i th entry of remain_dims specifies whether the i th 
dimension is kept in the subgrid ( true ) or is dropped ( false ) (logical 
vector) 

Output Parameter:
. newcomm - communicator containing the subgrid that includes the calling 
process (handle) 

@*/
Int MPI_Cart_sub ( MPI_Comm comm, Int far *remain_dims, MPI_Comm far *comm_new)
{
  Int i, j, ndims;
  Int num_remain_dims = 0;
  Int remain_total = 1;
  Int rank;
  Int color = 0;
  Int key = 0;
  Int errno = MPI_SUCCESS;

  /* Check for valid arguments */
  if ( MPIR_TEST_COMM(comm,comm) ||
	   ((comm->topology.type != MPI_CART)     && ((errno = MPI_ERR_TOPOLOGY) != 0)) ||
       MPIR_TEST_ARG(remain_dims) || MPIR_TEST_ARG(comm_new))
    return MPIR_ERROR( comm, errno, "Error in MPI_CART_SUB" );
  
  /* Determine the remaining dimension total */
  ndims = comm->topology.tag.cart.ndims;
  for ( i=0; i<ndims; i++ )
    if ( remain_dims[i] ) {
      num_remain_dims++;
      remain_total *= comm->topology.tag.cart.dims[i];
	}

  /* Check for special case */
  if ( num_remain_dims == 0 ) {
    (*comm_new) = MPI_COMM_NULL;
    return (errno);
  }

  /* Split the old communicator */
  for ( i=0; i< ndims; i++ ) 
	if ( remain_dims[i] ) {
	  key = (key * comm->topology.tag.cart.dims[i]) +
		    comm->topology.tag.cart.position[i];
	}
	else {
	  color = (color * comm->topology.tag.cart.dims[i]) +
		      comm->topology.tag.cart.position[i];
	}
  MPI_Comm_split ( comm, color, key, comm_new );

  /* Store topology information in new communicator */
  if ( (*comm_new) != MPI_COMM_NULL ) {
    (*comm_new)->topology.type                    = MPI_CART;
    (*comm_new)->topology.tag.cart.nnodes         = remain_total;
    (*comm_new)->topology.tag.cart.ndims          = num_remain_dims;
    (*comm_new)->topology.tag.cart.dims           = 
      (Int far *)MPI_MALLOC( sizeof(Int) * 3 * num_remain_dims );
    (*comm_new)->topology.tag.cart.periods        = 
      (*comm_new)->topology.tag.cart.dims + num_remain_dims;
    (*comm_new)->topology.tag.cart.position       = 
      (*comm_new)->topology.tag.cart.periods + num_remain_dims;
    for ( i=j=0; i<ndims; i++ )
      if ( remain_dims[i] ) {
		(*comm_new)->topology.tag.cart.dims[j]    = 
		  comm->topology.tag.cart.dims[i];
		(*comm_new)->topology.tag.cart.periods[j] = 
		  comm->topology.tag.cart.periods[i];
		j++;
	  }
  
    /* Compute my position */
    MPI_Comm_rank ( (*comm_new), &rank );
    for ( i=0; i < num_remain_dims; i++ ) {
      remain_total = remain_total / (*comm_new)->topology.tag.cart.dims[i];
      (*comm_new)->topology.tag.cart.position[i]  = rank / remain_total;
      rank = rank % remain_total;
    }
  }
  return (errno);
}

