/*
 *  $Id: util_hbt.c,v 1.4 1994/06/07 21:30:17 gropp Exp $
 *
 *  (C) 1993 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

/*
  util_util - height balanced tree source code
*/

/***********************************************************************
*                                                                      *
*   util_hbt.c                                                         *
*   MPI for MS-Windows 3.1                                             *
*   current version: 0.99b          06/10/95                           *
*                                                                      *
*   Joerg Meyer                                                        *
*   University of Nebraska at Omaha (UNO)                              *
*   Department of Computer Science                                     *
*                                                                      *
*   This is an MPI implementation for MS-Windows 3.1                   *
*   It is based on the MPI implementation from Argonne National        *
*   Laboratory and Mississippi State University, version from          *
*   June 17, 1994. Note their COPYRIGHT.                               *
*   ( source code and user's guide available by anonymous FTP from     *
*     info.mcs.anl.gov in directory /pub/mpi )                         *
*   Anyone is free to copy and modify this code to suit his or her     *
*   own purposes as long as these notices are retained.                *
*                                                                      *
***********************************************************************/

#include <mpiimpl.h>
#include <mpisys.h>
#pragma hdrstop

/*===========================================================================*
  Interface for height balanced trees

 int MPIR_HBT_new_tree  ( MPIR_HBT **tree ) 
 int MPIR_HBT_free_tree ( MPIR_HBT *tree )
 int MPIR_HBT_free_subtree ( MPIR_HBT_node *node )

 int MPIR_HBT_new_node  ( int keyval, void *value, MPIR_HBT_node **node_out ) 
 int MPIR_HBT_free_node ( MPIR_HBT_node *node )

 int MPIR_HBT_lookup ( MPIR_HBT *tree, int keyval, MPIR_HBT_node **node )
 int MPIR_HBT_insert ( MPIR_HBT *tree, MPIR_HBT_node *node )
 int MPIR_HBT_delete ( MPIR_HBT *tree, int keyval )

 *===========================================================================*/

/*===========================================================================*/
/* The stack size for the deletion algorithm.                                */
/*===========================================================================*/
#define MPIR_STACK_SIZE 100

/*===========================================================================*/
/* Each node has flag indicating whether it is balanced or not.              */
/* NOTE: comparisons are done using ">" and "<" so sign is important         */
/*===========================================================================*/
#define MPIR_BALANCED 0
#define MPIR_UNBALANCED_LEFT -1
#define MPIR_UNBALANCED_RIGHT 1

/*===========================================================================*/
/* Some macros for easy access to fields in the HBT_obj structure            */
/*===========================================================================*/
#define LEFT(e) (e) -> left
#define RIGHT(e) (e) -> right
#define B(e) (e) -> balance

/*===========================================================================*/
/* Macro to compare keyval's with                                            */
/*===========================================================================*/
#define MPIR_COMPARE(a,b)  ((a)-(b))

/* Temporary */
#ifndef MPIR_TRUE
#define MPIR_TRUE  1
#define MPIR_FALSE 0
#endif

/*===========================================================================*/

/*+


MPIR_HBT_new_tree -

+*/
Int MPIR_HBT_new_tree ( LPPMPIR_HBT tree_out)
{
	MPIR_HBT far *new_tree;
	new_tree = (*tree_out) = (MPIR_HBT far *) MPIR_SBalloc ( MPIR_hbts );
//	new_tree->root = (MPIR_HBT_node *)0;
	new_tree->root = NULL;
	new_tree->height = 0;
	return (MPI_SUCCESS);
}

/*+

MPIR_HBT_new_node -

+*/
Int MPIR_HBT_new_node ( Int keyval, void far *value, LPPMPIR_HBT_node node_out)
{
	MPIR_HBT_node far *new_node;
	
	new_node = (*node_out) = (MPIR_HBT_node far *) MPIR_SBalloc (MPIR_hbt_els);
	new_node->keyval       = keyval;
	new_node->value        = value;
	new_node->balance      = MPIR_BALANCED;
//	new_node->left         = new_node->right = (MPIR_HBT_node far *)0;
	new_node->left         = new_node->right = NULL;
	return (MPI_SUCCESS);
}

/*+

MPIR_HBT_free_node -

+*/                                       
Int MPIR_HBT_free_node ( MPIR_HBT_node far *node)
{
//  if (node != (MPIR_HBT_node *)0)
	if (node != NULL)
	/* Free memory used by node */
		MPIR_SBfree( MPIR_hbt_els, node );

	return (MPI_SUCCESS);
}

/*+

MPIR_HBT_free_subtree -

+*/
Int MPIR_HBT_free_subtree ( MPIR_HBT_node far *subtree)
{
//	if(subtree != (MPIR_HBT_node *)0) 
	if(subtree != NULL) 
	{
		(void) MPIR_HBT_free_subtree ( subtree -> left );
		(void) MPIR_HBT_free_subtree ( subtree -> right );
		(void) MPIR_HBT_free_node ( subtree );
	}
	return (MPI_SUCCESS);
}

/*+

MPIR_HBT_free_tree -

+*/
Int MPIR_HBT_free_tree ( MPIR_HBT far *tree)
{
	if ( tree != (MPIR_HBT *)0 ) 
	{
//		if ( tree->root != (MPIR_HBT_node *)0 )
		if ( tree->root != NULL )
			(void) MPIR_HBT_free_subtree ( tree->root );
			
		MPIR_SBfree ( MPIR_hbts, tree );
	}
	return (MPI_SUCCESS);
}

/*+

MPIR_HBT_lookup( -

+*/
Int MPIR_HBT_lookup( MPIR_HBT far *tree, Int keyval, LPPMPIR_HBT_node node_out)
{
	Int test;
	MPIR_HBT_node far *temp = tree->root;
	
	while(temp)
		if ( (test = MPIR_COMPARE( keyval, temp->keyval )) < 0 )
			temp = LEFT(temp);
		else if ( test > 0)
			temp = RIGHT(temp);
		else 
		{
			(*node_out) = temp;
			return (MPI_SUCCESS);
		} 
		
//	(*node_out) = (MPIR_HBT_node *)0;
	(*node_out) = NULL;
	return (MPI_SUCCESS);
}

/*+

MPIR_HBT_insert( -

+*/
Int MPIR_HBT_insert( MPIR_HBT far *tree, MPIR_HBT_node far *node)
{
  MPIR_HBT_node far *temp, far *inserted, far *rebalance_son, 
  				far *rebalance, far *rebalance_father;
  Int done = 0;
  Int test, test_rebalance, rebalance_B;

  /* If tree is currently empty then just add new node. */
//  if ( tree->root == (MPIR_HBT_node *)0 ) {
  if ( tree->root == NULL) {
    tree->root = node;
    tree->height = 1;
    return (MPI_SUCCESS);
  }
  
  /* Initialize start location for balance adj and rebalancing efforts*/
  rebalance_father = (MPIR_HBT_node far *)tree;
  rebalance = temp = tree->root;
  
  /* Find the place where the new node should go. */
  while(!done) {
    if( (test = MPIR_COMPARE( node->keyval , temp->keyval )) == 0) {
	  /* The key already exists in the tree can't add. */
	  return (MPI_SUCCESS);
    }
    else if (test < 0) {
	  /* Go left. */
//	  if( (inserted = LEFT(temp)) == (MPIR_HBT_node *)0 ) {
	  if( (inserted = LEFT(temp)) == NULL ) {
        /* Found place to insert the new node */
        inserted = LEFT(temp) = node;
        done = 1;
	  }
	  else {
        /* If not balanced at this point the move the starting location*/
        /* for rebalancing effort here. */
        if ( B(inserted) != MPIR_BALANCED ) {
          rebalance_father = temp;
          rebalance = inserted;
        }
        temp = inserted;
	  }
    }
    else if ( test > 0) {
	  /* Go left. */
	  if( (inserted = RIGHT(temp) ) == NULL) {
        /* Found place to insert the new node */
        inserted = RIGHT(temp) = node;
        done = 1;
	  }
	  else {
        /* If not balanced at this point the move the starting location*/
        /* for rebalancing effort here. */
        if ( B(inserted) != MPIR_BALANCED ) {
		  rebalance_father = temp;
		  rebalance = inserted;
        }
        temp = inserted;
	  }
    }
  }
    
  /* Adjust the balance factors along the path just traversed.  Only need to */
  /* do this on part of the path. */
  if( (test_rebalance = MPIR_COMPARE(node->keyval, rebalance->keyval)) < 0 )
    rebalance_son = temp = LEFT(rebalance);
  else
    rebalance_son = temp = RIGHT(rebalance);

  while( temp != inserted) {
    if ( (test = MPIR_COMPARE( node->keyval, temp->keyval )) < 0 ) {
	  B(temp) = MPIR_UNBALANCED_LEFT;
	  temp = LEFT(temp);
	}
    else if (test > 0) {
	  B(temp) = MPIR_UNBALANCED_RIGHT;
	  temp = RIGHT(temp);
	}
  }

  /* Rebalance the tree.  There is only one point where rebalancing might    */
  /* be needed. */
  rebalance_B = (test_rebalance<0)?MPIR_UNBALANCED_LEFT:MPIR_UNBALANCED_RIGHT;

  if( B(rebalance) == MPIR_BALANCED) {
    /* Tree was balanced, adding new node simply unbalances it.            */
    B(rebalance) = rebalance_B;
    tree -> height++;
  }
  else if ( B(rebalance) == -rebalance_B )
    /* Tree was unbalanced towards the opposite side of insertion so       */
    /* with new node it is balanced. */
    B(rebalance) = MPIR_BALANCED;
  else {
    /* Tree needs rotated. */
    /* See Knuth or Reingold for picture of the rotations much easier and  */
    /* clearer than any word description. */
    if (B(rebalance_son) == rebalance_B) {
	  /* Single rotation. */
	  temp = rebalance_son;
	  if ( rebalance_B == MPIR_UNBALANCED_LEFT) {
        LEFT(rebalance) = RIGHT(rebalance_son);
        RIGHT(rebalance_son) = rebalance;
	  }
	  else {
        RIGHT(rebalance) = LEFT(rebalance_son);
        LEFT(rebalance_son) = rebalance;
	  }
	  B(rebalance) = ( B(rebalance_son) = MPIR_BALANCED );
    }
    else if ( B(rebalance_son) == -rebalance_B) {
	  /* double rotation */
	  if ( rebalance_B == MPIR_UNBALANCED_LEFT) {
        temp = RIGHT(rebalance_son);
        RIGHT(rebalance_son) = LEFT(temp);
        LEFT(temp) = rebalance_son;
        LEFT(rebalance) = RIGHT(temp);
        RIGHT(temp) = rebalance;
	  }
	  else {
        temp = LEFT(rebalance_son);
        LEFT(rebalance_son) = RIGHT(temp);
        RIGHT(temp) = rebalance_son;
        RIGHT(rebalance) = LEFT(temp);
        LEFT(temp) = rebalance;
	  }
	  
	  if ( B(temp) == rebalance_B ) {
        B(rebalance) = -rebalance_B;
        B(rebalance_son) = MPIR_BALANCED;
	  }
	  else if ( B(temp) == 0 )
        B(rebalance) = (B(rebalance_son) = MPIR_BALANCED);
	  else {
        B(rebalance) = MPIR_BALANCED;
        B(rebalance_son) = rebalance_B;
	  }
      B(temp) = MPIR_BALANCED;
    }

    /* Need to adjust what the father of the this subtree points at since  */
    /* we rotated. */
    if( rebalance_father == (MPIR_HBT_node far *)tree )
	  tree -> root = temp;
    else {
	  if ( rebalance == RIGHT(rebalance_father) )
        RIGHT(rebalance_father) = temp;
	  else
        LEFT(rebalance_father) = temp;
    }
  }
  return (MPI_SUCCESS);
}

/*+

MPIR_HBT_delete -

+*/
Int MPIR_HBT_delete(MPIR_HBT far *tree, Int keyval, LPPMPIR_HBT_node node_out)
{
  /* The stack keeps elements from root to the node to be deleted.         */
  /* element_stack has pointers to the nodes.  dir_stack has the direction */
  /* taken down the path.                                                  */
  Int size = 0;                    
  MPIR_HBT_node  far *element_stack[MPIR_STACK_SIZE];
  short       dir_stack[MPIR_STACK_SIZE];
  short       father_dir;
  MPIR_HBT_node  far *del, far *successor, far *father, far *current,
  				 far *son, far *grandson, far *temp;
  Int test, done,dir, top_of_stack;
  
  /* Find the node to be deleted.  Keep track of path on the stack.        */

  /* put the tree as indicator on the top of the stack.  This simplifies   */
  /* logic a little. */
  element_stack[size++] = (MPIR_HBT_node far *) tree;
  element_stack[size] = (del = tree -> root);

  while( del && ((test = MPIR_COMPARE(keyval, del->keyval)) != 0) ) {
	if(test < 0) {
      dir_stack[size] = MPIR_UNBALANCED_LEFT;
      del = LEFT(del);
	}
	else {
      dir_stack[size] = MPIR_UNBALANCED_RIGHT;
      del = RIGHT(del);
	}
	element_stack[++size] = del;
  }
    
  /* Node was not found so can't delete it. */
  if (del == NULL)
    return (MPI_SUCCESS);
  
  /* Set father to be the parent of the node to be deleted */
  father = element_stack[size-1];
  father_dir = dir_stack[size-1];
  
  /* Do the actual deletion of the node from the tree. */
  /* Special processing is done if del is at the root of the tree. */
  if( RIGHT(del) == NULL ) {
	/* RIGHT of del is null so we can just delete the node */
	/* set dir to indicate where the NULL child is */
	dir_stack[size] = MPIR_UNBALANCED_RIGHT;
    
	if(father == (MPIR_HBT_node far *)tree) {
      tree -> root = LEFT(del);
      tree -> height--;
	}
	else
      if(father_dir < 0)
		LEFT(father) = LEFT(del);
      else
		RIGHT(father) = LEFT(del);
  }
  else if( LEFT(del) == NULL ) {
	/* LEFT of del is null so we can just delete the node                */
	/* set dir to indicate where the NULL child is */
	dir_stack[size] = MPIR_UNBALANCED_LEFT;
	
	if(father == (MPIR_HBT_node far *)tree) {
      tree -> root = RIGHT(del);
      tree -> height--;
	}
	else
      if(father_dir < 0)
		LEFT(father) = RIGHT(del);
      else
		RIGHT(father) = RIGHT(del);
  }
  else {
	/* The "trick" employed here is finding the successor to del with a  */
	/* left link that is NULL.  This successor node is then swapped with */
	/* the node that we want to delete.  Thus the number of cases for    */
	/* actual deletion are small.  The tree is out of order (del has     */
	/* had been placed behind successor) but this does not matter        */
	/* since the tree is not accessed during deletion and the del        */
	/* node will be deleted anyway.  The swapping process just moves     */
	/* successor to del position; del is not reinserted since it is to be*/
	/* removed.                                                          */
	temp = RIGHT(del);
	if(LEFT(temp) == NULL) {
      /* This is a special case when the successor is the son of del.  */
      /* Need to fix the stack since del and successor have swapped.   */
      dir_stack[size] = MPIR_UNBALANCED_RIGHT;
      element_stack[size++] = temp; 
      
      /* Here is the swap of del and successor.                        */
      LEFT(temp) = LEFT(del);
      if(father == (MPIR_HBT_node far *)tree)
		tree -> root = temp;
      else
		if(father_dir < 0)
          LEFT(father) = temp;
		else
          RIGHT(father) = temp;

      /* successor will be rebalanced but it would have had same       */
      /* balance as del with del in successors place                   */
      B(temp) = B(del);
	}
	else {
      /* Successor is not the son of del so a search must be done.     */
      /* Need to fix the stack since del and successor have swapped.   */
      dir_stack[size] = MPIR_UNBALANCED_RIGHT;
      element_stack[size++] = temp; 
      
      /* Find the first successor to del that has no left subtree.     */
      successor = LEFT(temp);
      while((LEFT(successor) != NULL)) {
		dir_stack[size] = MPIR_UNBALANCED_LEFT;
		element_stack[size++] = successor;
		successor = LEFT(successor);
      };

      /* Here is the swap of del and successor.                        */
      LEFT(successor) = LEFT(del);
      LEFT(element_stack[size -1]) = RIGHT(successor);
      RIGHT(successor) = RIGHT(del);
      if(father == (MPIR_HBT_node far *)tree)
		tree -> root = successor;
      else
		if(father_dir < 0)
          LEFT(father) = successor;
		else
          RIGHT(father) = successor;
      
      /* successor will be rebalanced but it would have had same       */
      /* balance as del with del in successors place                   */
      B(successor) = B(del);
	}
  }
     
  /* Rebalance the tree.  Search up the stack that was kept and rebalance  */
  /* at each node if needed.  The search can be terminated if the subtree  */
  /* height has not changed; the balance of higher noded could not have    */
  /* changed.                                                              */
  done = 0;
  
  /* NOTE that the element 0 is the tree HBT so we don't visit it */
  for(top_of_stack = size-1; (top_of_stack > 0) && (!done); 
      top_of_stack--) {
	current = element_stack[top_of_stack];
	dir = dir_stack[top_of_stack];
	
	if ( B(current) == MPIR_BALANCED ) {
      /* The subtree was balanced at this point.                   */
      /* Unbalance it since a node was deleted.  Since the height  */
      /* of this subtree was not changed we are done               */
      if(dir == MPIR_UNBALANCED_LEFT)
		B(current) = MPIR_UNBALANCED_RIGHT;
      else
		B(current) = MPIR_UNBALANCED_LEFT;
      
      done = 1;
	}	    
	else if ( B(current) == dir) {
      /* The subtree was unbalanced toward the side the deletion   */
      /* occurred on so the new subtree is balanced but it has     */
      /* height one less than before so the rebalencing must       */
      /* continue                                                  */
      B(current) = MPIR_BALANCED;
      
      if(top_of_stack == 1)
		tree -> height--;
	}
	else {
      /* The del node was on the unbalanced side so the subtree    */
      /* is unbalanced by two.  Need to do a rotation.             */
	    
      /* The rotation that needs to be done can be determined from */
      /* the son of del.  Again refering to Knuth or Reingold would*/
      /* be more valuable than any written description that I could*/
      /* write.  One day, perhaps, we can include pictures in      */
      /* in comments.                                              */
      if ( dir == MPIR_UNBALANCED_LEFT )
		son = RIGHT(current);
      else
		son = LEFT(current);
      
      if ( B(son) == MPIR_BALANCED ) {
		/* Son was balanced do a single rotation.                */
		/* Since the subtree at father has not changed in        */
		/* height we are done.                                   */
		if(dir == MPIR_UNBALANCED_LEFT) {
          RIGHT(current) = LEFT(son);
          LEFT(son) = current;
		}
		else {
          LEFT(current) = RIGHT(son);
          RIGHT(son) = current;
		}
		
		B(son) = dir;
		done = 1;
      }
      else if (B(son) == -dir ) {
		/* son is balanced the opposite direction we             */
		/* took at current.  Need to reblance and continue       */
		/* since the tree is one shorter than before.            */
		
		if( dir == MPIR_UNBALANCED_LEFT) {
          RIGHT(current) = LEFT(son);
          LEFT(son) = current;
		}
		else {
          LEFT(current) = RIGHT(son);
          RIGHT(son) = current;
		}
		
		/* current and son are balanced now */
		B(current) = ( B(son) = MPIR_BALANCED);
		
		if(top_of_stack == 1)
          tree -> height--;
      }
      else {
		/* son is balanced the same direction we took at current */
		/* Need to do a double rotation and continue.            */
		if(dir == MPIR_UNBALANCED_LEFT) {
          grandson = LEFT(son);
          RIGHT(current) = LEFT(grandson);
          LEFT(son) = RIGHT(grandson);
          LEFT(grandson) = current;
          RIGHT(grandson) = son;
		}
		else {
          grandson = RIGHT(son);
          LEFT(current) = RIGHT(grandson);
          RIGHT(son) = LEFT(grandson);
          RIGHT(grandson) = current;
          LEFT(grandson) = son;
		}
		
		/* adjust the balance factors */
		if( B(grandson) == MPIR_BALANCED)
          B(son) = (B(current) = MPIR_BALANCED);
		else if( B(grandson) == dir) {
          B(son) = -dir;
          B(current) = MPIR_BALANCED;
		}
		else {
          B(son) = MPIR_BALANCED;
          B(current) = dir;
		}
		
		if(top_of_stack == 1)
          tree -> height--;
		
		/* Double rotation puts grandson at root of subtree.     */
		son = grandson;
      }
	    
      /* Since we rotated the subtree has a new root; point father */
      /* of del at this new root.                                  */
      if (( father = element_stack[top_of_stack - 1]) ==
          (MPIR_HBT_node far *)tree) {
		tree -> root = son;
      }
      else {
		if ( dir_stack[top_of_stack -1] == MPIR_UNBALANCED_LEFT)
          LEFT(father) = son;
		else
          RIGHT(father) = son;
      }
	} 
  }
    
  /* Delete the node */
  (*node_out) = del;
  return (MPI_SUCCESS);
}

